package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import java.util.Stack;

/**
 * One level in callstack.
 */
class StackElement<T extends ElementExecutor> {

    public final String name;
    public final Class clazz;
    public final T executor;

    StackElement(Class clazz, String name, T executor) {
        this.clazz = clazz;
        this.name = name;
        this.executor = executor;
    }

    @Override
    public String toString() {
        return "StackElement[" + clazz.getSimpleName() + ":" + name + ":" + executor + "]";
    }

    T getExecutor() {
        return executor;
    }

    /**
     * Get results of trigger for this element. Basically, unless the trigger
     * is fulfilled, this element won't fire.
     * @return Trigger result of executor
     * @see StackElement#getExecutor() 
     */
    TriggerResult getTriggerResults() {
        return executor.getTriggerResult();
    }

    /**
     * Get type of stack element, e.g. {@link TriggeredAction}, 
     * {@link CompetenceElement}, etc. 
     * @return Type name, without white spaces
     */
    public String getTypeName() {
        return clazz.getSimpleName();
    }
}

/**
 * Callstack used for storing info what was call hiearchy of elements.
 *
 * @author Honza
 */
final public class ElementStackTrace extends Stack<StackElement> {

    public synchronized void printStackTrace() {
        System.out.println("ElementStackTrace:");
        for (StackElement element : this) {
            System.out.println(element);
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (StackElement<ElementExecutor> element : this) {
            sb.append('/');
            String type = element.getTypeName();
            sb.append(type);
            sb.append(':');
            sb.append(element.name);
            // If this element has some sense, show them in the path
            TriggerResult triggerResult = element.getTriggerResults();
            if (triggerResult != null) {
                for (SenseResult senseResult : element.getTriggerResults()) {
                    sb.append('[');
                    sb.append(senseResult.name);
                    sb.append(':');
                    sb.append(senseResult.wasSuccessful() ? '1' : '0');
                    sb.append(']');
                }
            }
        }
        return sb.toString();
    }
}
