package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.elements.Goal;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.Sense.SenseCall;
import cz.cuni.amis.pogamut.sposh.elements.Triggers;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Logger;

/**
 * Class that immutably stores the result of evaluation of trigger. Trigger is 
 * basically a list of sense calls (i.e. {@link SenseListExecutor}).
 */
class TriggerResult implements Iterable<SenseResult> {
    private final List<SenseResult> senses;
    private final boolean success;

    /**
     * Create new result of trigger
     * @param senses list of results, what did each sense ended up as
     * @param success was the trigger as a whole successful
     */
    public TriggerResult(List<SenseResult> senses, boolean success) {
        this.success = success;
        this.senses = Collections.unmodifiableList(new ArrayList<SenseResult>(senses));
    }

    /**
     * Create new result of trigger, without specifying any senses. 
     * @param success was the trigger successful
     */
    public TriggerResult(boolean success) {
        this(Collections.<SenseResult>emptyList(), success);
    }
    
    public boolean wasSuccess() {
        return success;
    }

    /**
     * Get iterator to the all senses of this trigger. Unmodifiable.
     * @return iterator to the start of sens's list.
     */
    @Override
    public Iterator<SenseResult> iterator() {
        return senses.iterator();
    }
}

/**
 * Executor that decicdes if goal or trigger are fulfilled. That happens only
 * is all its senses are fired.
 * 
 * @author Honza
 */
final class SenseListExecutor extends AbstractExecutor {

    private List<SenseExecutor> sensesExecutors = new ArrayList<SenseExecutor>();

    /**
     * Create executor for goal.
     * @param goal null for empty SenseListExecutor or source of senses
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(Goal goal, VariableContext ctx, Logger log) {
        super(ctx, log);
        if (goal == null) {
            return;
        }

        for (Sense sense : goal.getSenses()) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }

    /**
     * Create executor for triggers.
     * @param triggers source of senses, can be null, then no senses will be used.
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(Triggers triggers, VariableContext ctx, Logger log) {
        super(ctx, log);
        if (triggers == null) {
            return;
        }

        for (Sense sense : triggers.getSenses()) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }

    /**
     * Create executor for triggers.
     * @param triggers source of senses, can be null, then no senses will be used.
     * @param log logger to record actions of this executor, can be null
     */
    SenseListExecutor(List<Sense> triggers, VariableContext ctx, Logger log) {
        super(ctx, log);

        for (Sense sense : triggers) {
            sensesExecutors.add(new SenseExecutor(sense, ctx, log));
        }
    }

    /**
     * Evaluate all senses until first one fails. 
     * If no senses were specified, consider it a fail.
     * @param defaultReturn what to return if no senses were specified. In trigger true, in goal false
     * @return Result of the senselist. The result is true if none of senses fails, false if at least one fails.
     */
    public TriggerResult fire(IWorkExecutor workExecuter, boolean defaultReturn) {
        List<SenseResult> senses = new LinkedList<SenseResult>();
        
        for (SenseExecutor senseExecutor : sensesExecutors) {
            defaultReturn = true;
            
            SenseResult res = senseExecutor.fire(workExecuter);
            senses.add(res);
            
            if (!res.wasSuccessful()) {
                return new TriggerResult(senses, false);
            }
        }
        return new TriggerResult(senses, defaultReturn);
    }
}
