package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.exceptions.DuplicateNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.FubarException;
import cz.cuni.amis.pogamut.sposh.exceptions.InvalidNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.UnexpectedElementException;
import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is root of POSH plan in execution sense. In source of POSH plan, this is
 * leaf of root. Every POSH plan can have only one DriveCollection.
 *
 * drive collection: this is the root of the POSH hierarchy, and on every POSH
 * action-selection cycle, this reconsiders which goal the agent should be
 * working on. This is how a BOD agent can respond rapidly to changes in the
 * environment.
 *
 * @author HonzaH
 */
public final class DriveCollection extends NamedLapElement {

    /**
     * Name of the collection
     */
    private String name;
    /**
     * Goal of drive collection. If goal is not fulfilled, run the engine.
     * Useful when there are multiple plans (e.g. one for shooting and one for
     * moving), goal of the shooting plan would be "no-enemy-in-sight" so when
     * enemy would be in sight, it would work, but w/o enemy, it would be
     * dormant.
     */
    private Goal goal;
    /**
     * List of drive elements of this DC. Basically one choice at the top of
     * decision tree what to do next.
     */
    private final List<DriveElement> elements = new LinkedList<DriveElement>();
    /**
     * Unmodifiable list of drive elements of this DC, basically a wrapper for {@link DriveCollection#elements}.
     */
    private final List<DriveElement> elementsUm = Collections.unmodifiableList(elements);
    /**
     * Property name for change of name.
     */
    public static final String dcName = "dcName";
    /**
     * Data flavor of DC, used for drag and drop
     */
    public static final DataFlavor dataFlavor = new DataFlavor(DriveCollection.class, "drive-collection-node");

    /**
     * Create a new drive collection.
     *
     * @param name Name of new DC
     * @param goal Goal of new collection, not null
     * @param drives List of drives
     */
    DriveCollection(String name, Goal goal, List<DriveElement> drives) {
        assert goal != null;
        assert goal.getParent() == null;
        assert drives != null;

        this.name = name;
        this.goal = goal;
        this.goal.setParent(this);

        for (DriveElement element : drives) {
            assert element.getParent() == null;

            elements.add(element);
            element.setParent(this);
        }
    }

    /**
     * Add passed drive as the last drive of this DC and emit new it.
     *
     * @param drive drive to add
     */
    public void addDrive(DriveElement drive) throws DuplicateNameException {
        // sanity: check that new drive's parent doesn't have drive listed as a child
        if (drive.getParent() != null) {
            assert !drive.getParent().getChildDataNodes().contains(drive);
        }

        if (isUsedName(drive.getName(), elements)) {
            throw new DuplicateNameException("DC " + name + " already have drive with name " + drive.getName());
        }

        elements.add(drive);
        drive.setParent(this);

        emitChildNode(drive);
    }

    /**
     * Set goal of the drive collection. In nearly all collection, the goal is
     * fail. Remove old goal, add and emit new children.
     *
     * @param goal new goal of the DC
     */
    public void setGoalNode(Goal goal) {
        this.goal.remove();

        this.goal = goal;
        goal.setParent(this);

        // Emit the goal and its children
        emitChildNode(goal);
    }

    /**
     * Get goal of the DC
     *
     * @return goal of the DC
     */
    public Goal getGoal() {
        return goal;
    }

    /**
     * Get list of all drives of this DC in correct order (drive with higest
     * priority is first, drive with lowest priority is last).
     *
     * @return unmodifiable list of drives.
     */
    public List<DriveElement> getDrives() {
        return elementsUm;
    }

    /**
     * Serialize DC into a parser readable form.
     *
     * @return multi-line string that parser can read.
     */
    @Override
    public String toString() {
        String res;
        res = "\t(DC " + name + " " + goal.toString() + '\n';
        res += "\t\t(drives \n";
        for (DriveElement element : elements) {
            // Keep the extra braces for compatibility
            res += "\t\t\t  (" + element.toString() + ")\n";
        }
        res += "\t\t)\n";
        res += "\t)";
        return res;
    }

    /**
     * Get all child nodes of the DC. It consists of goal (at first place) and
     * all drives of DC from second place forward (in correct order).
     *
     * @return all children of this DC,
     */
    @Override
    public List<PoshElement> getChildDataNodes() {
        List<PoshElement> children = new ArrayList<PoshElement>(elements);

        children.add(0, goal);

        return children;
    }

    /**
     * Get name of the DC
     * @return name of the DC
     */
    public String getName() {
        return name;
    }

    /**
     * Change name of the DC and notify property listeners.
     *
     * @param newName New name of the DC.
     * @throws InvalidNameException throw if name is not valid (spaces,
     * braces...)
     */
    public void setName(String newName) throws InvalidNameException {
        newName = newName.trim();
        if (newName.matches(IDENT_PATTERN)) {
            String oldName = name;
            name = newName;
            firePropertyChange(dcName, oldName, name);
        } else {
            throw new InvalidNameException("Name " + newName + " is not valid.");
        }
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return moveNodeInList(elements, child, relativePosition);
    }

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement child) throws DuplicateNameException {
        if (child instanceof Goal) {
            setGoalNode((Goal) child);
        } else if (child instanceof DriveElement) {
            addDrive((DriveElement) child);
        } else {
            throw new UnexpectedElementException("Class " + child.getClass().getSimpleName() + " not expected.");
        }
    }

    @Override
    public void neutralizeChild(PoshElement child) {
        if (child == goal) {
            setGoalNode(Goal.createFail());
        } else if (elements.contains(child)) {
            if (elements.size() == 1) {
                String unusedName = getUnusedName("drive-", elementsUm);
                try {
                    addDrive(LapElementsFactory.createDriveElement(unusedName));
                } catch (DuplicateNameException ex) {
                    String msg = "Unused name " + unusedName + " is not unused.";
                    Logger.getLogger(DriveCollection.class.getName()).log(Level.SEVERE, msg, ex);
                    throw new FubarException(msg, ex);
                }
            }
            elements.remove(child);
            child.remove();
        } else {
            throw new UnexpectedElementException("Not expecting " + child.toString());
        }
    }
}
