package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import cz.cuni.amis.pogamut.sposh.exceptions.DuplicateNameException;
import cz.cuni.amis.pogamut.sposh.exceptions.InvalidNameException;
import java.beans.PropertyChangeEvent;
import java.util.*;
import org.junit.Test;
import static org.junit.Assert.*;
import org.junit.Before;

/**
 * Tests for {@link Competence}. Some are trivial/wrappers and thus skipped.
 *
 * @author HonzaH
 */
public class CompetenceTest {

    @Test
    public void testAddElement() throws DuplicateNameException {
        System.out.println("addElement");
        Competence instance = new Competence("test", new FormalParameters(), Collections.<CompetenceElement>emptyList());
        DummyListener listener = new DummyListener();
        instance.addElementListener(listener);

        CompetenceElement element = CompetenceElement.create();
        instance.addElement(element);

        assertTrue(listener.lastChild == element && listener.lastEvent == PoshTreeEvent.NEW_CHILD_NODE);
    }

    @Test(expected = DuplicateNameException.class)
    public void testAddElementDuplicateName() throws DuplicateNameException {
        System.out.println("addElement - duplicate name");
        Competence instance = new Competence("test", new FormalParameters(), Collections.<CompetenceElement>emptyList());
        instance.addElement(CompetenceElement.create());
        instance.addElement(CompetenceElement.create());
    }

    /**
     * Could happen during drag&drop - what if dragged is not released from original.
     * @throws DuplicateNameException 
     */
    @Test(expected = AssertionError.class)
    public void testAddElementLeakingParent() throws DuplicateNameException {
        System.out.println("addElement - leaking parent");

        Competence instance = new Competence("test", new FormalParameters(), Collections.<CompetenceElement>emptyList());
        instance.addElement(CompetenceElement.create("test-element"));
        
        Competence leak = new Competence("leak", new FormalParameters(), Collections.<CompetenceElement>emptyList());
        CompetenceElement leakElement = CompetenceElement.create("leak-element");
        leak.addElement(leakElement);
        
        instance.addElement(leakElement);
    }

    @Test
    public void testSetName() throws Exception {
        System.out.println("setName");
        String orgName = "test-set-name";
        String newName = "new-name";

        Competence instance = new Competence(orgName, new FormalParameters(), Collections.<CompetenceElement>emptyList());;
        DummyListener listener = new DummyListener();
        instance.addElementListener(listener);

        instance.setName(newName);

        PropertyChangeEvent evt = listener.lastPropertyChange;
        assertTrue(evt.getPropertyName().equals(Competence.cnName) && evt.getOldValue().equals(orgName) && evt.getNewValue().equals(newName));
        assertEquals(instance.getName(), newName);
    }

    @Test(expected = InvalidNameException.class)
    public void testSetNameInvalidName() throws Exception {
        System.out.println("setName - invalid name");
        String orgName = "test-set-name";
        String newName = "new name";

        Competence instance = new Competence(orgName, new FormalParameters(), Collections.<CompetenceElement>emptyList());;
        DummyListener listener = new DummyListener();
        instance.addElementListener(listener);

        try {
            instance.setName(newName);
        } catch (InvalidNameException ex) {
            assertNull(listener.lastPropertyChange);
            assertEquals(instance.getName(), orgName);
            
            throw ex;
        }
        fail("Name shouldn't be valid");
    }

    public static class DummyListener implements PoshElementListener {

        public PoshTreeEvent lastEvent;
        public PoshElement lastChild;
        public PropertyChangeEvent lastPropertyChange;

        @Override
        public void nodeChanged(PoshTreeEvent event, PoshElement child) {
            lastEvent = event;
            lastChild = child;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            lastPropertyChange = evt;
        }
    }
    List<CompetenceElement> elements;
    List<DummyListener> listeners;
    int num = 10;
    int reps = 100;

    @Before
    public void setUp() {
        elements = new ArrayList<CompetenceElement>();
        listeners = new ArrayList<DummyListener>();

        for (int i = 0; i < num; ++i) {
            CompetenceElement element = new CompetenceElement("e" + i, new Sense("t" + i), "a" + i, null);
            DummyListener listener = new DummyListener();
            listeners.add(listener);
            element.addElementListener(listener);
            elements.add(element);
        }
    }

    @Test
    public void testNeutralizeChild() {
        System.out.println("neutralizeChild");
        Competence instance = new Competence("testC", new FormalParameters(), elements);

        for (int i = 0; i < num; ++i) {
            CompetenceElement element = elements.get(i);

            assertTrue(instance.getChildDataNodes().size() == num - i);

            assertTrue(instance.getChildDataNodes().contains(element));
            instance.neutralizeChild(element);
            assertFalse(instance.getChildDataNodes().contains(element));

            // it is no longer among children
            assertFalse(instance.getChildDataNodes().contains(element));
            // its listeners were notified
            assertTrue(listeners.get(i).lastEvent == PoshTreeEvent.NODE_DELETED
                    && listeners.get(i).lastChild == elements.get(i));
        }
        assertEquals(instance.getChildDataNodes().size(), 1);
    }
}
