package cz.cuni.amis.pogamut.sposh.executor;

import cz.cuni.amis.pogamut.sposh.engine.VariableContext;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.logging.Logger;

/**
 * This class is an {@link IWorkExecutor}, that considers actions to be have state,
 * in this executor, primitive is working in phases INIT, RUN*, DONE.
 * <p/>
 * Thanks to state nature of primitive, primitive is far easier to program 
 * in discrete time-slice (time slice = primitive can be called multiple times consequentially)
 * nature of Pogamut. If primitive wasn't notified that it is about to end,
 * things like switching from primitive "walk" to "shoot" could be troublesome.
 * Imagine it, "walk" primitive will compute path to destination and suddently
 * new primitive "shoot" is now called. What about path the bot is following? It
 * will still follow it, although it is supposed to stop and start shooting.
 * To handle this correctlty is troublesome for few states, for many states, it is madness.
 * <p/>
 * StateWorkExecutor would do this: {@code ..., walk.INIT, walk.RUN*, walk.DONE, shoot.INIT, shoot.RUN*, shoot.DONE....},
 * primitive walk would have DONE called before shoot.INIT would be called,
 * allowing it to stop walking. Same thing is valid for state shoot too.
 * <p/>
 * Since we have phase DONE to cleanup some stuff before we switch to another,
 * where another can be nearly anything, what state bot should be in when DONE
 * phase is DONE. In neutral bot state (precise neutral state is defined by programmer,
 * in unreal, that would probably be standing, not shooting.).
 * <p/>
 * What if we don't want to switch to neutral bot state after primitive is DONE?
 * Don't, there is no explicit need, and in many situation it is meaningless (such as
 * primtive "enter_ducts" where bot would entering INIT in standing state, but left 
 * DONE crouching).
 *
 * @see IAction
 * @author Honza
 */
public class StateWorkExecutor implements IWorkExecutor {

    /**
     * Map that maps primtive name to {@link IAction}.
     */
    protected final HashMap<String, IAction> actions = new HashMap<String, IAction>();
    /**
     * Map that maps primitive name to its respective {@link ISense}.
     */
    protected final HashMap<String, ISense> senses = new HashMap<String, ISense>();
    /**
     * Primitive that is currently being executed. If no such primitive, null is used.
     */
    protected String current;

    /**
     * Log where we put
     */
    protected Logger log;
    
    public StateWorkExecutor() {
        this.log = Logger.getLogger(getClass().getSimpleName());
    }

    public StateWorkExecutor(Logger log) {
        this.log = log;
    }

    /**
     * Get logger of this {@link IWorkExecutor}.
     * @return
     */
    public Logger getLogger() {
        return log;
    }

    /**
     * Is name used in some primtive in this the work executor.
     * @param name queried name
     * @return if the name is not yet used in the {@link StateWorkExecutor}.
     */
    public synchronized boolean isNameUsed(String name) {
        return isSense(name) || isAction(name);
    }

    /**
     * Is there an action with the name.
     * @param name queries name
     */
    protected boolean isAction(String name) {
        return actions.containsKey(name);
    }

    /**
     * Is there a sense with the name.
     * @param name queries name
     */
    protected boolean isSense(String name) {
        return senses.containsKey(name);
    }

    /**
     * Add new {@link IAction} with primitive name.
     * @param name name that will be used for this {@link IAction} in posh plan.
     * @param action primitive that will be executed when executor will be asked to execute primtive with name.
     * @throws IllegalArgumentException if primitive with name already exists in {@link StateWorkExecutor}.
     */
    public synchronized void addAction(String name, IAction action) {
        if (isNameUsed(name)) {
            throw new IllegalArgumentException("Primtive with name \"" + name + "\" is already present in executor.");
        }
        actions.put(name, action);
    }

    /**
     * Add new {@link ISense} with primtive name.
     * @param name name of primtive to be associated with passed sense object
     * @param sense sense object to be used, when sense with name is supposed to execute.
     */
    public synchronized void addSense(String name, ISense sense) {
        if (isNameUsed(name)) {
            throw new IllegalArgumentException("Primtive with name \"" + name + "\" is already present in executor.");
        }
        senses.put(name, sense);
    }

    /**
     * Add primitive, use name from annotations.
     * @param action primitive that will be added
     */
    public void addAction(IAction action) {
        throw new UnsupportedOperationException("Not supported yet.");
    }


    public void addSense(ISense sense) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Execute 
     * @param primitive
     * @param ctx
     * @return
     */
    @Override
    public synchronized Object executePrimitive(String primitive, VariableContext ctx) {
        // if primtive to execute is sense, query it
        if (isSense(primitive)) {
            return senses.get(primitive).query(ctx);
        }

        if (!isAction(primitive)) {
            throw new IllegalArgumentException("Primitive \"" + primitive + "\" is not specified in the worker.");
        }

        // is it action?
        // get current primtive
        IAction currentPrimitive = actions.get(current);
        // is primitive that is supposed to execute same as last time this was called?
        boolean samePrimtive = current == null ? primitive == null : current.equals(primitive);
        // do we switch actions?
        if (!samePrimtive) {
            if (currentPrimitive != null) {
                log.info(MessageFormat.format("{0}.DONE({1})", current, ctx));
                currentPrimitive.done(ctx);
            }
            current = primitive;
            currentPrimitive = actions.get(current);
            log.info(MessageFormat.format("{0}.INIT({1})", current, ctx));
            currentPrimitive.init(ctx);
        }
        Object result = currentPrimitive.run(ctx);
        log.info(MessageFormat.format("{0}.RUN({1}) = {2}", current, ctx, result));
        return result;
    }
}
