package cz.cuni.pogamut.shed.presenter;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.DriveCollection;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.LapChain;
import cz.cuni.amis.pogamut.sposh.elements.LapPath;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.Trigger;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.pogamut.shed.widget.ExpandedActionEnvelope;
import cz.cuni.pogamut.shed.widget.ShedActionsEnvelope;
import cz.cuni.pogamut.shed.widget.ShedChoicesEnvelope;
import cz.cuni.pogamut.shed.widget.ShedScene;
import cz.cuni.pogamut.shed.widget.ShedSenseWidget;
import cz.cuni.pogamut.shed.widget.ShedTriggerEnvelope;
import cz.cuni.pogamut.shed.widget.ShedVariableWidget;
import cz.cuni.pogamut.shed.widget.ShedWidget;

/**
 * Factory that is creating various presenters used in the Shed visual view.
 *
 * @author Honza
 */
public class ShedPresenterFactory implements IPresenterFactory {

    /**
     * Scene that is showing all widgets and other stuff. Passed to presenters
     * so they can update scene and create other widgets.
     */
    private final ShedScene scene;
    /**
     * Presenter is giving access to name mapping of primitives.
     */
    private final ShedPresenter presenter;
    /**
     * Plan that is being presented.
     */
    private final PoshPlan plan;

    public ShedPresenterFactory(ShedScene scene, PoshPlan plan, ShedPresenter presenter) {
        this.plan = plan;
        this.scene = scene;
        this.presenter = presenter;
    }

    @Override
    public IPresenter createActionPresenter(LapPath actionPath, TriggeredAction action, ShedVariableWidget actionWidget, LapChain actionChain) {
        assert actionPath.traversePath(plan) == action;
        return new ActionPresenter(scene, presenter, action, actionWidget, actionChain);
    }

    /**
     * 
     * @param chain Chain up to the branch before @action. The @action is not included.
     * @return 
     */
    @Override
    public IPresenter createExpandedActionPresenter(TriggeredAction action, ExpandedActionEnvelope envelope, LapChain chain) {
        return new ExpandedActionPresenter(scene, presenter, envelope, action, chain);
    }

    @Override
    public IPresenter createSensePresenter(LapPath sensePath, Sense sense, ShedSenseWidget senseWidget, LapChain senseChain) {
        assert sensePath.traversePath(plan) == sense;
        return new SensePresenter(scene, presenter, sense, senseWidget, senseChain);
    }

    @Override
    public <TRIGGER_PARENT extends PoshElement> IPresenter createTriggerPresenter(TRIGGER_PARENT parent, Trigger<TRIGGER_PARENT> trigger, ShedTriggerEnvelope triggerEnvelope, LapChain chain) {
        return new TriggerPresenter<TRIGGER_PARENT>(scene, presenter, triggerEnvelope, parent, trigger, chain);
    }

    @Override
    public IPresenter createDriveCollectionPresenter(LapPath driveCollectionPath, DriveCollection driveCollection) {
        assert driveCollectionPath.traversePath(plan) == driveCollection;
        return new DCPresenter(scene, presenter, driveCollection);
    }

    @Override
    public IPresenter createActionPatternPresenter(LapPath actionPatternPath, TriggeredAction referencingAction, ActionPattern actionPattern, ShedVariableWidget actionPatternWidget, LapChain chain) {
        LapPath referencePath = actionPatternPath.subpath(0, actionPatternPath.length() - 1);
        assert referencePath.traversePath(plan) == referencingAction;
        assert actionPatternPath.traversePath(plan) == actionPattern;

        return new ActionPatternPresenter(scene, presenter, referencingAction, actionPattern, actionPatternWidget, chain);
    }

    @Override
    public IPresenter createCompetencePresenter(LapPath competencePath, TriggeredAction referencingAction, Competence competence, ShedVariableWidget competenceWidget, LapChain chain) {
        LapPath referencePath = competencePath.subpath(0, competencePath.length() - 1);
        assert referencePath.traversePath(plan) == referencingAction;
        assert competencePath.traversePath(plan) == competence;

        return new CompetencePresenter(scene, presenter, referencingAction, competence, competenceWidget, chain);
    }

    @Override
    public IPresenter createChoicePresenter(LapPath choicePath, CompetenceElement choice, ShedVariableWidget choiceWidget) {
        assert choicePath.traversePath(plan) == choice;
        return new ChoicePresenter(scene, presenter, choice, choiceWidget);
    }

    @Override
    public IPresenter createDrivePresenter(LapPath drivePath, DriveElement drive, ShedVariableWidget widget) {
        assert drivePath.traversePath(plan) == drive;
        return new DrivePresenter(scene, presenter, drive, widget);
    }

    @Override
    public IPresenter createActionsPresenter(LapPath actionPatternPath, ActionPattern actionPattern, ShedActionsEnvelope actionsEnvelope, LapChain chain) {
        assert actionPatternPath.traversePath(plan) == actionPattern;
        return new ActionsPresenter(scene, presenter, actionsEnvelope, actionPattern, chain);
    }

    @Override
    public IPresenter createChoicesPresenter(LapPath competencePath, Competence competence, ShedChoicesEnvelope choicesEnvelope, LapChain chain) {
        assert competencePath.traversePath(plan) == competence;
        return new ChoicesPresenter(scene, presenter, choicesEnvelope, competence, chain);
    }
}
