package cz.cuni.pogamut.shed.widget;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.util.LinkedList;
import java.util.List;
import org.netbeans.api.visual.widget.Scene;
import org.netbeans.api.visual.widget.Widget;

/**
 * This class represents a node in the YAPOSH plan that should show title text
 * on the first line and some variables on rest of lines.
 *
 * @author Honza
 */
public class ShedVariableWidget extends ShedWidget {

    private List<String> present = new LinkedList<String>();
    private List<String> missing = new LinkedList<String>();
    private List<String> unused = new LinkedList<String>();
    private final static Font titleFont = new Font("Helvetica", Font.BOLD, 16);
    private final static Font presentFont = new Font("Helvetica", Font.BOLD, 12);
    private final static Font missingFont = new Font("Helvetica", Font.BOLD, 12);
    private final static Font unusedFont = new Font("Helvetica", Font.ITALIC, 12);

    public ShedVariableWidget(ShedScene scene, String title, Color color) {
        super(scene, title, color);

        setPreferredBounds(null);
        setMinimumSize(new Dimension(ShedWidget.width, ShedWidget.height));
    }

    public final void setPresent(List<String> newPresent) {
        present.clear();
        present.addAll(newPresent);
        revalidate();
    }

    public final void setMissing(List<String> newPresent) {
        missing.clear();
        missing.addAll(newPresent);
        revalidate();
    }

    public final void setUnused(List<String> newPresent) {
        unused.clear();
        unused.addAll(newPresent);
        revalidate();
    }

    @Override
    protected Rectangle calculateClientArea() {
        Graphics2D gr = getGraphics();
        if (gr == null) {
            throw new IllegalStateException("Trying to compute client area without graphics2D.");
        }
        FontMetrics titleFontMetrics = gr.getFontMetrics(titleFont);
        FontMetrics boldFontMetrics = gr.getFontMetrics(presentFont);
        FontMetrics plainFontMetrics = gr.getFontMetrics(missingFont);
        FontMetrics italicFontMetrics = gr.getFontMetrics(unusedFont);

        Rectangle2D titleBounds = titleFontMetrics.getStringBounds(getDisplayName(), gr);
        Rectangle rectangle = roundRectangle(titleBounds);
        rectangle.x = 0;
        rectangle.y = 0;

        int widgetWidth = rectangle.width;
        int presentLinesWidth = getLinesWidth(present, boldFontMetrics, gr);
        if (widgetWidth < presentLinesWidth) {
            widgetWidth = presentLinesWidth;
        }
        int missingLinesWidth = getLinesWidth(missing, plainFontMetrics, gr);
        if (widgetWidth < missingLinesWidth) {
            widgetWidth = missingLinesWidth;
        }
        int unusedLinesWidth = getLinesWidth(unused, italicFontMetrics, gr);
        if (widgetWidth < unusedLinesWidth) {
            widgetWidth = unusedLinesWidth;
        }

        widgetWidth += titleFontMetrics.getMaxAdvance();
        widgetWidth += ShedWidget.textOfs;

        rectangle.width = ShedWidget.width;

        int widgetHeight = titleFontMetrics.getHeight();
        widgetHeight += boldFontMetrics.getHeight() * present.size();
        widgetHeight += plainFontMetrics.getHeight() * missing.size();
        widgetHeight += italicFontMetrics.getHeight() * unused.size();
        widgetHeight += italicFontMetrics.getDescent() * 2;


        rectangle.height = widgetHeight;

        return rectangle;
    }

    private static Rectangle roundRectangle(Rectangle2D rectangle) {
        int x1 = (int) Math.floor(rectangle.getX());
        int y1 = (int) Math.floor(rectangle.getY());
        int x2 = (int) Math.ceil(rectangle.getMaxX());
        int y2 = (int) Math.ceil(rectangle.getMaxY());
        return new Rectangle(x1, y1, x2 - x1, y2 - y1);
    }

    int getTitleWidth(FontMetrics fontMetrics, Graphics gr) {
        Rectangle2D titleBounds = fontMetrics.getStringBounds(getDisplayName(), gr);
        Rectangle rectangle = roundRectangle(titleBounds);
        return rectangle.width;
    }

    int getLinesWidth(List<String> lines, FontMetrics fontMetrics, Graphics gr) {
        int linesWidth = 0;
        for (String line : lines) {
            Rectangle lineBounds = roundRectangle(fontMetrics.getStringBounds(line, gr));
            int lineWidth = lineBounds.width;
            if (lineWidth > linesWidth) {
                linesWidth = lineWidth;
            }
        }
        return linesWidth;
    }
    private int intensity = 0;

    public void setActiveIntensity(int percent) {
        intensity = percent;
        this.repaint();
    }
    private static int BORDER_WIDTH = 4;

    @Override
    protected void paintWidget() {
        Graphics2D gr = getGraphics();
        gr.setColor(this.color);
        Rectangle clientArea = getClientArea();

        if (intensity > 0) {
            Color fullColor = Color.RED;
            
            Color borderColor = new Color(
                    color.getRed() + (fullColor.getRed() * intensity - color.getRed() * intensity) / 100 , 
                    color.getGreen() + (fullColor.getGreen() * intensity - color.getGreen() * intensity) / 100 ,
                    color.getBlue() + (fullColor.getBlue() * intensity - color.getBlue() * intensity) / 100 );
            
            gr.setColor(borderColor);
            gr.fillRect(clientArea.x, clientArea.y, clientArea.width, clientArea.height);
            gr.setColor(this.color);
            gr.fillRect(clientArea.x + BORDER_WIDTH, clientArea.y + BORDER_WIDTH, clientArea.width - 2 * BORDER_WIDTH, clientArea.height - 2 * BORDER_WIDTH);
        } else {
            gr.setColor(this.color);
            gr.fillRect(clientArea.x, clientArea.y, clientArea.width, clientArea.height);
        }

        if (showBreakpointStrip != ShowBreakpoint.NONE) {
            gr.setColor(showBreakpointStrip.getColor());
            gr.fillRect(clientArea.width - BREAKPOINT_STRIP_WIDTH, clientArea.y, BREAKPOINT_STRIP_WIDTH, clientArea.height);
        }

        int x = ShedWidget.textOfs;
        int y = 0;

        FontMetrics titleFontMetrics = gr.getFontMetrics(titleFont);
        FontMetrics boldFontMetrics = gr.getFontMetrics(presentFont);
        FontMetrics plainFontMetrics = gr.getFontMetrics(missingFont);
        FontMetrics italicFontMetrics = gr.getFontMetrics(unusedFont);

        gr.setColor(Color.BLACK);

        y = titleFontMetrics.getLeading() + titleFontMetrics.getAscent();
        gr.setFont(titleFont);
        drawFittingString(x, y, getDisplayName(), gr, titleFontMetrics);

        y += titleFontMetrics.getHeight();
        gr.setFont(presentFont);
        for (String presentLine : present) {
            drawFittingString(x, y, presentLine, gr, boldFontMetrics);
            y += boldFontMetrics.getHeight();
        }

        gr.setColor(Color.RED);
        gr.setFont(missingFont);
        for (String missingLine : missing) {
            drawFittingString(x, y, missingLine, gr, plainFontMetrics);
            y += plainFontMetrics.getHeight();
        }

        gr.setColor(Color.BLACK);
        gr.setFont(unusedFont);
        for (String missingLine : unused) {
            drawFittingString(x, y, missingLine, gr, italicFontMetrics);
            y += italicFontMetrics.getHeight();
        }
    }

    private void drawFittingString(int x, int y, String text, Graphics2D gr, FontMetrics fm) {
        AffineTransform originalTransform = gr.getTransform();
        gr.translate(x, y);
        String fittingText = getFittingString(text, fm, ShedWidget.width);
        gr.drawString(fittingText, 0, 0);
        gr.setTransform(originalTransform);
    }

    enum ShowBreakpoint {

        NONE(Color.WHITE),
        SINGLE(new Color(240, 72, 72)),
        PERMANENT(new Color(178, 15, 15));
        private final Color color;

        private ShowBreakpoint(Color color) {
            this.color = color;
        }

        public Color getColor() {
            return color;
        }
    }
    private ShowBreakpoint showBreakpointStrip = ShowBreakpoint.NONE;
    /**
     * Width of strip on the right side that indicates that this node has a
     * breakpoint.
     */
    final static int BREAKPOINT_STRIP_WIDTH = 10;

    /**
     * Show strip at the right side of the widget
     *
     * @since What color, dark red or light red
     */
    public void addBreakpoint(boolean single) {
        showBreakpointStrip = single ? ShowBreakpoint.SINGLE : ShowBreakpoint.PERMANENT;
        this.repaint();
    }

    public void removeBreakpoint() {
        showBreakpointStrip = ShowBreakpoint.NONE;
        this.repaint();
    }
}
