package cz.cuni.pogamut.posh.palette;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.CompetencePriorityElement;
import cz.cuni.amis.pogamut.sposh.elements.DriveCollection;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.DrivePriorityElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElementListener;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.pogamut.posh.palette.external.BehaviourInterfaceBuilder;
import java.beans.PropertyChangeEvent;
import java.lang.reflect.Method;
import org.openide.nodes.Children;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.openide.util.Lookup;

/**
 * List of all TriggeredActions in posh plan. This is listener of ALL nodes in POSH plan.
 * Whenever new TriggeredAction is created and added, I will get NEW CHILD NODE message.
 * 
 * Show it only if it isn't a duplicate and if it isn't Competence or AP
 * 
 * @author Honza
 */
public class ActionsChildren extends Children.SortedMap<TriggeredAction> implements PoshElementListener {

    private List<TriggeredAction> externalActions = new ArrayList<TriggeredAction>();
    private List<String> undefinedActions = new ArrayList<String>();
    private List<String> oldActions = new ArrayList<String>();
    private Lookup lookup;

    ActionsChildren(Lookup lookup) {
        this.lookup = lookup;
    }

    @Override
    public void nodeChanged(PoshTreeEvent event, PoshElement child) {
        refresh(child.getRootNode());
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        refresh(((PoshElement) evt.getSource()).getRootNode());
    }

    public synchronized void refresh(PoshPlan plan) {
        // Get all actions
        List<TriggeredAction> actions = getAllActions(plan);
        // Get their unique names
        List<String> actionNames = new ArrayList<String>();
        for (TriggeredAction action : actions) {
            String actionName = action.getActionName();
            if (!actionNames.contains(actionName) && plan.isUniqueAPorComp(actionName)) {
                actionNames.add(actionName);
            }
        }
        // remove same names as have the external
        removeDuplicateActions(actionNames, externalActions);


        // Update old actions = actions that were at some point defined in the plan
        // but are not in current external actions or list of current actions defined in plan
        oldActions.addAll(undefinedActions);
        oldActions.removeAll(actionNames);
        removeDuplicateActions(oldActions, externalActions);

        // Sort them.
        Collections.sort(actionNames);

        // remove all old actions in palette
        for (TriggeredAction ta : this.nodes.keySet().toArray(new TriggeredAction[0])) {
            remove(ta);
        }
        undefinedActions.clear();

        // Add normal actions
        for (String actionName : actionNames) {
            TriggeredAction ta = new TriggeredAction(actionName);

            put(ta, new ActionNode(ta));
            refreshKey(ta);

            undefinedActions.add(actionName);
        }
        // add old actions
        for (String oldActionName : oldActions) {
            TriggeredAction oa = new TriggeredAction(oldActionName);
            this.put(oa, new OldActionNode(oa));
        }
        // Add external one
        for (TriggeredAction ta : externalActions) {
            this.put(ta, new ExternalActionNode(ta));
        }
        refresh();
        new Thread(new BehaviourInterfaceBuilder(lookup)).start();
    }

    private List<TriggeredAction> getAllActions(PoshPlan plan) {
        List<TriggeredAction> actions = new LinkedList<TriggeredAction>();

        List<ActionPattern> aps = plan.getActionPatterns();
        List<Competence> competences = plan.getCompetences();

        for (ActionPattern ap : aps) {
            for (TriggeredAction action : ap.getTriggeredActions()) {
                actions.add(action);
            }
        }
        for (Competence competence : competences) {
            for (CompetencePriorityElement cpe : competence.getPriorityElements()) {
                for (CompetenceElement ce : cpe.getElements()) {
                    actions.add(ce.getAction());
                }
            }
        }

        DriveCollection dc = plan.getDriveCollection();
        for (DrivePriorityElement dpe : dc.getPriorityElements()) {
            for (DriveElement de : dpe.getDriveElements()) {
                actions.add(de.getTriggeredAction());
            }
        }

        return actions;
    }

    /**
     * Get list of actions that are used in the posh tree, but
     * are not defined in some external behaviour library.
     * @return unmodifiable list of such actions
     */
    public synchronized List<String> getUndefinedActions() {
        return Collections.unmodifiableList(undefinedActions);
    }

    public synchronized void setExternalActions(List<Method> externalMethods) {
        // Take old external actions and remove them from category
        removeActions(externalActions);
        externalActions.clear();

        // Add new external actions
        for (Method m : externalMethods) {
            TriggeredAction ta = new TriggeredAction(m.getName());
            externalActions.add(ta);
            this.put(ta, new ExternalActionNode(ta));
        }
    }

    private void removeActions(List<TriggeredAction> list) {
        for (TriggeredAction action : list) {
            remove(action);
        }
    }

    /**
     * remove all external actions from action names
     * @param actionNames
     * @param tobeRemoved
     */
    private void removeDuplicateActions(List<String> actionNames, List<TriggeredAction> tobeRemoved) {
        for (String actionName : actionNames.toArray(new String[0])) {
            for (TriggeredAction ta : tobeRemoved) {
                if (ta.getActionName().equals(actionName)) {
                    actionNames.remove(ta.getActionName());
                }
            }
        }
    }

    void removeOldAction(TriggeredAction oldAction) {
        oldActions.remove(oldAction.getActionName());
        remove(oldAction);
    }
}
