/*
 * Copyright (C) 2013 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh;

import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh.drawing.IUT2004ServerProvider;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh.drawing.NavMeshDraw;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.utils.NullCheck;

/**
 * Main class of Navigation Mesh module
 * 
 * @author Jakub Tomek
 * @author Jakub Gemrot aka Jimmy
 */
public class NavMeshModule {
	
	private IWorldView worldView;	
	private Logger log;
	
	private NavMeshDraw navMeshDraw;
	
	//
	// STATE
	//
	
	private boolean loaded = false;
	private GameInfo loadedForMap = null;
	
	//
	// NAVMESH DATA STRUCTURES
	//
	
    private NavMesh navMesh;
    
    //
    // LISTENER
    //
    
    private IWorldObjectEventListener<GameInfo, IWorldObjectEvent<GameInfo>> gameInfoListener = new IWorldObjectEventListener<GameInfo, IWorldObjectEvent<GameInfo>>() {
		@Override
		public void notify(IWorldObjectEvent<GameInfo> event) {
			load(event.getObject());
		}
	};
	
    public NavMeshModule(IUT2004ServerProvider serverProvider, IWorldView worldView, IAgentLogger logger) {
    	if (logger == null) {
    		log = new LogCategory("NavMesh");
    	} else {
    		log = logger.getCategory("NavMesh");
    	}
    	
    	this.worldView = worldView;
    	NullCheck.check(this.worldView, "worldView");
    	
    	navMesh = new NavMesh(worldView, log);
    	
    	worldView.addObjectListener(GameInfo.class, WorldObjectFirstEncounteredEvent.class, gameInfoListener);
    	
    	GameInfo info = worldView.getSingle(GameInfo.class);
    	if (info != null) {
    		load(info);
    	}
    	
    	navMeshDraw = new NavMeshDraw(navMesh, log, serverProvider);
    }
    
    private void clear() {
    	log.warning("NavMesh has been cleared...");
    	
    	navMesh.clear();
        
        loaded = false;
        loadedForMap = null;
    }
    	
    private void load(GameInfo info) {
    	if (info == null) {
    		log.severe("Could not load for 'null' GameInfo!");
    		return;
    	}
        if (loaded) {
        	if (loadedForMap == null) {
        		// WTF?
        		clear();
        	} else {
        		if (loadedForMap.getLevel().equals(info.getLevel())) {
        			// ALREADY INITIALIZED FOR THE SAME LEVEL
        			return;
        		}
        	}
        }
        
        if (navMesh.load(info)) {
        	loaded = true;
        	loadedForMap = info;
        } else {
        	loaded = false;
        	loadedForMap = null;
        }

	}
        	
	// ================
    // PUBLIC INTERFACE
    // ================
    
    /**
     * Tells whether NavMesh has been initialized and {@link #getNavMesh()} is usable. 
     * @return
     */
    public boolean isInitialized() {
    	return loaded;
    }
    
    /**
     * Always non-null, always returns {@link NavMesh}. However, you should check {@link #isInitialized()} whether the {@link NavMesh} is usable.
     * @return
     */
    public NavMesh getNavMesh() {
    	return navMesh;
    }
    
    public NavMeshDraw getNavMeshDraw() {
    	return navMeshDraw;
    }
    
}
