package cz.cuni.amis.pogamut.ut2004.bot.impl;

import java.util.Random;
import java.util.logging.Level;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathPlanner;
import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.AnnotationListenerRegistrator;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.EventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectListener;
import cz.cuni.amis.pogamut.base.utils.math.DistanceUtils;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.unreal.agent.navigation.IUnrealPathExecutor;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.AgentConfig;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Raycasting;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentStats;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Game;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.ItemDescriptors;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Items;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.NavigationGraphBuilder;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Players;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Senses;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPrefs;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UT2004AStarPathPlanner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.UT2004PathExecutor;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.floydwarshall.FloydWarshallMap;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.loquenavigator.KefikRunner;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.loquenavigator.LoqueNavigator;
import cz.cuni.amis.pogamut.ut2004.bot.IUT2004BotController;
import cz.cuni.amis.pogamut.ut2004.bot.command.AdvancedLocomotion;
import cz.cuni.amis.pogamut.ut2004.bot.command.CompleteBotCommandsWrapper;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.GetPath;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.AutoTraceRay;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.PathList;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.communication.translator.itemdescriptor.ItemDescriptor;

/**
 * The most advanced controller that is available. This controller contains all useful modules instantiated.
 * <p><p>
 * Modules currently available:<p>
 * <ol>
 * <li></li>
 * </ol>
 * 
 * @author Jimmy
 *
 * @param <BOT>
 */
public class UT2004BotModuleController<BOT extends UT2004Bot> extends UT2004BotLogicController<BOT> {

	/**
	 * Random number generator that is usually useful to have during decision making.
	 */
	protected Random random = new Random(System.currentTimeMillis());
	
	/**
	 * Memory module specialized on general info about the game - game type, time limit, frag limit, etc.
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Game game;
	
	/**
	 * Memory module specialized on general info about the agent whereabouts - location, rotation, health, current weapon, who is enemy/friend, etc.
	 * <p><p>
	 * May be used since first {@link Self} message is received, i.e, since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected AgentInfo info;
	
	/**
	 * Memory module specialized on whereabouts of other players - who is visible, enemy / friend, whether bot can see anybody, etc.
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Players players;
	
	/**
	 * Sensory module that provides mapping between {@link ItemType} and {@link ItemDescriptor} providing
     * an easy way to obtain item descriptors for various items in UT2004.
     * <p><p>
     * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected ItemDescriptors descriptors;
	
	/**
	 * Memory module specialized on items on the map - which are visible and which are probably spawned.
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Items items;
	
	/**
	 * Memory module specialized on agent's senses - whether the bot has been recently killed, collide with level's geometry, etc.
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Senses senses;
	
	/**
	 * Memory module specialized on info about the bot's weapon and ammo inventory - it can tell you which weapons are loaded, melee/ranged, etc.
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Weaponry weaponry;
	
	/**
	 * Memory module specialized on the agent's configuration inside UT2004 - name, vision time, manual spawn, cheats (if enabled at GB2004).
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected AgentConfig config;
	
	/**
	 * Support for creating rays used for raycasting (see {@link AutoTraceRay} that is being utilized).
	 * <p><p>
	 * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage)}
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected Raycasting raycasting;
	
	/**
	 * Wraps all available commands that can be issued to the virtual body of the bot inside UT2004.
     * <p><p>
     * May be used since since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected CompleteBotCommandsWrapper body;
	
	/**
	 * Shortcut for <i>body.getAdvancedShooting()</i> that allows you to shoot at opponent.
	 * <p><p>
	 * Note: more weapon-handling methods are available through {@link UT2004BotModuleControllerNew#weaponry}.
	 * <p><p>
	 * May be used since since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
	 * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected ImprovedShooting shoot;
	
	/**
	 * Shortcut for <i>body.getAdvancedLocomotion()</i> that allows you to manually steer the movement through the environment.
	 * <p><p>
	 * Note: navigation is done via {@link UT2004BotModuleControllerNew#pathExecutor} that needs {@link PathHandle} from the {@link UT2004BotModuleControllerNew#pathPlanner}.
	 * <p><p>
	 * May be used since since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
	 * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
	 */
	protected AdvancedLocomotion move;
	
	/**
     * Executor is used for following a path in the environment.
     * <p><p>
     * May be used since since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializePathFinding(UT2004Bot)}.
     */
	protected IUnrealPathExecutor<ILocated> pathExecutor = null;
	
    /**
     * Planner used to compute the path (consisting of navigation points) inside the map.
     * <p><p>
     * May be used since since the first {@link IUT2004BotController#botSpawned(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage, Self)} 
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializePathFinding(UT2004Bot)}.
     */
    protected IPathPlanner<ILocated> pathPlanner = null;
    
    /**
     * Navigation graph builder that may be used to manually extend the navigation graph of the UT2004.
     * <p><p>
     * May be used since {@link IUT2004BotController#botInitialized(cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo, ConfigChange, InitedMessage)} 
     * is called.
     * <p><p>
     * Initialized inside {@link UT2004BotModuleController#initializeModules(UT2004Bot)}.
     */
    protected NavigationGraphBuilder navBuilder = null;
    
    /**
     * Listener registrator that probes declared methods for the presence of {@link EventListener}, {@link ObjectClassEventListener},
     * {@link ObjectClassListener}, {@link ObjectEventListener} and {@link ObjectListener} annotations and automatically registers
     * them as listeners on a specific events.
     * <p><p>
     * Note that this registrator is usable for 'this' object only! It will work only for 'this' object.
     */
    protected AnnotationListenerRegistrator listenerRegistrator;
    
    /**
     * Weapon preferences for your bot. See {@link WeaponPrefs} class javadoc. It allows you to define preferences for
     * weapons to be used at given distance (together with their firing mode).
     */
    protected WeaponPrefs weaponPrefs;
    
    /**
     * Shortcut for the {@link UT2004BotModuleController#getWorldView()}.
     */
    protected IVisionWorldView world;
    
    /**
     * Shortcut for the {@link UT2004BotModuleController#getAct()}.
     */
    protected IAct act;
    
    /**
     * Module that is providing various statistics about the bot. You may also used it to output these stats (in CSV format)
     * into some file using {@link AgentStats#startOutput(String)} or {@link AgentStats#startOutput(String, boolean)}.
     */
    protected AgentStats stats;

    /**
     * Path-planner ({@link IPathPlanner} using {@link NavPoint}s), you may use it to find paths inside the environment wihtout
     * waiting for round-trip of {@link GetPath} command and {@link PathList}s response from UT2004. It is much faster than 
     * {@link UT2004BotModuleController#pathPlanner} but you need to pass {@link NavPoint} instances to planner instead of
     * {@link ILocated} ... to find the nearest {@link NavPoint} instance, {@link DistanceUtils} is a handy, check especially
     * {@link DistanceUtils#getNearest(java.util.Collection, ILocated)}.
     */
	protected FloydWarshallMap fwMap;
	
    @Override
	public void initializeController(BOT bot) {    	
		super.initializeController(bot);
		world = getWorldView();
		act = getAct();
		initializeModules(bot);
		initializePathFinding(bot);
		initializeListeners(bot);
	}
	
    /**
     * Initializes {@link UT2004BotModuleControllerNew#listenerRegistrator} and calls {@link AnnotationListenerRegistrator#addListeners()} method
     * to probe all declared methods for event-annotation presence.
     * @param bot
     */
	protected void initializeListeners(BOT bot) {
		listenerRegistrator = new AnnotationListenerRegistrator(this, getWorldView(), bot.getLogger().getCategory("Listeners"));
		listenerRegistrator.addListeners();
	}

	/**
	 * Initializes path-finding modules: {@link UT2004BotModuleControllerNew#pathPlanner}, {@link UT2004BotModuleController#fwMap} and {@link UT2004BotModuleControllerNew#pathExecutor}.
	 * If you need different path planner / path executor - override this method and initialize your own modules.
	 * @param bot
	 */
	protected void initializePathFinding(BOT bot) {
		pathPlanner  = new UT2004AStarPathPlanner(bot);
		fwMap        = new FloydWarshallMap(bot);
		pathExecutor = 
        	new UT2004PathExecutor<ILocated>(
        		bot, 
        		new LoqueNavigator<ILocated>(bot, 
        			new KefikRunner(bot, info, move, log), 
        		log)
        	);
	}

	/**
	 * Initializes memory/command modules of the bot.
	 * 
	 * @param bot
	 */
	protected void initializeModules(BOT bot) {
		game        = new Game(bot);
		info        = new AgentInfo(bot, game);
		players     = new Players(bot);
		descriptors = new ItemDescriptors(bot);
		weaponry    = new Weaponry(bot, descriptors);
		items       = new Items(bot, info, game, weaponry, null);
		senses      = new Senses(bot, info, players);
		config      = new AgentConfig(bot);
		raycasting  = new Raycasting(bot);
		body        = new CompleteBotCommandsWrapper(bot);		
		shoot       = new ImprovedShooting(weaponry, bot, body.getLog());
		move        = body.getLocomotion();
		navBuilder  = new NavigationGraphBuilder(bot);
		stats       = new AgentStats(bot);
		weaponPrefs = new WeaponPrefs(weaponry, bot);
	}
	
	@Override
	public void finishControllerInitialization() {
		if (navBuilder.isUsed()) {
			log.info("Navigation graph has been altered by 'navBuilder', triggering recomputation of Floyd-Warshall path matrix...");
			Level oldLevel = fwMap.getLog().getLevel();
			fwMap.getLog().setLevel(Level.FINER);
			fwMap.refreshPathMatrix();
			fwMap.getLog().setLevel(oldLevel);
		}
	}

	//
	//
	// MODULE GETTERS
	//
	//
	
	public Random getRandom() {
		return random;
	}

	public Game getGame() {
		return game;
	}

	public AgentInfo getInfo() {
		return info;
	}

	public Players getPlayers() {
		return players;
	}

	public ItemDescriptors getDescriptors() {
		return descriptors;
	}

	public Items getItems() {
		return items;
	}

	public Senses getSenses() {
		return senses;
	}

	public Weaponry getWeaponry() {
		return weaponry;
	}

	public AgentConfig getConfig() {
		return config;
	}

	public Raycasting getRaycasting() {
		return raycasting;
	}

	public CompleteBotCommandsWrapper getBody() {
		return body;
	}

	public ImprovedShooting getShoot() {
		return shoot;
	}

	public AdvancedLocomotion getMove() {
		return move;
	}

	public IUnrealPathExecutor<ILocated> getPathExecutor() {
		return pathExecutor;
	}

	public IPathPlanner<ILocated> getPathPlanner() {
		return pathPlanner;
	}

	public NavigationGraphBuilder getNavBuilder() {
		return navBuilder;
	}

	public WeaponPrefs getWeaponPrefs() {
		return weaponPrefs;
	}

	public IVisionWorldView getWorld() {
		return world;
	}

	@Override
	public IAct getAct() {
		return bot.getAct();
	}

	public AgentStats getStats() {
		return stats;
	}

	public FloydWarshallMap getFwMap() {
		return fwMap;
	}
	
	

}
