package cz.cuni.amis.pogamut.udk.utils;

import java.util.List;
import java.util.logging.Level;

import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.agent.params.impl.RemoteAgentParameters;
import cz.cuni.amis.pogamut.base.agent.utils.runner.impl.AgentRunner;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutPlatform;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.udk.agent.params.UDKAgentParameters;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Class used for creating, connecting and starting servers with default settings that are taken from the properties.
 * <p><p>
 * The address where the instances will connect are defined either in the constructor
 * or taken from the properties of the {@link PogamutPlatform}.
 * <p><p>
 * For more information about the class see {@link AgentRunner}.
 * 
 * @author ik
 * @author Jimmy
 */
public class UDKServerRunner<SERVER extends IUDKServer, PARAMS extends UDKAgentParameters> extends AgentRunner<SERVER, PARAMS> {

	/**
	 * Default host where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected String host;
    
    /**
	 * Default port where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected int port;
    
    /**
	 * Default name that will serve as a basis for {@link IAgentId}, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
	protected String name;

	/** 
	 * Construct the runner + specify all defaults.
	 * 
	 * @param factory to be used for creating new {@link IUDKServer} instances
	 * @param name default name that serve as a basis for {@link IAgentId}
	 * @param host default host where the instances are going to be connected
	 * @param port default port where the instances are going to be connected
	 */
	public UDKServerRunner(IAgentFactory<SERVER, PARAMS> factory, String name, String host, int port) {
        super(factory);
        this.name = name;
        this.port = port;
        this.host = host;
    }

	/**
	 * Construct the runner + specify the default name, host:port will be taken from the Pogamut platform properties.
	 * 
	 * @param factory factory to be used for creating new {@link IUDKServer} instances
	 * @param name default name that serve as a basis for {@link IAgentId}
	 */
    public UDKServerRunner(IAgentFactory<SERVER, PARAMS> factory, String name) {
        this(
        	factory, 
        	name, 
        	Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_HOST.getKey()), 
        	Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_PORT.getKey())
        );
    }
    
    @Override
    public SERVER startAgent() throws PogamutException {
    	return super.startAgent();
    }
    
    @Override
    public List<SERVER> startAgents(int count) throws PogamutException {
    	return super.startAgents(count);
    }
    
    @Override
    public List<SERVER> startAgents(PARAMS... agentParameters) throws PogamutException {
    	return super.startAgents(agentParameters);
    };
    
    /**
     * Construct the runner without specifying anything as default. Default name for server agents will be "UTServer Factory"
     * and host:port will be taken from the Pogamut platform properties.
     * 
     * @param factory factory to be used for creating new {@link IUDKServer} instances
     */
    public UDKServerRunner(IAgentFactory<SERVER, PARAMS> factory) {
        this(factory, "UDKServer");
    }

    /**
     * Provides default parameters that is, {@link IAgentId} using {@link UDKServerRunner#name} and {@link SocketConnectionAddress}
     * using {@link UDKServerRunner#host} and {@link UDKServerRunner#port}.
     */
	@Override
	protected IAgentParameters newDefaultAgentParameters() {
		return new UDKAgentParameters().setAgentId(new AgentId(name)).setWorldAddress(new SocketConnectionAddress(host, port));
	}
	
}
