package cz.cuni.amis.pogamut.udk.agent.navigation;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutionEstimator;
import cz.cuni.amis.pogamut.base.agent.navigation.IStuckDetector;
import cz.cuni.amis.pogamut.base.agent.navigation.PathExecutorState;
import cz.cuni.amis.pogamut.base.agent.navigation.impl.BasePathExecutor;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.udk.agent.navigation.loquenavigator.LoqueNavigator;
import cz.cuni.amis.pogamut.udk.agent.navigation.timeoutestimator.UDKBasicTimeoutEstimator;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.SetRoute;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.utils.NullCheck;

public class UDKPathExecutor<PATH_ELEMENT extends ILocated> extends BasePathExecutor<PATH_ELEMENT> {

	private IUDKPathNavigator<PATH_ELEMENT> navigator;
	
	private UDKBot bot;
	
	private Self self;
	
	private long pathExecutionStart = Long.MIN_VALUE;
	
	private double pathExecutionTimeout = Double.POSITIVE_INFINITY;
	
	private IWorldObjectEventListener<Self, WorldObjectFirstEncounteredEvent<Self>> selfListener = new IWorldObjectEventListener<Self, WorldObjectFirstEncounteredEvent<Self>>() {
		@Override
		public void notify(WorldObjectFirstEncounteredEvent<Self> event) {
			self = event.getObject();
		}
	};
	
	private IWorldEventListener<EndMessage> endMessageListener = new IWorldEventListener<EndMessage>() {
		@Override
		public void notify(EndMessage event) {
			eventEndMessage();
		}
	};

	private IPathExecutionEstimator<PATH_ELEMENT> timeoutEstimator;

	public UDKPathExecutor(UDKBot bot) {
		this(bot, null, null);
	}
	
	public UDKPathExecutor(UDKBot bot, IUDKPathNavigator<PATH_ELEMENT> navigator) {
		this(bot, navigator, null);
	}
	
	public UDKPathExecutor(UDKBot bot, IUDKPathNavigator<PATH_ELEMENT> navigator, Logger log) {
		super(log);
		if (getLog() == null) {
			setLog(bot.getLogger().getCategory(getClass().getSimpleName()));
		}
		NullCheck.check(bot, "bot");
		this.bot = bot;		
		this.navigator = navigator;
		if (this.navigator == null) {
			this.navigator = new LoqueNavigator<PATH_ELEMENT>(bot, getLog());
		}
		this.navigator.setBot(bot);
		this.navigator.setExecutor(this);
		bot.getWorldView().addObjectListener(Self.class, WorldObjectFirstEncounteredEvent.class, selfListener);
		bot.getWorldView().addEventListener(EndMessage.class, endMessageListener);
		this.timeoutEstimator = new UDKBasicTimeoutEstimator<PATH_ELEMENT>();
	}
	
	public UDKPathExecutor<PATH_ELEMENT> setTimeoutEstimator(IPathExecutionEstimator<PATH_ELEMENT> timeoutEstimator) {
		this.timeoutEstimator = timeoutEstimator;
		return this;
	}
	
	@Override
	protected void stopped() {		
	}
	
	@Override
	protected void followPathImpl() {		
	}
	
	/**
	 * If the path is not zero-length, recalls {@link IUDKPathNavigator#newPath(List)}
	 * and set the path into the GB2004 via {@link SetRoute}.
	 */
	@Override
	protected void pathComputedImpl() {
		if (getPath().size() == 0) {
			targetReached();
		} else {
			bot.getAct().act(new SetRoute().setRoute(getPath()));
			navigator.newPath(getPath());
			pathExecutionStart = System.currentTimeMillis();
			calculateTimeout();
		}
	}

	@Override
	protected void pathComputationFailedImpl() {
	}
	
	@Override
	protected void stuckImpl() {
		navigator.reset();
	}

	/**
	 * Sets the path into the GB2004 via {@link SetRoute} whenever switch occurs and the rest of the path is greater than
	 * 32 path elements.
	 */
	@Override
	protected void switchToAnotherPathElementImpl() {
		List<PATH_ELEMENT> path = getPath();
		if (path.size() > 31 + getPathElementIndex()) {
			List<PATH_ELEMENT> pathPart = new ArrayList<PATH_ELEMENT>(32);
			for (int i = getPathElementIndex(); i < path.size() && i < getPathElementIndex() + 31; ++i) {
				pathPart.add(path.get(i));
			}
			bot.getAct().act(new SetRoute().setRoute(pathPart));
		}
	}
	
	protected void calculateTimeout() {
		IPathExecutionEstimator<PATH_ELEMENT> estimator = timeoutEstimator;
		if (estimator != null) {
			pathExecutionTimeout = estimator.getTimeout(getPath());
		} else {
			pathExecutionTimeout = Long.MAX_VALUE;
		}

	}

	@Override
	protected void targetReachedImpl() {
		navigator.reset();
	}
	
	protected void eventEndMessage() {
		if (inState(PathExecutorState.PATH_COMPUTED) || inState(PathExecutorState.SWITCHED_TO_ANOTHER_PATH_ELEMENT)) {
			navigate();
		}
	}
	
	protected void navigate() {
		if (log != null && log.isLoggable(Level.FINER)) log.finer("navigating");
		double timeDelta = System.currentTimeMillis() - pathExecutionStart; 
		if (timeDelta > pathExecutionTimeout) {
			if (log != null && log.isLoggable(Level.WARNING)) log.finer("TIMEOUT! (" + pathExecutionTimeout + "ms)");
			stuck();
			return;			
		}
		IStuckDetector detector = checkStuckDetectors();
		if (detector != null) {
			if (log != null && log.isLoggable(Level.INFO)) log.info(detector.getClass().getSimpleName() + " has reported that the bot has stuck");
			stuck();
		} else {
			navigator.navigate();
		}
	}
	
}
