package cz.cuni.amis.pogamut.udk.agent.module.logic;

import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.agent.exceptions.AgentException;
import cz.cuni.amis.pogamut.base.agent.module.IAgentLogic;
import cz.cuni.amis.pogamut.base.agent.module.LogicModule;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.react.EventReact;
import cz.cuni.amis.pogamut.base.communication.worldview.react.EventReactOnce;
import cz.cuni.amis.pogamut.base.communication.worldview.react.ObjectEventReact;
import cz.cuni.amis.pogamut.base.communication.worldview.react.ObjectEventReactOnce;
import cz.cuni.amis.pogamut.base.component.bus.event.BusAwareCountDownLatch;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencyType;
import cz.cuni.amis.pogamut.base.component.exception.ComponentCantStartException;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GamePaused;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GameResumed;

import java.util.logging.Level;

public class UDKBotLogic<BOT extends UDKBot> extends LogicModule<BOT> {

	private EventReact<GamePaused>        pauseReaction;
	private EventReact<GameResumed>       resumeReaction;
	private ObjectEventReact<GameInfo, ?> gameInfoReaction;
	private EventReact<EndMessage>        endReaction;
	private BusAwareCountDownLatch        latch;
	
	@Inject
	public UDKBotLogic(BOT agent, IAgentLogic logic) {
		this(agent, logic, null, new ComponentDependencies(ComponentDependencyType.STARTS_WITH).add(agent.getWorldView()));
	}
	
	public UDKBotLogic(BOT agent, IAgentLogic logic, Logger log) {
		this(agent, logic, log, new ComponentDependencies(ComponentDependencyType.STARTS_WITH).add(agent.getWorldView()));
	}
	
	public UDKBotLogic(BOT agent, IAgentLogic logic, Logger log, ComponentDependencies dependencies) {
		super(agent, logic, log, dependencies);
		pauseReaction = new EventReact<GamePaused>(GamePaused.class, agent.getWorldView()) {
			@Override
			protected void react(GamePaused event) {
				controller.manualPause("Game paused.");
			}
		};
		resumeReaction = new EventReact<GameResumed>(GameResumed.class, agent.getWorldView()) {
			@Override
			protected void react(GameResumed event) {
				controller.manualResume("Game resumed.");
			}
		};
		gameInfoReaction = new ObjectEventReactOnce<GameInfo, IWorldObjectEvent<GameInfo>>(GameInfo.GameInfoId, agent.getWorldView()) {
			@Override
			protected void react(IWorldObjectEvent<GameInfo> event) {
				if (event.getObject().isBotsPaused() || event.getObject().isGamePaused()) {
					controller.manualPause("Bot launched into paused game.");
				}
			}
		};
		endReaction = new EventReactOnce<EndMessage>(EndMessage.class, agent.getWorldView()) {
			@Override
			protected void react(EndMessage event) {
				latch.countDown();
			}
		};		
	}
	
	@Override
	protected void logicLatch(String threadName) {
		super.logicLatch(threadName);
		if (log.isLoggable(Level.FINE)) log.fine(threadName + ": Waiting for the first End message.");
		if (!latch.await(60, TimeUnit.SECONDS)) {
			throw new ComponentCantStartException("End message was not received in 60secs.", this);
		}		
		if (log.isLoggable(Level.INFO)) log.info(threadName + ": First END message received, starting logic cycles.");
	}
	
	@Override
	protected void start(boolean startPaused) throws AgentException {
		super.start(startPaused);
		gameInfoReaction.enable();
		endReaction.enable();
		latch = new BusAwareCountDownLatch(1, agent.getEventBus(), agent.getWorldView().getComponentId());		
	}

}
