package cz.cuni.amis.pogamut.udk.agent.module.utils;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.BeginMessage;

/**
 * This class is a simple implementation of TabooSet (similar to TabooList). It allows you to insert
 * items that are taboo either for infinite time (via {@link TabooSet#add(Object)}) or for a specified
 * amount of UT2004 time (via {@link TabooSet#add(Object, double)}).
 * <p><p>
 * Items inside taboo set are either removed automatically due to a timeout (a specified amount of time has passed)
 * or manually via {@link TabooSet#remove(Object)}.
 */
public class TabooSet<T> {
	
	/**
	 * Map of tabu items together with their time until which they are valid (negative time == valid for infinity).
	 */
	private Map<T, Double> taboo = new HashMap<T, Double>();	
	
	/**
	 * Adds a tabu item that is valid for an infinite amount of time.
	 * @param item
	 */
	public void add(T item) {
		taboo.put(item, (double)-1);
	}
	
	/**
	 * Adds a tabu item that is valid for a period of 'timeout' time (in seconds).
	 * @param item
	 * @param timeout in seconds
	 */
	public void add(T item, double timeout) {
		taboo.put(item, time+timeout);
	}
	
	/**
	 * Removes a tabu item from the set.
	 * @param item
	 */
	public void remove(T item) {
		taboo.remove(item);
	}
	
	/**
	 * Determines whether an 'item' is inside tabu set or not based on the current
	 * UT2004 time.
	 * @param item
	 * @return
	 */
	public boolean isTaboo(T item) {
		if (taboo.containsKey(item)) {
			double tabuTime = taboo.get(item);
			if (tabuTime < 0) {
				return true;
			}
			if (tabuTime < time) {
				taboo.remove(item);
				return false;
			} else {
				return true;
			}
		} else {
			return false;
		}
	}
	
	/**
	 * Filters collection according to the current state of the tabu set. It returns a new hash set
	 * containing items from 'collection' that are not inside tabu set.
	 * 
	 * @param collection
	 * @return
	 */
	public Set<T> filter(Collection<T> collection) {
		Set<T> set = new HashSet<T>();
		for (T t : collection) {
			if (isTaboo(t)) continue;
			set.add(t);
		}
		return set;
	}
	
	/**
	 * Clears the taboo set.
	 */
	public void clear() {
		taboo.clear();
	}
	
	/**
	 * Returns current UT2004 time that is used by the TabuSet.
	 */
	public double getTime() {
		return time;
	}
	
	/**
	 * Current UT2004 time updated by {@link TabooSet#beginMessageListener}.
	 */
	private double time;
	
	/**
	 * {@link BeginMessage} listener that updates {@link TabooSet#time}.
	 * @author Jimmy
	 */
	private class BeginMessageListener implements IWorldEventListener<BeginMessage> {

		
		public BeginMessageListener(IWorldView worldView) {
			worldView.addEventListener(BeginMessage.class, this);
		}

		@Override
		public void notify(BeginMessage event) {
			time = event.getTime();
		}
		
	};
	
	/**
	 * {@link BeginMessage} listener that updates {@link TabooSet#time}.
	 */
	BeginMessageListener beginMessageListener;
	
	/**
	 * Constructor of the TabuSet.
	 * @param bot
	 */
	public TabooSet(UDKBot bot) {
		beginMessageListener = new BeginMessageListener(bot.getWorldView());
	}

}
