package cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectDestroyedEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.batchClock.SimulationTime;
import cz.cuni.amis.utils.listener.IListener;
import cz.cuni.amis.utils.listener.Listeners;

/** Implementation of world object registry
 * 
 * @author Paletz
 */
public class WorldObjectRegistry implements IWorldObjectRegistry {

	/** World object listing
	 */
	protected ObjectListing<WorldObjectId, IWorldObject> worldObjectListing;
	
	/** World object event  (added/changed/removed) listeners
	 */
	protected Listeners<IListener<IWorldObjectEvent<?>>> worldObjectEventListeners;
	
	public WorldObjectRegistry() {
		worldObjectListing = new ObjectListing<WorldObjectId, IWorldObject>();
		worldObjectEventListeners = new Listeners<IListener<IWorldObjectEvent<?>>>();
	}
	
	@Override
	public IObjectListing<WorldObjectId, IWorldObject> getWorldObjectListing() {
		return worldObjectListing;
	}
	
	@Override
	public void registerObjectEventListener( IListener<IWorldObjectEvent<?>> listener) {
		worldObjectEventListeners.addStrongListener( listener );
	}
	
	@Override
	public void forgetObjectEventListener(IListener<IWorldObjectEvent<?>> listener) {
		worldObjectEventListeners.removeListener( listener );
	}
	
	@Override
	public void clear() {
		worldObjectListing.clear();
	}
	
	@Override
	public void applyWorldObjectEvent(IWorldChangeEvent event) {
		SimulationTime time = SimulationTime.fromMilliSeconds( event.getSimTime() );
		
		if ( event instanceof WorldObjectFirstEncounteredEvent ) {
			registerObject( ((WorldObjectFirstEncounteredEvent<?>) event).getObject(), time );
		} else if ( event instanceof WorldObjectUpdatedEvent ) {
			notifyObjectUpdated( ((WorldObjectUpdatedEvent<?>) event).getObject(), time );
		} else if ( event instanceof WorldObjectDestroyedEvent ) {
			forgetObject( ((WorldObjectDestroyedEvent<?>) event).getObject(), time );
		}
	}
	
	@Override
	public void registerObject( IWorldObject object, SimulationTime time ) {
		assert( object.getId() != null );
		
		worldObjectListing.add( object.getId(), object );
		worldObjectEventListeners.notify( 
			new IListener.Notifier<IListener<IWorldObjectEvent<?>>>(
				new WorldObjectFirstEncounteredEvent<IWorldObject>( object, time.getMilliSeconds() )
			)
		);
		worldObjectEventListeners.notify(
			new IListener.Notifier<IListener<IWorldObjectEvent<?>>>(
				new WorldObjectUpdatedEvent<IWorldObject>( object, time.getMilliSeconds() )
			)
		);
	}
	
	@Override
	public void notifyObjectUpdated( IWorldObject object, SimulationTime time ) {
		worldObjectEventListeners.notify( 
			new IListener.Notifier<IListener<IWorldObjectEvent<?>>>(
				new WorldObjectUpdatedEvent<IWorldObject>( object, time.getMilliSeconds() )
			)
		);
	}
	
	@Override
	public void forgetObject( IWorldObject object, SimulationTime time ) {
		worldObjectEventListeners.notify( 
			new IListener.Notifier<IListener<IWorldObjectEvent<?>>>(
				new WorldObjectDestroyedEvent<IWorldObject>( object, time.getMilliSeconds() )
			)
		);

		worldObjectListing.remove( object.getId() );		
	}
}
