package cz.cuni.amis.pogamut.emohawk.communication.worldView;

import java.util.Collection;
import java.util.Map;

import com.google.inject.Inject;
import com.google.inject.name.Named;

import cz.cuni.amis.pogamut.base.communication.mediator.IMediator;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base.component.bus.IComponentBus;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.batchClock.ISimulationClock;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.eventBus.IListenerRegistry;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry.IObjectListing;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry.IViewableObjectBeliefRegistry;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.IHistoricWorldObjectUpdater;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.iface.worldObject.IViewableObjectBelief;
import cz.cuni.amis.pogamut.ut2004.communication.worldview.UT2004SyncLockableWorldView;
import cz.cuni.amis.utils.listener.IListener;

/** Emohawk world view
 * 
 * Extended with the ability to track replicas
 * 
 * @author Paletz
 */
public class EmohawkWorldView extends UT2004SyncLockableWorldView implements IHistoricWorldView {

	public static final String WORLDVIEW_DEPENDENCY = "EmohawkWorldViewDependency";
	
	protected IHistoricWorldObjectUpdater historicWorldObjectUpdater;
	protected IViewableObjectBeliefRegistry viewableObjectBeliefRegistry;
	protected ISimulationClock simulationClock;
	
	@Inject
	public EmohawkWorldView(
		@Named(WORLDVIEW_DEPENDENCY) ComponentDependencies dependencies,
		IMediator mediator,
		IComponentBus bus,
		IAgentLogger log,
		IHistoricWorldObjectUpdater historicWorldObjectUpdater,
		IViewableObjectBeliefRegistry viewableObjectBeliefRegistry,
		ISimulationClock simulationClock
	) {
		super(dependencies, mediator, bus, log);
		this.historicWorldObjectUpdater = historicWorldObjectUpdater;
		this.viewableObjectBeliefRegistry = viewableObjectBeliefRegistry;
		this.simulationClock = simulationClock;
		
		historicWorldObjectUpdater.initWorldObjectRegistry( viewableObjectBeliefRegistry );
		historicWorldObjectUpdater.initSimulationClock( simulationClock );
		
		viewableObjectBeliefRegistry.registerObjectEventListener(
			new IListener<IWorldObjectEvent<?>>() {
				@Override
				public void notify(IWorldObjectEvent<?> event) {
					raiseEvent(event);
				}
			}
		);
	}
	
	
	
	
	
	
	// override legacy object API
	
	@Override
	public Map<WorldObjectId, IWorldObject> get() {
		return getSeenObjects().getMap();
	}
	
	@Override
	public IWorldObject get(WorldObjectId id) {
		return getSeenObjects().getById( id );
	}
	
	@Override
	public <T extends IWorldObject> T getSingle(Class<T> singletonClass) {
		Collection<T> objects = getSeenObjects().filterByClass( singletonClass ).values();
		assert( objects.size() == 1 );
		return objects.iterator().next();
	}
	
	@Override
	public <T extends IWorldObject> T get(WorldObjectId id, Class<T> filterClass) {
		return getSeenObjects().filterByClass( filterClass ).get( id );
	}
	
	@SuppressWarnings("rawtypes")
	@Override
	public Map<Class, Map<WorldObjectId, IWorldObject>> getAll() {
		Map<Class<? extends IWorldObject>, Map<WorldObjectId, IWorldObject>> retval = getSeenObjects().getClassGroupedMap();
		@SuppressWarnings("unchecked")
		Map<Class, Map<WorldObjectId, IWorldObject>> shitTypedRetval =  (Map<Class, Map<WorldObjectId, IWorldObject>>) (Map<?,?>) retval;
		return shitTypedRetval;
	}
	
	@Override
	public <T extends IWorldObject> Map<WorldObjectId, T> getAll(Class<T> filterClass) {
		return getSeenObjects().filterByClass( filterClass );
	}
	
	@Override
	public IViewable getVisible(WorldObjectId id) {
		return getVisibleObjects().getById( id );
	}
	
	@Override
	public Map<WorldObjectId, IViewable> getVisible() {
		return getVisibleObjects().getMap();
	}
	
	@SuppressWarnings("rawtypes")
	@Override
	public Map<Class, Map<WorldObjectId, IViewable>> getAllVisible() {
		Map<Class<? extends IViewable>, Map<WorldObjectId, IViewable>> retval = getVisibleObjects().getClassGroupedMap();
		@SuppressWarnings("unchecked")
		Map<Class, Map<WorldObjectId, IViewable>> shitTypedRetval =  (Map<Class, Map<WorldObjectId, IViewable>>) (Map<?,?>) retval;
		return shitTypedRetval;
	}
	
	@Override
	public <T extends IViewable> Map<WorldObjectId, T> getAllVisible( Class<T> filterClass ) {
		return getVisibleObjects().filterByClass( filterClass );
	}
	
	
	
	
	
	
	@Override
	public IObjectListing<WorldObjectId, IWorldObject> getSeenObjects() {
		return viewableObjectBeliefRegistry.getWorldObjectListing();
	}
	
	@Override
	public IObjectListing<WorldObjectId, IViewable> getVisibleObjects() {
		return viewableObjectBeliefRegistry.getVisibleObjectListing();
	}

	@Override
	public IObjectListing<WorldObjectId, IViewableObjectBelief> getPlausibleBeliefs() {
		return viewableObjectBeliefRegistry.getPlausibleBeliefs();
	}

	
	
	@Override
	public double getShortTermMemoryDuration() {
		return historicWorldObjectUpdater.getShortTermMemoryDuration();
	}

	@Override
	public void setShortTermMemoryDuration(double durationSeconds) {
		historicWorldObjectUpdater.setShortTermMemoryDuration( durationSeconds );
	}
	
	@Override
	public void contradict(IViewableObjectBelief belief) {
		historicWorldObjectUpdater.contradict( belief );
	}
	
	
	
	
	
	
	// pass events to historic world object updater too
	
	@Override
    protected void innerNotify(IWorldChangeEvent event) {
		simulationClock.applyWorldChange( event );
		historicWorldObjectUpdater.applyWorldObjectUpdate( event );
		super.innerNotify(event);
    }
	
	@Override
	protected void cleanUp() {
		super.cleanUp();
		viewableObjectBeliefRegistry.clear();
	}
	
	
	
	
	
	
	// redirect world object tracking to world object belief registry
	
	@Override
	protected void objectCreated( IWorldObject worldObject ) {
		viewableObjectBeliefRegistry.registerObject( worldObject, simulationClock.getTime() );
	}
	
	@Override
	protected void objectDestroyed(IWorldObject worldObject ) {
		viewableObjectBeliefRegistry.forgetObject( worldObject, simulationClock.getTime() );
	}
	
	@Override
	protected void objectAppeared( IViewable worldObject ) {
		viewableObjectBeliefRegistry.notifyObjectUpdated( worldObject, simulationClock.getTime() );
		currentObjectBatch.add( worldObject );
	}
	
	@Override
	protected void objectDisappeared( IViewable worldObject ) {
		viewableObjectBeliefRegistry.notifyObjectUpdated( worldObject, simulationClock.getTime() );
	}






	@Override
	public IListenerRegistry accessEvents() {
		// delegate to worldview itself
		
		return new IListenerRegistry() {
			public void addEventListener(Class<?> eventClass,
				IWorldEventListener<?> listener) {
				EmohawkWorldView.this.addEventListener(eventClass, listener);
			}

			public void addObjectListener(Class<?> objectClass,
				IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.addObjectListener(objectClass, listener);
			}

			public void addObjectListener(Class<?> objectClass,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.addObjectListener(objectClass, eventClass, listener);
			}

			public void addObjectListener(WorldObjectId objectId,
				IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.addObjectListener(objectId, listener);
			}

			public void addObjectListener(WorldObjectId objectId,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.addObjectListener(objectId, eventClass, listener);
			}

			public void removeEventListener(Class<?> eventClass,
				IWorldEventListener<?> listener) {
				EmohawkWorldView.this.removeEventListener(eventClass, listener);
			}

			public void removeObjectListener(Class<?> objectClass,
				IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.removeObjectListener(objectClass, listener);
			}

			public void removeObjectListener(Class<?> objectClass,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this
					.removeObjectListener(objectClass, eventClass, listener);
			}

			public void removeObjectListener(WorldObjectId objectId,
				IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.removeObjectListener(objectId, listener);
			}

			public void removeObjectListener(WorldObjectId objectId,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				EmohawkWorldView.this.removeObjectListener(objectId, eventClass, listener);
			}

			public void removeListener(IWorldEventListener<?> listener) {
				EmohawkWorldView.this.removeListener(listener);
			}

			public boolean isListening(Class<?> eventClass,
				IWorldEventListener<?> listener) {
				return EmohawkWorldView.this.isListening(eventClass, listener);
			}

			public boolean isListening(Class<?> objectClass,
				IWorldObjectEventListener<?, ?> listener) {
				return EmohawkWorldView.this.isListening(objectClass, listener);
			}

			public boolean isListening(Class<?> objectClass,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				return EmohawkWorldView.this.isListening(objectClass, eventClass, listener);
			}

			public boolean isListening(WorldObjectId objectId,
				IWorldObjectEventListener<?, ?> listener) {
				return EmohawkWorldView.this.isListening(objectId, listener);
			}

			public boolean isListening(WorldObjectId objectId,
				Class<?> eventClass, IWorldObjectEventListener<?, ?> listener) {
				return EmohawkWorldView.this.isListening(objectId, eventClass, listener);
			}

			public boolean isListening(IWorldEventListener<?> listener) {
				return EmohawkWorldView.this.isListening(listener);
			}
		};
	}
}
