package cz.cuni.amis.pogamut.emohawk.agent.beliefContradictor;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.emohawk.agent.EhAgentInfo;
import cz.cuni.amis.pogamut.emohawk.agent.losChecker.ILosChecker;
import cz.cuni.amis.pogamut.emohawk.agent.losChecker.LosReportEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.IPureHistoricWorldView;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.iface.worldObject.IViewableObjectBelief;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.utils.listener.IListener;

/** Contradicts beliefs by detecting object absence at believed location
 * 
 * @author Paletz
 */
public class AbsenceBeliefContradictor {
	
	protected ILosChecker losChecker;
	protected HashMap<WorldObjectId,Importance> importanceMap; // world object ID -> importance
	protected EhAgentInfo agentInfo;
	protected IPureHistoricWorldView worldView;
	
	protected int losChecksPerFrame;
	protected double losCheckExposingRadius;
	
	/** Constructor
	 * 
	 * @param losChecker line-of-sight checker
	 * @param agentInfo agentInfo info
	 * @param worldView world view
	 * @param losChecksPerFrame los checks per frame limit
	 * @param losCheckExposingRadius radius of area around successful line-of-sight check that is considered to be visible 
	 */
	public AbsenceBeliefContradictor( 
		ILosChecker losChecker,
		EhAgentInfo agentInfo,
		IPureHistoricWorldView worldView,
		int losChecksPerFrame,
		double losCheckExposingRadius
	) {
		this.losChecker = losChecker;
		this.agentInfo = agentInfo;
		this.worldView = worldView;
		this.losChecksPerFrame = losChecksPerFrame;
		this.losCheckExposingRadius = losCheckExposingRadius;
		
		importanceMap = new HashMap<WorldObjectId,Importance>();
		
		worldView.accessEvents().addEventListener( EndMessage.class, endMessageListener );
		losChecker.registerLosReportListener( losReportListener );
	}
	
	public void dispose() {
		worldView.accessEvents().removeEventListener( EndMessage.class, endMessageListener );
		losChecker.forgetLosReportListener( losReportListener );
	}
	
	public void overrideImportance( WorldObjectId id, double importance, double duration ) {
		if ( !importanceMap.containsKey(id) ) {
			importanceMap.put( id, new Importance( agentInfo.getTime() ) );
		}
		
		importanceMap.get( id ).addOverride( new ImportanceOverride( importance, agentInfo.getTime()+duration ) );
	}
	
	protected void updateImportances() {
		// clean old overrides and 
		Iterator<Entry<WorldObjectId,Importance>> entryIterator = importanceMap.entrySet().iterator();
		while ( entryIterator.hasNext() ) {
			Entry<WorldObjectId,Importance> entry = entryIterator.next();
			Importance importance = entry.getValue();
			importance.cleanOldOverrides( agentInfo.getTime() );
			IViewableObjectBelief belief = worldView.getPlausibleBeliefs().getById( entry.getKey() );
			if ( importance.isDefaultImportance() && ( belief == null  || !belief.isPlausible() ) ) {
				entryIterator.remove();
			}
		}
		
		// log new observations
		for ( IViewable viewable : worldView.getVisibleObjects().getMap().values() ) {
			if ( viewable instanceof IViewableObjectBelief ) {
				if ( !importanceMap.containsKey( viewable.getId() ) ) {
					importanceMap.put( viewable.getId(), new Importance( agentInfo.getTime() ) );
				} else {
					importanceMap.get( viewable.getId() ).setLastLosCheckAttemptTime( agentInfo.getTime() );
				}
			}
		}
	}
	
	protected void requestLosChecks() {
		for ( int i=0; i < losChecksPerFrame; i++ ) {
			LocatedBelief locatedBelief = findBeliefWithTopLosCheckPriority();
			if ( locatedBelief == null ) {
				break;
			}
			losChecker.requestLosCheck( locatedBelief.getLocated().getLocation() );
			importanceMap.get( locatedBelief.getBelief().getId() ).setLastLosCheckAttemptTime( agentInfo.getTime() );
		}
	}
	
	protected void observe() {
		updateImportances();
		requestLosChecks();	
	}
	
	protected final IWorldEventListener<EndMessage> endMessageListener = new IWorldEventListener<EndMessage>() {
		@Override
		public void notify(EndMessage event) {
			observe();
		}
	};
	
	protected final IListener<LosReportEvent> losReportListener = new IListener<LosReportEvent>() {
		@Override
		public void notify(LosReportEvent event) {
			
			if ( !event.isInLos() ) {
				return;
			}
			
			for ( IViewableObjectBelief belief : worldView.getPlausibleBeliefs().getMap().values() ) {
				if ( !belief.isVisible() && belief.isPlausible() && belief instanceof ILocated ) {
					Location location = ((ILocated) belief).getLocation();
					if ( location.getDistance( event.getLocation() ) <= losCheckExposingRadius ) {
						contradict( belief );
					}
				}
			}
		}
	};
	
	protected void contradict( IViewableObjectBelief belief ) {
		worldView.contradict( belief );
	}
	
	/** Find belief with top ray-trace priority
	 * 
	 * @return belief with top ray-trace priority or null if no record needs to be ray-traced
	 */
	protected LocatedBelief findBeliefWithTopLosCheckPriority() {
		LocatedBelief beliefWithTopLosCheckPriority = null;
		double topLosCheckPriority = 0.0;
		for ( IViewableObjectBelief belief : worldView.getPlausibleBeliefs().getMap().values() ) {
			double losCheckPriority = 0.0;
			LocatedBelief locatedBelief = null;
			if ( belief instanceof ILocated ) {
				locatedBelief = new LocatedBelief( (ILocated) belief, belief );
				losCheckPriority = computeLosCheckPriority( locatedBelief );
			}
			
			if ( topLosCheckPriority < losCheckPriority ) {
				beliefWithTopLosCheckPriority = locatedBelief;
				topLosCheckPriority = losCheckPriority;
			}
		}
		
		return beliefWithTopLosCheckPriority;
	}
	
	protected double computeLosCheckPriority( LocatedBelief belief ) {
		if ( 
			belief.getLocated().getLocation() == null 
			||
			belief.getBelief().isVisible()
			||
			!belief.getBelief().isPlausible() 
		) {
			return 0.0;
		}
		
		if ( !losChecker.isWithinFov( belief.getLocated().getLocation() ) ) {
			return 0.0;
		}
		
		Importance importance = importanceMap.get( belief.getBelief().getId() );
		double timeSinceSeen = agentInfo.getTime() - belief.getBelief().getMemorization().getSnapshotTime();
		double timeSinceLosCheckAttempt = agentInfo.getTime() - importance.getLastLosCheckTime();
		double distance = Math.max( 50.0, agentInfo.getPawn().getLocation().getDistance( belief.getLocated().getLocation() ) );
		
		return Math.max( 0.0, importance.getImportance()*timeSinceSeen*timeSinceLosCheckAttempt/distance );
	}
		
	protected class Importance {
				
		protected List<ImportanceOverride> overrides;
		protected double lastLosCheckAttemptTime;
		
		public Importance( double currentTime ) {
			overrides = new LinkedList<ImportanceOverride>();
			lastLosCheckAttemptTime = currentTime;
		}
				
		public double getLastLosCheckTime() {
			return lastLosCheckAttemptTime;
		}

		public void setLastLosCheckAttemptTime( double value ) {
			lastLosCheckAttemptTime = value;
		}
		
		public double getImportance() {
			if ( overrides.isEmpty() ) {
				return 1.0;
			}
			
			return overrides.get(0).value;
		}
		
		public void addOverride( ImportanceOverride override ) {
			updateImportances();
			while ( overrides.size()>0 && overrides.get(0).overrideEndTime <= override.overrideEndTime ) {
				overrides.remove(0);
			}
			overrides.add( 0, override );
		}
		
		/** Clean outdated overrides
		 * 
		 * @param currentTime current time
		 */
		public void cleanOldOverrides( double currentTime ) {
			while ( overrides.size()>0 && overrides.get(0).overrideEndTime < currentTime ) {
				overrides.remove(0);
			}
		}
		
		/** Tell whether importance is default (not overriden)
		 * 
		 * @return true if importance is default (not overriden)
		 */
		public boolean isDefaultImportance() {
			return overrides.isEmpty();
		}
		
	}
	
	protected class ImportanceOverride {
		public double overrideEndTime;
		public double value;
		
		public ImportanceOverride( double value, double overrideEndTime ) {
			super();
			this.overrideEndTime = overrideEndTime;
			this.value = value;
		}
	}
	
	protected class LocatedBelief {
		protected ILocated located;
		protected IViewableObjectBelief belief;
		
		public LocatedBelief( ILocated located, IViewableObjectBelief belief ) {
			assert( located == belief );
			this.located = located;
			this.belief = belief;
		}
		
		public ILocated getLocated() {
			return located;
		}
		
		public IViewableObjectBelief getBelief() {
			return belief;
		}
	}
}