package cz.cuni.amis.pogamut.defcon.communication.worldview.modules.bitmap;

import java.awt.image.BufferedImage;
import java.util.Collections;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.defcon.communication.worldview.AbstractMapSource;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.BasicFlag;
import cz.cuni.amis.pogamut.defcon.utils.Pair;

/**
 * Map source using bitmaps from the Bitmaps/ directory to provide access to map
 * features.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class BitmapMapSource extends AbstractMapSource {

	private final BufferedImage[] territories;
	private final BufferedImage sailable;
	private final SortedMap<Integer, BufferedImage> ownTerritories =
			new TreeMap<Integer, BufferedImage>();
	private final SortedMap<Integer, SortedMap<Integer, BufferedImage>> enemyTerritories =
			new TreeMap<Integer, SortedMap<Integer, BufferedImage>>();

	private final int LAND_THRESHOLD = 200;
	private final int SEA_THRESHOLD = 100;
	private final double X_SPAN = 180d;
	private final double Y_SPAN = 100d;

	private Logger log;

	public BitmapMapSource(BufferedImage[] territories, int[] ownTerritories,
			SortedMap<Integer, int[]> enemyIdsAndTerritories,
			BufferedImage sailable) {
		this(territories, ownTerritories, enemyIdsAndTerritories, sailable,
				null);
	}

	public BitmapMapSource(BufferedImage[] territories, int[] ownTerritories,
			SortedMap<Integer, int[]> enemyIdsAndTerritories,
			BufferedImage sailable, Logger log) {

		this.log = log;
		this.territories = territories;
		this.sailable = sailable;
		for (int ownTerritory : ownTerritories) {
			this.ownTerritories.put(ownTerritory, territories[ownTerritory]);
		}

		for (int enemyId : enemyIdsAndTerritories.keySet()) {
			SortedMap<Integer, BufferedImage> enemyTerritoriesList = new TreeMap<Integer, BufferedImage>();
			this.enemyTerritories.put(enemyId, enemyTerritoriesList);
			for (int territoryId : enemyIdsAndTerritories.get(enemyId)) {
				enemyTerritoriesList.put(territoryId, territories[territoryId]);

				// if (log != null) {
				// log.info("enemyTerritory: "
				// + enemyId
				// + " " + territoryId);
				// }
			}
		}
	}

	public void setTerritorry(int index, BufferedImage bitmap) {
		territories[index] = bitmap;
	}

	public BufferedImage getTerritorry(int index) {
		return territories[index];
	}

	public SortedMap<Integer, BufferedImage> getOwnTerritories() {
		return Collections.unmodifiableSortedMap(ownTerritories);
	}

	private boolean overThresh(BufferedImage image, int x, int y,
			int overThreshold) {

		return (image.getRGB(x, y) & 0x000000ff) > overThreshold ? true : false;
	}

	private boolean underThresh(BufferedImage image, int x, int y,
			int underThreshold) {
		return !overThresh(image, x, y, underThreshold);
	}

	private Pair<Integer, Integer> convertToImageCoords(double x, double y,
			BufferedImage image) {
		Pair<Integer, Integer> result = new Pair<Integer, Integer>();

		result.first = (int) ((x + X_SPAN) / (2 * X_SPAN) * (image.getWidth() - 1));
		result.second = image.getHeight() - 1
				- (int) ((y + Y_SPAN) / (2 * Y_SPAN) * (image.getHeight() - 1));

		return result;
	}

	@Override
	public boolean hasFlag(double x, double y, BasicFlag flag) {
		if (x < -X_SPAN || x > X_SPAN || y < -Y_SPAN || y > Y_SPAN)
			return false;

		switch (flag) {
			case SEA: {
				Pair<Integer, Integer> coords = convertToImageCoords(
						x,
						y,
						sailable);
				return overThresh(sailable, coords.first, coords.second, 10);
			}
			case LAND: {
				Pair<Integer, Integer> coords = convertToImageCoords(
						x,
						y,
						sailable);
				return underThresh(sailable, coords.first,
						coords.second, 5);
			}
			case OWN_TERRITORY: {

				Boolean ok = false;
				for (BufferedImage ownTerritory : getOwnTerritories().values()) {

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							ownTerritory);

					ok = overThresh(ownTerritory, coords.first,
							coords.second, 10);
					if (ok)
						break;
				}


				return ok;
			}
			case OWN_PLACEABLE_LAND: {

				Boolean ok = false;
				for (BufferedImage ownTerritory : getOwnTerritories().values()) {

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							ownTerritory);

					ok = overThresh(ownTerritory, coords.first,
							coords.second, LAND_THRESHOLD);
					if (ok)
						break;
				}

				return ok;
			}
			case OWN_PLACEABLE_SEA: {
				Boolean ok = false;
				for (BufferedImage ownTerritory : getOwnTerritories().values()) {

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							ownTerritory);

					ok = overThresh(ownTerritory, coords.first,
							coords.second, SEA_THRESHOLD) &&
								underThresh(
										ownTerritory,
										coords.first,
										coords.second,
										LAND_THRESHOLD);
					if (ok)
						break;
				}

				return ok;
			}
			case ENEMY_TERRITORY: {

				Boolean ok = false;
				for (int i = 0; i < territories.length; ++i) {

					if (ownTerritories.containsKey(i))
						continue;

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							territories[i]);

					ok = overThresh(
							territories[i],
							coords.first,
							coords.second,
							10);
					if (ok)
						break;
				}
				return ok;
			}
			case ENEMY_PLACEABLE_SEA: {
				Boolean ok = false;
				for (int i = 0; i < territories.length; ++i) {

					if (ownTerritories.containsKey(i))
						continue;

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							territories[i]);

					ok = overThresh(territories[i], coords.first,
							coords.second, SEA_THRESHOLD) &&
								underThresh(
										territories[i],
										coords.first,
										coords.second,
										LAND_THRESHOLD);
					if (ok)
						break;
				}

				return ok;
			}
			case ENEMY_PLACEABLE_LAND: {
				Boolean ok = false;
				for (int i = 0; i < territories.length; ++i) {

					if (ownTerritories.containsKey(i))
						continue;

					Pair<Integer, Integer> coords = convertToImageCoords(
							x,
							y,
							territories[i]);

					ok = overThresh(
							territories[i],
							coords.first,
							coords.second,
							LAND_THRESHOLD);
					if (ok)
						break;
				}

				return ok;
			}

		}
		return false;
	}

	@Override
	public boolean hasFlag(Location location, BasicFlag flag) {
		return hasFlag(location.getX(), location.getY(), flag);
	}

	@Override
	public boolean hasEnemyTerritoryFlag(Location location, int enemyId) {
		return hasEnemyTerritoryFlag(
				location.getX(),
				location.getY(),
				enemyId);
	}

	@Override
	public boolean hasEnemyTerritoryFlag(double x, double y, int enemyId) {
		if (x < -X_SPAN || x > X_SPAN || y < -Y_SPAN || y > Y_SPAN)
			return false;

		boolean ok = false;
		for (BufferedImage territory : enemyTerritories.get(enemyId).values()) {

			Pair<Integer, Integer> coords = convertToImageCoords(
					x,
					y,
					territory);

			ok = overThresh(
					territory,
					coords.first,
					coords.second,
					SEA_THRESHOLD);

			if (ok)
				break;
		}

		return ok;
	}

	@Override
	public boolean hasEnemyTerritoryFlag(Location location, int enemyId,
			boolean seaArea) {
		return hasEnemyTerritoryFlag(
				location.getX(),
				location.getY(),
				enemyId, seaArea);
	}

	@Override
	public boolean hasEnemyTerritoryFlag(double x, double y, int enemyId,
			boolean seaArea) {
		if (x < -X_SPAN || x > X_SPAN || y < -Y_SPAN || y > Y_SPAN)
			return false;


		boolean ok = false;

		if (!enemyTerritories.containsKey(enemyId)) {
			if (log != null) {
				log.info("enemyId: " + enemyId);
			}
			return false;
		}

		if (seaArea) {

			for (BufferedImage territory : enemyTerritories.get(enemyId)
					.values()) {

				Pair<Integer, Integer> coords = convertToImageCoords(
						x,
						y,
						territory);


				ok = overThresh(
						territory,
						coords.first,
						coords.second,
						SEA_THRESHOLD) && underThresh(
								territory,
						coords.first,
						coords.second,
						LAND_THRESHOLD);

				if (ok)
					break;
			}

			return ok;

		} else {

			for (BufferedImage territory : enemyTerritories.get(enemyId)
					.values()) {

				Pair<Integer, Integer> coords = convertToImageCoords(
						x,
						y,
						territory);


				ok = overThresh(
						territory,
						coords.first,
						coords.second,
						LAND_THRESHOLD);

				if (ok)
					break;
			}

			return ok;

		}
	}

}
