package cz.cuni.amis.pogamut.defcon.communication.messages.infos;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldObjectUpdateResult;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldObjectUpdateResult.Result;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldObjectUpdateResult.WorldObjectUpdateResult;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.defcon.communication.messages.Updatable;
import cz.cuni.amis.utils.exception.PogamutException;

import java.lang.reflect.Field;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javabot.PogamutJBotSupport;


/**
 * Generic updater for arbitrary objects. All that you need is to annotate some fields with
 * {@link Updatable} annotation.<p></p>
 *  <p>Follows the philosophy of {@link IWorldObjectUpdateEvent}.</p>
 *
 * @author Jimmy
 */
public class DefConObjectUpdate implements IWorldObjectUpdatedEvent {
    /**
     * Maps of existing updaters (cache).
     */
    private static Map<Class<?>, Updater> updaters = new HashMap<Class<?>, Updater>();

    /**
     * TODO: FIX!
     */
    private double time = 0;

    /**
     * Source object of the update.
     */
    private DefConObject source;

/**
     * Creates new object update with 'source' as object that contains updates.
     *
     * @param source
     */
    public DefConObjectUpdate(DefConObject source) {
        this.source = source;
    }

    /**
     * Returns updater for class 'cls' from cache or creates new one.
     *
     * @param cls
     *
     * @return
     */
    public synchronized static Updater getUpdater(Class<?> cls) {
        Updater updater = updaters.get(cls);

        if (updater != null) {
            return updater;
        }

        updater = new Updater(cls);
        updaters.put(cls, updater);

        return updater;
    }

    /**
     * Returns source of the updates.
     *
     * @return
     */
    public DefConObject getSource() {
        return this.source;
    }

    /**
     * Returns ID of the object that should be updated.
     *
     * @return ID
     */
    @Override
    public WorldObjectId getId() {
        return (source == null) ? null : source.getId();
    }

    /**
     * Updates object 'obj' with the source event.
     *
     * @param obj object that should be updated
     *
     * @return result of the object update
     */
    @Override
    public IWorldObjectUpdateResult update(IWorldObject obj) {
        if (source == null) {
            return null;
        }

        if (obj == null) {
            return new WorldObjectUpdateResult(Result.CREATED, source);
        }

        if (!obj.getClass().equals(source.getClass())) {
            throw new PogamutException("Can't update object of class " + obj.getClass().getName() +
                " with information from object of class " + source.getClass().getName(), this);
        }

        if (source.getDestroyed()) {
            return new WorldObjectUpdateResult(Result.DESTROYED, obj);
        }

        getUpdater(obj.getClass()).update(obj, source);

        return new WorldObjectUpdateResult(Result.UPDATED, obj);
    }

    /**
     * Returns the time of the last update of the object.
     *
     * @return time
     */
    public double getLastSeenTime() {
        return time;
    }

    /**
     * Sets the last-seen-time of the object.
     *
     * @param time
     */
    protected void setLastSeenTime(double time) {
        this.time = time;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    @Override
    public long getSimTime() {
        return (long) time;
    }

    /**
     * Updater class that is using Java Reflection API to get the list of fields that has
     * to be updated over some class.
     *
     * @author Jimmy
     */
    private static class Updater {
        /**
         * Class that the updater handles
         */
        private Class<?> cls;

        /**
         * Fields of 'cls' that contains {@link Updatable} annotation.
         */
        private List<Field> fields = new ArrayList<Field>();

/**
         * Creates updater class for 'cls'
         *
         * @param cls
         */
        public Updater(Class<?> cls) {
            this.cls = cls;

            do {
                for (Field field : cls.getDeclaredFields()) {
                    if (field.isAnnotationPresent(Updatable.class)) {
                        fields.add(field);
                    }
                }
            } while ((cls = cls.getSuperclass()) != null);
        }

        /**
         * Returns class that updater handles.
         *
         * @return
         */
        public Class getUpdaterClass() {
            return cls;
        }

        /**
         * Updates 'oldObj' with {@link Updatable} fields of 'newObj'. Both object must
         * be of class 'cls' (returned by getUpdaterClass()).
         *
         * @param oldObj
         * @param newObj
         */
        public void update(Object oldObj, Object newObj) {
            if (!oldObj.getClass().equals(cls) || !newObj.getClass().equals(cls)) {
                throw new PogamutException("Can't update object of class " +
                    oldObj.getClass().getName() + " with information from object of class " +
                    newObj.getClass().getName() + " because they bot must be of class " +
                    cls.getName(), this);
            }

            for (Field field : fields) {
                try {
                    field.setAccessible(true);

                    Object value = field.get(newObj);

                    if (value == null) {
                        continue;
                    }

                    try {
                        // PogamutJBotSupport.writeToConsole("updating field " + field.toString() + " of " + oldObj.toString());
                        field.set(oldObj, value);
                    } catch (Exception e) {
                        throw new PogamutException("Could not set value " + value +
                            " to the field '" + field.getName() + "' of the object " + newObj +
                            " because: " + e.getMessage(), e, this);
                    }
                } catch (Exception e) {
                    throw new PogamutException("Could not obtain field '" + field.getName() +
                        "' from object " + newObj + " because: " + e.getMessage(), e, this);
                } finally {
                    field.setAccessible(false);
                }
            }
        }
    }
}
