package cz.cuni.amis.pogamut.base.utils.math;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.utils.IFilter;
import cz.cuni.amis.utils.Tuple2;

/**
 * DistanceUtils consists of usual routines for selecting "X" from some "collections of Xes" that are in some "distance" relation to the "target",
 * e.g., "get nearest weapon from collection of available weapons to my position".
 * <p><p>
 * Note that you may always use custom metric via {@link IGetDistance} interface.
 * <p><p>
 * Note that you may always use some custom filters via {@link IDistanceFilter} interface.
 * 
 * @author Jimmy
 * @author ik
 */
public class DistanceUtils {
	
	// --------------
	// =====================
	// METRICS
	// =====================
	// --------------
	
	/**
	 * Distance estimator between object of types T and some {@link Location} target.
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static interface IGetDistance<T> {
		
		public double getDistance(T object, ILocated target);
		
	}
	
	/**
	 * Simple implementation of {@link IGetDistance} that uses {@link Location#getDistance(Location)} method.
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static class GetLocatedDistance3D<T extends ILocated> implements IGetDistance<T> {

		/**
		 * Uses {@link Location#getDistance(Location)} method.
		 */
		@Override
		public double getDistance(T object, ILocated target) {
			if (object.getLocation() == null) return Double.MAX_VALUE;
			return object.getLocation().getDistance(target.getLocation());
		}
		
	}
	
	/**
	 * See {@link GetLocatedDistance3D}.
	 */
	public static final GetLocatedDistance3D<ILocated> getLocatedDistance3D = new GetLocatedDistance3D<ILocated>();
	
	/**
	 * Simple implementation of {@link IGetDistance} that uses {@link Location#getDistance2D(Location)} method.
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static class GetLocatedDistance2D<T extends ILocated> implements IGetDistance<T> {

		/**
		 * Uses {@link Location#getDistance(Location)} method.
		 */
		@Override
		public double getDistance(T object, ILocated target) {
			if (object.getLocation() == null) return Double.MAX_VALUE;
			return object.getLocation().getDistance2D(target.getLocation());
		}
		
	}
	
	/**
	 * See {@link GetLocatedDistance2D}.
	 */
	public static final GetLocatedDistance2D<ILocated> getLocatedDistance2D = new GetLocatedDistance2D<ILocated>();
	
	// --------------
	// =====================
	// FILTERS
	// =====================
	// --------------
	
	/**
	 * Filter that allows to check whether "object" is accepted with respect to "distanceToTarget" for given "target".
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static interface IDistanceFilter<T> {
		
		/**
		 * @param object
		 * @param target
		 * @param distanceToTarget
		 * @return TRUE == can be result, FALSE == filter out
		 */
		public boolean isAccepted(T object, ILocated target, double distanceToTarget);
		
	}
	
	
	/**
	 * Filter that accepts all "objects" (does not filter anything out).
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static final class AcceptAllDistanceFilter<T> implements IDistanceFilter<T> {

		@Override
		public boolean isAccepted(T object, ILocated target, double distanceToTarget) {
			return true;
		}
		
	}
	
	/**
	 * See {@link AcceptAllDistanceFilter}.
	 */
	@SuppressWarnings("unchecked")
	public static final AcceptAllDistanceFilter acceptAllDistanceFilter = new AcceptAllDistanceFilter();
	
	/**
	 * Filter that accepts all "objects" that are within range of min/max distance (inclusive).
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static class RangeDistanceFilter<T> implements IDistanceFilter<T> {

		private double minDistance;
		private double maxDistance;

		public RangeDistanceFilter(double minDistance, double maxDistance) {
			this.minDistance = minDistance;
			this.maxDistance = maxDistance;
		}
		
		@Override
		public boolean isAccepted(T object, ILocated target, double distanceToTarget) {
			return minDistance <= distanceToTarget && distanceToTarget <= maxDistance;
		}
		
	}
	
	/**
	 * Accepts only VISIBLE ({@link IViewable#isVisible()} == TRUE) objects.
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static class VisibleFilter<T extends IViewable> implements IDistanceFilter<T> {

		@Override
		public boolean isAccepted(T object, ILocated target, double distanceToTarget) {
			return object.isVisible();
		}
		
	}
	
	/**
	 * See {@link VisibleFilter}.
	 */
	public static final VisibleFilter<IViewable> visibleFilter = new VisibleFilter<IViewable>();
	
	/**
	 * Adapter that wraps {@link IFilter} making it into {@link IDistanceFilter}.
	 * 
	 * @author Jimmy
	 *
	 * @param <T>
	 */
	public static class FilterAdapter<T> implements IDistanceFilter<T> {
		
		private IFilter<T> filter;

		public FilterAdapter(IFilter<T> filter) {
			this.filter = filter;
		}

		@Override
		public boolean isAccepted(T object, ILocated target, double distanceToTarget) {
			return filter.isAccepted(object);
		}
		
	}
	
	// --------------
	// =====================
	// getNearest()
	// =====================
	// --------------
	
	/**
     * Returns the nearest object to 'target'. 
     * <p><p>
     * Distance is obtained via provided {@link IGetDistance#getDistance(Object, Location)}.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param getDistance distance computer between 'locations' and 'target'.
     * @return nearest object from collection of objects
     */
    public static <T> T getNearest(Collection<T> locations, ILocated target, IGetDistance getDistance) {
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (target.getLocation() == null) return null;
    	if (getDistance == null) return null;
        
        T nearest = null;        
        double minDistance = Double.MAX_VALUE;
        double d;
        
        for(T l : locations) {        	
        	d = getDistance.getDistance(l, target);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        
        return nearest;
    }
    
    /**
     * Returns the nearest object to 'target' that is accepted by all 'filters'.
     * <p><p> 
     * Distance is obtained via provided {@link IGetDistance#getDistance(Object, Location)}.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param getDistance distance computer between 'locations' and 'target'.
     * @param filters filters to be used (that can filter out unsuitable results)
     * @return nearest object from collection of objects
     */
    public static <T> T getNearest(Collection<T> locations, ILocated target, IGetDistance getDistance, IDistanceFilter... filters) {
    	if (filters == null || filters.length == 0 || (filters.length == 1 && filters[0] instanceof AcceptAllDistanceFilter)) {
    		return getNearest(locations, target, getDistance);
    	}
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (target.getLocation() == null) return null;
        if (getDistance == null) return null;
        
        T nearest = null;
        double minDistance = Double.MAX_VALUE;
        double d;
        
        for(T l : locations) {
        	d = getDistance.getDistance(l, target);
        	boolean accepted = true;
        	for (IDistanceFilter<T> filter : filters) {
        		if (!filter.isAccepted(l, target, d)) {
        			accepted = false;
        			break;
        		}
        	}
        	if (!accepted) {
        		continue;
        	}
            if (d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }
    
    /**
     * Returns the nearest object to 'target' that is accepted by all 'filters'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filters
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearest(Collection<T> locations, ILocated target, IDistanceFilter... filters) {
    	return getNearest(locations, target, getLocatedDistance3D, filters);
    }
	
    /**
     * Returns the nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearest(Collection<T> locations, ILocated target) {
    	return getNearest(locations, target, getLocatedDistance3D);
    }
    
    /**
     * Returns the nearest object to 'target' that is not further than 'maxDistance'.
     * <p><p>
     * Using {@link RangeDistanceFilter} (minDistance = 0, maxDistance is provided).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param maxDistance
     * @return nearest object from collection of objects that is not further than 'maxDistance'.
     */
    public static <T extends ILocated> T getNearest(Collection<T> locations, ILocated target, double maxDistance) {
    	return getNearest(locations, target, new RangeDistanceFilter<T>(0, maxDistance));
    }
    
    /**
     * Returns the nearest object to 'target' that is accepted by filter.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filter if null behave as if ALL locations are accepted
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearestFiltered(Collection<T> locations, ILocated target, IFilter filter) {
    	if (filter == null) {
    		return getNearest(locations, target);
    	}
    	return getNearest(locations, target, new FilterAdapter<T>(filter));
    }
    
    /**
     * Returns the nearest object to 'target' that is visible (using {@link VisibleFilter}).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest visible object from collection of objects
     */
    public static <T extends IViewable> T getNearestVisible(Collection<T> locations, ILocated target) {
    	return getNearest(locations, target, getLocatedDistance3D, visibleFilter);
    }
    
	// --------------
	// =====================
	// getDistanceSorted()
	// =====================
	// --------------

    private static class DistancesComparator implements Comparator<Tuple2<Object, Double>> {

		@Override
		public int compare(Tuple2<Object, Double> o1, Tuple2<Object, Double> o2) {
			double result = o1.getSecond() - o2.getSecond();
			if (result < 0) return -1;
			if (result > 0) return 1;
			return 0;
		}
    	
    }
    
    private static final DistancesComparator distancesComparator = new DistancesComparator();

    /**
     * Returns "locations" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * Distance is provided by {@link IGetDistance#getDistance(Object, Location)}.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param getDistance
     * @return
     */
    public static <T> List<T> getDistanceSorted(Collection<T> locations, ILocated target, IGetDistance getDistance) {
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (target.getLocation() == null) return null;
    	if (getDistance == null) return null;
    	
    	List<Tuple2<T, Double>> distances = new ArrayList<Tuple2<T, Double>>(locations.size());
    	
    	for (T location : locations) {
    		double distance = getDistance.getDistance(location, target);
    		distances.add(new Tuple2<T, Double>(location, distance));
    	}
    	
    	Collections.sort((List)distances, (Comparator)distancesComparator);
    	
    	List<T> result = new ArrayList<T>(distances.size());
    	
    	for (Tuple2<T, Double> location : distances) {
    		result.add(location.getFirst());
    	}
    	
    	return result;
    }
    
    /**
     * Returns "locations" accepted by all "filters" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * Distance is provided by {@link IGetDistance#getDistance(Object, Location)}.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param getDistance
     * @return
     */
    public static <T> List<T> getDistanceSorted(Collection<T> locations, ILocated target, IGetDistance getDistance, IDistanceFilter... filters) {
    	if (filters == null || filters.length == 0 || (filters.length == 1 && filters[0] instanceof AcceptAllDistanceFilter)) {
    		return getDistanceSorted(locations, target, getDistance);
    	}
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (getDistance == null) return null;
    	Location targetLoc = target.getLocation();
    	if (targetLoc == null) return null;
    	
    	List<Tuple2<T, Double>> distances = new ArrayList<Tuple2<T, Double>>(locations.size());
    	
    	for (T location : locations) {
    		boolean accepted = true;
    		double distance = getDistance.getDistance(location, targetLoc);
    		for (IDistanceFilter filter : filters) {
    			if (!filter.isAccepted(location, targetLoc, distance)) {
    				accepted = false;
    				break;
    			}
    		}
    		if (!accepted) continue;
    		
    		distances.add(new Tuple2<T, Double>(location, distance));
    	}
    	
    	Collections.sort((List)distances, (Comparator)distancesComparator);
    	
    	List<T> result = new ArrayList<T>(distances.size());
    	
    	for (Tuple2<T, Double> location : distances) {
    		result.add(location.getFirst());
    	}
    	
    	return result;
    }
    
    /**
     * Returns "locations" accepted by all "filters" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filters
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> List<T> getDistanceSorted(Collection<T> locations, ILocated target, IDistanceFilter... filters) {
    	return getDistanceSorted(locations, target, getLocatedDistance3D, filters);
    }
	
    /**
     * Returns "locations" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> List<T> getDistanceSorted(Collection<T> locations, ILocated target) {
    	return getDistanceSorted(locations, target, getLocatedDistance3D);
    }
    
    /**
     * Returns "locations" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * Using {@link RangeDistanceFilter} (minDistance = 0, maxDistance is provided).
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param maxDistance
     * @return nearest object from collection of objects that is not further than 'maxDistance'.
     */
    public static <T extends ILocated> List<T> getDistanceSorted(Collection<T> locations, ILocated target, double maxDistance) {
    	return getDistanceSorted(locations, target, new RangeDistanceFilter<T>(0, maxDistance));
    }
    
    /**
     * Returns "locations" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filter if null behave as if ALL locations are accepted
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> List<T> getDistanceSortedFiltered(Collection<T> locations, ILocated target, IFilter filter) {
    	if (filter == null) {
    		return getDistanceSorted(locations, target);
    	}
    	return getDistanceSorted(locations, target, new FilterAdapter<T>(filter));
    }
    
    /**
     * Returns visible "locations" sorted according to the distance to "target". Sorted from the nearest to the farthest.
     * <p><p>
     * WARNING: 2*O(n) + O(n*log n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest visible object from collection of objects
     */
    public static <T extends IViewable> List<T> getDistanceSortedVisible(Collection<T> locations, ILocated target) {
    	return (List<T>) getDistanceSorted(locations, target, (IGetDistance)getLocatedDistance3D, (IDistanceFilter)visibleFilter);
    }
    
    // --------------
	// =====================
	// getSecondNearest()
	// =====================
	// --------------
    
    /**
     * Returns the second nearest object to 'target'.
     * <p><p>
     * Distance is provided by {@link IGetDistance#getDistance(Object, ILocated)}.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T> T getSecondNearest(Collection<T> locations, ILocated target, IGetDistance getDistance) {
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (target.getLocation() == null) return null;
    	
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {
        	double distance = getDistance.getDistance(l, target);
            
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }
    
    /**
     * Returns the second nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T> T getSecondNearest(Collection<T> locations, ILocated target, IGetDistance getDistance, IDistanceFilter... filters) {
    	if (filters == null || filters.length == 0 || (filters.length == 1 && filters[0] instanceof AcceptAllDistanceFilter)) {
    		return getSecondNearest(locations, target, getDistance);
    	}
    	if (locations == null) return null;
    	if (target == null) return null;
    	if (target.getLocation() == null) return null;
    	
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {
        	double distance = getDistance.getDistance(l, target);
            
            boolean accepted = true;
            for (IDistanceFilter<T> filter : filters) {
            	if (!filter.isAccepted(l, target, distance)) {
            		accepted = false;
            		break;
            	}
            }
            if (!accepted) continue;
            
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }
    
    /**
     * Returns the second nearest object to 'target' that is accepted by all 'filters'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filters
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest(Collection<T> locations, ILocated target, IDistanceFilter... filters) {
    	return getSecondNearest(locations, target, getLocatedDistance3D, filters);
    }
	
    /**
     * Returns the second nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return second nearest object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest(Collection<T> locations, ILocated target) {
    	return getSecondNearest(locations, target, getLocatedDistance2D);
    }
    
    /**
     * Returns the second nearest object to 'target' that is not further than 'maxDistance'.
     * <p><p>
     * Using {@link RangeDistanceFilter} (minDistance = 0, maxDistance is provided).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param maxDistance
     * @return second nearest object from collection of objects that is not further than 'maxDistance'.
     */
    public static <T extends ILocated> T getSecondNearest(Collection<T> locations, ILocated target, double maxDistance) {
    	return getSecondNearest(locations, target, new RangeDistanceFilter<T>(0, maxDistance));
    }
    
    /**
     * Returns the second nearest object to 'target' that is accepted by filter.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filter if null behave as if ALL locations are accepted
     * @return second nearest object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearestFiltered(Collection<T> locations, ILocated target, IFilter filter) {
    	if (filter == null) {
    		return getSecondNearest(locations, target);
    	}
    	return getSecondNearest(locations, target, new FilterAdapter(filter));
    }
    
    /**
     * Returns the second nearest object to 'target' that is visible (using {@link VisibleFilter}).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return second nearest visible object from collection of objects
     */
    public static <T extends IViewable> T getSecondNearestVisible(Collection<T> locations, ILocated target) {
    	return getSecondNearest(locations, target, getLocatedDistance3D, visibleFilter);
    }
    
    // --------------
	// =====================
	// getNearest2D()
	// =====================
	// --------------
    
    /**
     * Returns the nearest (in 2D) object to 'target' that is accepted by all 'filters'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filters
     * @return nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getNearest2D(Collection<T> locations, ILocated target, IDistanceFilter... filters) {
    	return (T) getNearest((Collection)locations, target, (IGetDistance)getLocatedDistance2D, filters);
    }
	
    /**
     * Returns the nearest (in 2D) object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getNearest2D(Collection<T> locations, ILocated target) {
    	return (T) getNearest((Collection<ILocated>)locations, target, getLocatedDistance2D);
    }
    
    /**
     * Returns the nearest (in 2D) object to 'target' that is not further than 'maxDistance'.
     * <p><p>
     * Using {@link RangeDistanceFilter} (minDistance = 0, maxDistance is provided).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param maxDistance
     * @return nearest (in 2D) object from collection of objects that is not further than 'maxDistance'.
     */
    public static <T extends ILocated> T getNearest2D(Collection<T> locations, ILocated target, double maxDistance) {
    	return getNearest(locations, target, getLocatedDistance2D, new RangeDistanceFilter<T>(0, maxDistance));
    }
    
    /**
     * Returns the nearest (in 2D) object to 'target' that is accepted by filter.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filter if null behave as if ALL locations are accepted
     * @return nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getNearest2DFiltered(Collection<T> locations, ILocated target, IFilter<T> filter) {
    	if (filter == null) {
    		return getNearest2D(locations, target);
    	}
    	return getNearest2D(locations, target, new FilterAdapter<T>(filter));
    }
    
    /**
     * Returns the nearest object to 'target' that is visible (using {@link VisibleFilter}).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest (in 2D) visible object from collection of objects
     */
    public static <T extends IViewable> T getNearest2DVisible(Collection<T> locations, ILocated target) {
    	return getNearest(locations, target, getLocatedDistance2D, visibleFilter);
    }
    
    // --------------
	// =====================
	// getSecondNearest2D()
	// =====================
	// --------------

    /**
     * Returns the second nearest (in 2D) object to 'target' that is accepted by all 'filters'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filters
     * @return second nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest2D(Collection<T> locations, ILocated target, IDistanceFilter... filters) {
    	return (T) getSecondNearest((Collection)locations, target, (IGetDistance)getLocatedDistance2D, filters);
    }
	
    /**
     * Returns the second nearest (in 2D) object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return second nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest2D(Collection<T> locations, ILocated target) {
    	return (T) getSecondNearest((Collection<ILocated>)locations, target, getLocatedDistance2D);
    }
    
    /**
     * Returns the second nearest (in 2D) object to 'target' that is not further than 'maxDistance'.
     * <p><p>
     * Using {@link RangeDistanceFilter} (minDistance = 0, maxDistance is provided).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param maxDistance
     * @return second nearest (in 2D) object from collection of objects that is not further than 'maxDistance'.
     */
    public static <T extends ILocated> T getSecondNearest2D(Collection<T> locations, ILocated target, double maxDistance) {
    	return getSecondNearest(locations, target, getLocatedDistance2D, new RangeDistanceFilter<T>(0, maxDistance));
    }
    
    /**
     * Returns the second nearest (in 2D) object to 'target' that is accepted by filter.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @param filter if null behave as if ALL locations are accepted
     * @return second nearest (in 2D) object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest2DFiltered(Collection<T> locations, ILocated target, IFilter<T> filter) {
    	if (filter == null) {
    		return getSecondNearest2D(locations, target);
    	}
    	return getSecondNearest2D(locations, target, new FilterAdapter<T>(filter));
    }
    
    /**
     * Returns the second nearest object to 'target' that is visible (using {@link VisibleFilter}).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return second nearest (in 2D) visible object from collection of objects
     */
    public static <T extends IViewable> T getSecondNearest2DVisible(Collection<T> locations, ILocated target) {
    	return getSecondNearest(locations, target, getLocatedDistance2D, visibleFilter);
    }

}
