/**
 * Emohawk Bot, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and Emohawk. 
 * 
 * Copyright (C) 2012 Emohawk Bot authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */

package nl.tudelft.goal.ut2004.environment;

import java.net.URI;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Map;

import nl.tudelft.goal.unreal.environment.AbstractUnrealEnvironment;
import nl.tudelft.goal.unreal.messages.BotParameters;
import nl.tudelft.goal.unreal.translators.LocationTranslator;
import nl.tudelft.goal.unreal.translators.RotationTranslator;
import nl.tudelft.goal.unreal.translators.TeamTranslator;
import nl.tudelft.goal.unreal.translators.UnrealIdTranslator;
import nl.tudelft.goal.unreal.translators.VelocityTranslator;
import nl.tudelft.goal.ut2004.agent.UT2004BotBehavior;
import nl.tudelft.goal.ut2004.translators.CategoryTranslator;
import nl.tudelft.goal.ut2004.translators.ComboTranslator;
import nl.tudelft.goal.ut2004.translators.FireModeTranslator;
import nl.tudelft.goal.ut2004.translators.FlagStateTranslator;
import nl.tudelft.goal.ut2004.translators.GameTypeTranslator;
import nl.tudelft.goal.ut2004.translators.ItemTypeTranslator;
import nl.tudelft.goal.ut2004.translators.NavigationStateTranslator;
import nl.tudelft.goal.ut2004.translators.NoneTranslator;
import nl.tudelft.goal.ut2004.translators.PerceptTranslator;
import nl.tudelft.goal.ut2004.translators.SelectorListTranslator;
import nl.tudelft.goal.ut2004.translators.SelectorTranslator;
import nl.tudelft.goal.ut2004.translators.UnrealIdOrLocationTranslator;
import nl.tudelft.goal.ut2004.translators.WeaponPrefListTranslator;
import nl.tudelft.goal.ut2004.translators.WeaponPrefTranslator;
import nl.tudelft.goal.ut2004.visualizer.connection.AddBotCommand;
import nl.tudelft.goal.ut2004.visualizer.connection.EnvironmentServiceListener;
import nl.tudelft.goal.ut2004.visualizer.connection.EnvironmentServiceMediator;
import nl.tudelft.goal.ut2004.visualizer.connection.client.RemoteVisualizer;
import nl.tudelft.goal.ut2004.visualizer.connection.client.VisualizerServiceDefinition;
import nl.tudelft.pogamut.base.server.ReconnectingServerDefinition;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotController;
import cz.cuni.amis.utils.flag.FlagListener;
import eis.eis2java.handlers.ActionHandler;
import eis.eis2java.handlers.AllPerceptPerceptHandler;
import eis.eis2java.handlers.DefaultActionHandler;
import eis.eis2java.handlers.PerceptHandler;
import eis.eis2java.translation.Translator;
import eis.eis2java.util.AllPerceptsProvider;
import eis.exceptions.EntityException;
import eis.exceptions.ManagementException;
import eis.iilang.Identifier;
import eis.iilang.Parameter;
import eis.iilang.ParameterList;

public class UT2004Environment extends AbstractUnrealEnvironment {

	/**
	 * Generated serialVersionUID.
	 */
	private static final long serialVersionUID = 8240549393243585632L;

	// Connection to the visualizer. Can be used to add bots to the environment.
	private ReconnectingServerDefinition<RemoteVisualizer> visualizerConnection;

	@Override
	protected void registerTranslators() {

		Translator translator = Translator.getInstance();

		/*
		 * Translators provided by the BaseUnrealEnvironment.
		 * 
		 * Please list these in lexical order.
		 */

		LocationTranslator locationTranslator = new LocationTranslator();
		translator.registerJava2ParameterTranslator(locationTranslator);
		translator.registerParameter2JavaTranslator(locationTranslator);

		/*
		 * To translate from Parameter2Java we are given an UnrealId. However we
		 * can not access the agents memory during translation. To work around
		 * this we store everything we have send to any agent. Hence the same
		 * object has to be used for both directions.
		 */
		RotationTranslator rotationTranslator = new RotationTranslator();
		translator.registerJava2ParameterTranslator(rotationTranslator);
		translator.registerParameter2JavaTranslator(rotationTranslator);

		TeamTranslator teamTranslator = new TeamTranslator();
		translator.registerJava2ParameterTranslator(teamTranslator);
		translator.registerParameter2JavaTranslator(teamTranslator);

		UnrealIdTranslator unrealIdTranslator = new UnrealIdTranslator();
		translator.registerJava2ParameterTranslator(unrealIdTranslator);
		translator.registerParameter2JavaTranslator(unrealIdTranslator);

		VelocityTranslator velocityTranslator = new VelocityTranslator();
		translator.registerJava2ParameterTranslator(velocityTranslator);
		translator.registerParameter2JavaTranslator(velocityTranslator);
		/*
		 * Translators provided by the UT2004 environment.
		 * 
		 * Please list these in lexical order.
		 */

		CategoryTranslator itemTypeTranslator = new CategoryTranslator();
		translator.registerJava2ParameterTranslator(itemTypeTranslator);

		ComboTranslator comboTranslator = new ComboTranslator();
		translator.registerParameter2JavaTranslator(comboTranslator);

		FireModeTranslator fireModeTranslator = new FireModeTranslator();
		translator.registerJava2ParameterTranslator(fireModeTranslator);
		translator.registerParameter2JavaTranslator(fireModeTranslator);

		FlagStateTranslator flagStateTranslator = new FlagStateTranslator();
		translator.registerJava2ParameterTranslator(flagStateTranslator);

		GameTypeTranslator gameTypeTranslator = new GameTypeTranslator();
		translator.registerJava2ParameterTranslator(gameTypeTranslator);

		ItemTypeTranslator categoryTranslator = new ItemTypeTranslator();
		translator.registerJava2ParameterTranslator(categoryTranslator);
		translator.registerParameter2JavaTranslator(categoryTranslator);

		NavigationStateTranslator navigationStateTranslator = new NavigationStateTranslator();
		translator.registerJava2ParameterTranslator(navigationStateTranslator);

		NoneTranslator noneTranslator = new NoneTranslator();
		translator.registerJava2ParameterTranslator(noneTranslator);

		PerceptTranslator perceptTranslator = new PerceptTranslator();
		translator.registerJava2ParameterTranslator(perceptTranslator);

		SelectorListTranslator selectorListTranslator = new SelectorListTranslator();
		translator.registerParameter2JavaTranslator(selectorListTranslator);

		SelectorTranslator selectorTranslator = new SelectorTranslator();
		translator.registerParameter2JavaTranslator(selectorTranslator);

		UnrealIdOrLocationTranslator unrealIdOrLocationTranslator = new UnrealIdOrLocationTranslator();
		translator.registerParameter2JavaTranslator(unrealIdOrLocationTranslator);

		WeaponPrefListTranslator weaponPrefListTranslator = new WeaponPrefListTranslator();
		translator.registerParameter2JavaTranslator(weaponPrefListTranslator);

		WeaponPrefTranslator weaponPrefTranslator = new WeaponPrefTranslator();
		translator.registerParameter2JavaTranslator(weaponPrefTranslator);
	}

	public static void main(String[] args) throws ManagementException {
		HashMap<String, Parameter> map = new HashMap<String, Parameter>();
		map.put("botNames", new ParameterList(new Identifier("Test")));
		new UT2004Environment().init(map);

	}

	@Override
	protected Class<UT2004BotBehavior> getControlerClass() {
		return UT2004BotBehavior.class;
	}

	@Override
	protected PerceptHandler createPerceptHandler(UT2004BotController controller) throws EntityException {
		if (!(controller instanceof AllPerceptsProvider))
			throw new EntityException("Expected a class that implements " + AllPerceptsProvider.class.getSimpleName());
		return new AllPerceptPerceptHandler((AllPerceptsProvider) controller);
	}

	@Override
	protected ActionHandler createActionHandler(UT2004BotController controller) throws EntityException {
		return new DefaultActionHandler(controller);
	}

	@Override
	protected synchronized void initializeEnvironment(Map<String, Parameter> parameters) throws ManagementException {
		super.initializeEnvironment(parameters);

		// Set up (future) connection to visualizer. Connecting is done later.
		try {
			visualizerConnection = new ReconnectingServerDefinition<RemoteVisualizer>(new VisualizerServiceDefinition());
			visualizerConnection.getServerFlag().addListener(new VisualizerServiceListener());
		} catch (RemoteException e) {
			log.severe("Could not start connection to Visualizer: " + e);
		}

	}

	/**
	 * Helper class that handles the connection to the visualizer. Registers a
	 * mediator with the visualizer and listens to the actions it requests.
	 * 
	 * @author M.P. Korstanje
	 * 
	 */
	private class VisualizerServiceListener implements EnvironmentServiceListener, FlagListener<RemoteVisualizer> {

		private final EnvironmentServiceMediator mediator;

		public VisualizerServiceListener() throws RemoteException {
			mediator = new EnvironmentServiceMediator(getComponentId());
			mediator.setListener(this);
		}

		@Override
		public void flagChanged(RemoteVisualizer visualizer) {
			if (visualizer != null) {
				visualizer.setEnvironment(mediator);
			}
		}

		/**
		 * 
		 * Listens to actions executed by the remote visualizer.
		 * 
		 * @throws ManagementException
		 * 
		 */
		@Override
		public void addBot(AddBotCommand command) throws ManagementException {

			BotParameters parameters = new BotParameters(environmentLogger);

			// Fill out the bot parameter
			if (command.getBotName() != null) {
				parameters.setAgentId(command.getBotName());
			}
			if (command.getLogLevel() != null) {
				parameters.setLogLevel(command.getLogLevel());
			}
			if (command.getShouldLeadTarget() != null) {
				parameters.setShouldLeadTarget(command.getShouldLeadTarget());
			}
			if (command.getSkill() != null) {
				parameters.setSkill(command.getSkill());
			}
			if (command.getSkin() != null) {
				parameters.setSkin(command.getSkin());
			}
			if (command.getTeam() != null) {
				parameters.setTeam(command.getTeam());
			}

			if (command.getRotation() != null) {
				parameters.setInitialRotation(command.getRotation());
			}

			if (command.getLocation() != null) {
				parameters.setInitialLocation(command.getLocation());
			}

			// Take connection and other settings from init.
			parameters.assignDefaults(botParameters);

			startAgents(parameters);

		}

	}

	@Override
	protected synchronized void connectEnvironment() throws ManagementException {
		super.connectEnvironment();

		// Connect to visualizer
		URI visualizerUri = environmentParameters.getVisualizerServer();
		if (visualizerUri != null) {
			log.info("Connecting to visualizer server at " + visualizerUri + " .");

			visualizerConnection.setUri(visualizerUri);
		} else {
			log.info("No address for the visualizer server was provided. The environment will not try to connect to the visualizer.");
		}
	}

	@Override
	protected synchronized void killEnvironment() {
		super.killEnvironment();
		// Close the connection to the visualizer.
		visualizerConnection.stopServer();

	}

}
