/**
 * BaseUnrealEnvironment, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and the UT2004 engine. 
 * 
 * Copyright (C) 2012 BaseUnrealEnvironment authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.tudelft.goal.unreal.messages;

import java.net.URI;
import java.util.Map;
import java.util.logging.Level;

import nl.tudelft.goal.EIS2Java.exception.TranslationException;
import nl.tudelft.goal.EIS2Java.translation.Translator;
import nl.tudelft.goal.unreal.environment.UnrealEnvironmentException;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.component.IComponent;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.Rotation;
import cz.cuni.amis.utils.token.IToken;
import cz.cuni.amis.utils.token.Tokens;
import eis.iilang.Parameter;

/**
 * Holds parameters specific for the bot.
 * <ul>
 * Parameters stored are:
 * <li>{@link Key#BOTNAME}</li>
 * <li>{@link Key#LEADTARGET}</li>
 * <li>{@link Key#LOGLEVEL}</li>
 * <li>{@link Key#BOTSERVER}</li>
 * <li>{@link Key#SKILL}</li>
 * <li>{@link Key#SKIN}</li>
 * <li>{@link Key#TEAM}</li>
 * <li>{@link Key#STARTLOCATION}</li>
 * <li>{@link Key#STARTROTATION}</li>
 * </ul>
 * 
 * Also provides functionality to assign defaults to parameters that have not
 * been assigned.
 * 
 * @author M.P. Korstanje
 * 
 */
public class BotParameters extends Parameters implements IComponent {

	// Bot parameters
	private Level logLevel;
	private Boolean shouldLeadTarget;
	private Integer skill;
	private Skin skin;
	private Location location;
	private Rotation rotation;

	public BotParameters(IAgentLogger logger) {
		super(logger);
	}

	public BotParameters(Map<String, Parameter> parameters, IAgentLogger logger) throws UnrealEnvironmentException {
		super(parameters, logger);
	}

	public BotParameters(Parameters defaults, IAgentLogger logger) {
		super(logger);
		assignDefaults(defaults);

		// Reset name to ensure name is unique.
		if (getAgentId() != null) {
			String name = getAgentId().getName().getFlag();
			setAgentId(name);
		}
	}

	@Override
	public void assignDefaults(IAgentParameters defaults) {
		super.assignDefaults(defaults);

		if (defaults instanceof BotParameters) {
			BotParameters parameters = (BotParameters) defaults;

			if (logLevel == null)
				logLevel = parameters.getLogLevel();
			if (shouldLeadTarget == null)
				shouldLeadTarget = parameters.shouldLeadTarget();
			if (skill == null)
				skill = parameters.getSkill();
			if (skin == null)
				skin = parameters.getSkin();
			if(location == null)
				location = parameters.getStartLocation();
			if(rotation == null)
				rotation = parameters.getStartRotation();
		}
	}

	@Override
	public IToken getComponentId() {
		return Tokens.get(getClass().getSimpleName());
	}

	public Level getLogLevel() {
		return logLevel;
	}

	public int getSkill() {
		return skill;
	}

	public Skin getSkin() {

		return skin;
	}

	public BotParameters setLogLevel(Level level) {
		assert level != null;
		log.info(String.format("Set %s to %s.", Key.LOGLEVEL, level));
		this.logLevel = level;
		return this;
	}

	public BotParameters setShouldLeadTarget(boolean shouldLeadTarget) {
		log.info(String.format("Set %s to %s.", Key.LEADTARGET, shouldLeadTarget));
		this.shouldLeadTarget = shouldLeadTarget;
		return this;
	}

	/**
	 * Sets the bots skill between 0..7. The bots skill only influences the bots
	 * accuracy. Values outside the valid range are clamped.
	 * 
	 * @param skill
	 *            an integer in the range 0..7
	 */
	public BotParameters setSkill(int skill) {
		// Clamp between 0 and 7.
		this.skill = Math.min(7, Math.max(skill, 0));
		log.info(String.format("Set %s to %s.", Key.SKILL, this.skill));
		return this;
	}

	public BotParameters setSkin(Skin skin) {
		log.info(String.format("Set %s to %s.", Key.SKIN, skin.toString()));
		this.skin = skin;
		return this;
	}

	/**
	 * Sets the bots preferred team. Typically teams are either 0 or 1.
	 * 
	 * @param team
	 */
	public BotParameters setTeam(Team team) {
		log.info(String.format("Set %s to %s.", Key.TEAM, team));
		super.setTeam(team.id());
		return this;
	}

	public BotParameters setStartLocation(Location location) {
		log.info(String.format("Set %s to %s.", Key.STARTLOCATION, location.toString()));
		this.location = location;
		return this;
	}

	public BotParameters setStartRotation(Rotation rotation) {
		log.info(String.format("Set %s to %s.", Key.STARTROTATION, rotation.toString()));
		this.rotation = rotation;
		return this;
	}

	public Boolean shouldLeadTarget() {
		return shouldLeadTarget;
	}

	public Location getStartLocation() {
		return location;
	}

	@Override
	public String toString() {
		return "BotParameters [logLevel=" + logLevel + ", shouldLeadTarget=" + shouldLeadTarget + ", skill=" + skill
				+ ", skin=" + skin + ", getTeam()=" + getTeam() + ", getWorldAddress()=" + getWorldAddress()
				+ ", getAgentId()=" + getAgentId() + "]";
	}

	@Override
	protected void setKey(Key key, Parameter value) throws TranslationException {

		switch (key) {
		case LEADTARGET:
			setShouldLeadTarget(value);
			break;
		case LOGLEVEL:
			setLogLevel(value);
			break;
		case SKILL:
			setSkill(value);
			break;
		case SKIN:
			setSkin(value);
			break;
		case TEAM:
			setTeam(value);
			break;
		case BOTSERVER:
			setWorldAddress(value);
			break;
		case STARTLOCATION:
			setStartLocation(value);
			break;
		case STARTROTATION:
			setStartRotation(value);
			break;
		default:
			// Not one of our keys.
			break;
		}

	}

	private void setStartRotation(Parameter value) throws TranslationException {
		setStartRotation(Translator.getInstance().translate2Java(value, Rotation.class));

	}

	private void setStartLocation(Parameter value) throws TranslationException {
		setStartLocation(Translator.getInstance().translate2Java(value, Location.class));
	}

	private void setLogLevel(Parameter value) throws TranslationException {
		setLogLevel(Translator.getInstance().translate2Java(value, Level.class));
	}

	private void setShouldLeadTarget(Parameter value) throws TranslationException {
		setShouldLeadTarget(Translator.getInstance().translate2Java(value, Boolean.class));
	}

	private void setSkill(Parameter value) throws TranslationException {
		setSkill(Translator.getInstance().translate2Java(value, Integer.class));
	}

	private void setSkin(Parameter value) throws TranslationException {
		setSkin(Translator.getInstance().translate2Java(value, Skin.class));
	}

	private void setTeam(Parameter value) throws TranslationException {
		setTeam(Translator.getInstance().translate2Java(value, Team.class));
	}

	private void setWorldAddress(Parameter value) throws TranslationException {
		URI uri = Translator.getInstance().translate2Java(value, URI.class);
		setWorldAddress(new SocketConnectionAddress(uri));
	}

	public static BotParameters getDefaults(IAgentLogger logger) {

		BotParameters parameters = new BotParameters(logger);

		parameters.logLevel = Level.WARNING;
		parameters.shouldLeadTarget = true;
		parameters.skill = 3;
		parameters.skin = Skin.BotB;
		parameters.location = null;
		parameters.rotation = null;

		parameters.setAgentIdSilent(DEFAULT_NAME);
		parameters.setWorldAddressSilent(LOCAL_HOST, BOT_SERVER_PORT);

		return parameters;
	}

	private void setWorldAddressSilent(String host, int port) {
		super.setWorldAddress(new SocketConnectionAddress(host, port));
	}

	private void setAgentIdSilent(String name) {
		super.setAgentId(new AgentId(name));
	}

	public Rotation getStartRotation() {
		return rotation;
	}

}
