/**
 * BaseUnrealEnvironment, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and the UT2004 engine. 
 * 
 * Copyright (C) 2012 BaseUnrealEnvironment authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.tudelft.goal.unreal.environment;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import nl.tudelft.goal.EIS2Java.environment.AbstractEnvironment;
import nl.tudelft.goal.EIS2Java.handlers.ActionHandler;
import nl.tudelft.goal.EIS2Java.handlers.AllPerceptPerceptHandler;
import nl.tudelft.goal.EIS2Java.handlers.DefaultActionHandler;
import nl.tudelft.goal.EIS2Java.handlers.PerceptHandler;
import nl.tudelft.goal.EIS2Java.translation.Translator;
import nl.tudelft.goal.unreal.messages.BotParameters;
import nl.tudelft.goal.unreal.messages.EnvironmentParameters;
import nl.tudelft.goal.unreal.messages.Parameters;
import nl.tudelft.goal.unreal.translators.AgentIdTranslator;
import nl.tudelft.goal.unreal.translators.LevelTranslator;
import nl.tudelft.goal.unreal.translators.LocationTranslator;
import nl.tudelft.goal.unreal.translators.RotationTranslator;
import nl.tudelft.goal.unreal.translators.SkinTranslator;
import nl.tudelft.goal.unreal.translators.StringListTranslator;
import nl.tudelft.goal.unreal.translators.TeamTranslator;
import nl.tudelft.goal.unreal.translators.URITranslator;
import nl.tudelft.pogamut.base.server.ReconnectingServerDefinition;
import nl.tudelft.pogamut.ut2004.server.UTServerDefinition;
import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.exceptions.AgentException;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.state.level0.IAgentState;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateDown;
import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.component.IComponent;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.logging.AgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotController;
import cz.cuni.amis.pogamut.ut2004.bot.params.UT2004BotParameters;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Pause;
import cz.cuni.amis.pogamut.ut2004.server.IUT2004Server;
import cz.cuni.amis.pogamut.ut2004.utils.UT2004BotRunner;
import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.flag.FlagListener;
import eis.exceptions.EntityException;
import eis.exceptions.ManagementException;
import eis.exceptions.RelationException;
import eis.iilang.Action;
import eis.iilang.EnvironmentState;
import eis.iilang.Parameter;

@SuppressWarnings("rawtypes")
public abstract class AbstractUnrealEnvironment extends SimpleTransitioningEnvironment implements IComponent {

	/**
	 * Generated serialVersionUID.
	 */
	private static final long serialVersionUID = 6786623950045095814L;
	private final IAgentId id;

	// Manager of logs.
	private final IAgentLogger environmentLogger;
	// Actual logger
	protected final LogCategory log;

	// Agent state listeners
	private final Map<String, AgentDownListener> agentDownListeners;

	// Parameters provided on init send to bots
	protected BotParameters botParameters;
	protected EnvironmentParameters environmentParameters;

	// Connection to the ut Server. Can be used to pause/resume the game.
	private ReconnectingServerDefinition<IUT2004Server> utServerConnection;

	/**
	 * Constructs the Unreal Environment. The environment won't be ready until
	 * until is has has been initialized.
	 * 
	 */
	public AbstractUnrealEnvironment() {
		id = new AgentId(getName());
		agentDownListeners = new HashMap<String, AgentDownListener>();
		environmentLogger = new AgentLogger(id);
		environmentLogger.addDefaultConsoleHandler();
		log = environmentLogger.getCategory(this);
		log.info("Environment has been created.");
		log.addConsoleHandler();

		// Register own translators.
		Translator translator = Translator.getInstance();
		translator.registerParameter2JavaTranslator(new AgentIdTranslator());
		translator.registerParameter2JavaTranslator(new LevelTranslator());
		translator.registerParameter2JavaTranslator(new SkinTranslator());
		translator.registerParameter2JavaTranslator(new StringListTranslator());
		translator.registerParameter2JavaTranslator(new TeamTranslator());
		translator.registerParameter2JavaTranslator(new URITranslator());
		translator.registerParameter2JavaTranslator(new LocationTranslator());
		translator.registerParameter2JavaTranslator(new RotationTranslator());

		// Register translators required by the bot controller.
		registerTranslators();
	}

	protected abstract void registerTranslators();

	/**
	 * Provides an unique identifier for this component for use by loggers.
	 */

	@Override
	public IAgentId getComponentId() {
		return id;
	}

	/**
	 * Returns a string representation of the environment based on ID of the
	 * environment.
	 * 
	 * @return a representation of the environment.
	 */
	@Override
	public String toString() {
		return simplefyID(getComponentId());
	}

	public String getName() {
		return "UnrealGoal Environment for EIS" + requiredVersion();
	}

	protected synchronized void initializeEnvironment(Map<String, Parameter> parameters) throws ManagementException {
		assert botParameters == null;
		assert environmentParameters == null;

		try {
			botParameters = new BotParameters(parameters, environmentLogger);
			environmentParameters = new EnvironmentParameters(parameters, environmentLogger);
		} catch (UnrealEnvironmentException e) {
			// clean up
			botParameters = null;
			environmentParameters = null;
			// and throw
			log.severe("Invalid parameters: " + e);
			throw new ManagementException("Invalid parameters.", e);
		}

		// Set defaults for environment and bot
		environmentParameters.assignDefaults(EnvironmentParameters.getDefaults(environmentLogger));
		botParameters.assignDefaults(BotParameters.getDefaults(environmentLogger));

		// Set log level
		// TODO: using same level as bot for now, might be different.
		log.setLevel(environmentParameters.getLogLevel());

		// Set up (future) connection the UT server. Connecting is done later.
		utServerConnection = new ReconnectingServerDefinition<IUT2004Server>(new UTServerDefinition());
	}

	protected synchronized void connectEnvironment() throws ManagementException {
		assert botParameters != null;
		assert environmentParameters != null;

		// 1. Connect to UT server
		URI utServerURI = environmentParameters.getUTServer();
		if (utServerURI != null) {
			log.info("Connecting to the control server at " + utServerURI + " .");
			utServerConnection.setUri(utServerURI);
		} else {
			log.info("No address for the ut control server was provided. The environment will not try to connect to the control server.");
		}

		// 2. Set names on bot parameters.
		List<BotParameters> agentParameters = new ArrayList<BotParameters>();
		for (String name : environmentParameters.getBotNames()) {
			BotParameters parameter = new BotParameters(botParameters, environmentLogger);
			parameter.setAgentId(name);
			agentParameters.add(parameter);
		}

		// 3. Start bots.
		BotParameters[] agentParameterArray = new BotParameters[agentParameters.size()];
		agentParameterArray = agentParameters.toArray(agentParameterArray);
		startAgents(agentParameterArray);

	}

	protected abstract Class<? extends UT2004BotController> getControlerClass();

	private synchronized void startAgents(BotParameters... parameters) throws ManagementException {

		// 1. Don't add bots if the environment has been killed.
		if (getState() == EnvironmentState.KILLED) {
			return;
		}

		// 2. Unpause the game. Can't add agents to paused game.
		IUT2004Server server = utServerConnection.getServerFlag().getFlag();
		if (server != null) {
			Pause resume = new Pause(false, false);
			server.getAct().act(resume);

			log.info("The server has been unpaused. Required to allow adding bots.");
		} else {
			log.warning("We are not connected to ut server and could not unpause the environment.");
		}

		// 3. Setting defaults again, this function may be called from else
		// where.
		for (BotParameters botParameters : parameters) {
			botParameters.assignDefaults(BotParameters.getDefaults(environmentLogger));
		}

		// 4. TODO: Using defaults here because pogamut can't load properties
		// when used inside a Jar.

		UT2004BotRunner<UT2004Bot<IVisionWorldView, IAct, UT2004BotController>, UT2004BotParameters> runner = new UT2004BotRunner<UT2004Bot<IVisionWorldView, IAct, UT2004BotController>, UT2004BotParameters>(
				getControlerClass(), Parameters.DEFAULT_NAME, Parameters.LOCAL_HOST, Parameters.BOT_SERVER_PORT);

		// 5. Launch bots using the parameters
		runner.setLog(log);
		List<UT2004Bot<IVisionWorldView, IAct, UT2004BotController>> agents;
		try {
			agents = runner.startAgents(parameters);
		} catch (PogamutException e) {
			throw new ManagementException(
			// Adding exception as String. While Pogamut exceptions
			// themselves are properly serializable, their contents may not be.
					"Pogmut was unable to start all agents. Cause: " + e.toString());
		}

		// 6. Notify EIS of new entity.
		try {
			for (UT2004Bot<IVisionWorldView, IAct, UT2004BotController> agent : agents) {
				String simpleID = simplefyID(agent.getComponentId());
				UT2004BotController controller = agent.getController();
				registerEntity(simpleID, "bot", controller, createActionHandler(controller), createPerceptHandler(controller));
			}
		} catch (EntityException e) {

			// Clean up agents if they could not be registered.
			for (UT2004Bot<IVisionWorldView, IAct, UT2004BotController> agent : agents) {
				agent.stop();
			}

			throw new ManagementException("Unable to register entity", e);
		}

		// 7. Add bot dead listeners
		for (UT2004Bot<IVisionWorldView, IAct, UT2004BotController> agent : agents) {
			String simpleID = simplefyID(agent.getComponentId());
			agentDownListeners.put(simpleID, new AgentDownListener(simpleID, agent));
		}

		// 8. Check if bots are still alive. Throw out if not.
		for (UT2004Bot<IVisionWorldView, IAct, UT2004BotController> agent : agents) {
			if (agent.inState(IAgentStateDown.class)) {
				String simpleID = simplefyID(agent.getComponentId());
				agentDownListeners.get(simpleID).removeListener();
				synchronizedDeleteEntity(simpleID);
			}
		}

		// 9. Everything aokay!

	}

	protected abstract PerceptHandler createPerceptHandler(UT2004BotController controller ) throws EntityException;

	protected abstract ActionHandler createActionHandler(UT2004BotController controller) throws EntityException;

	/**
	 * Simplifies a given string representation of an AgentID by removing the
	 * UUID making it human readable.
	 * 
	 * By the specs of the agentID this should still result in a unique but
	 * readable ID.
	 * 
	 * @param iAgentId
	 * @return the agentID without the UUID.
	 */
	private String simplefyID(IAgentId agentID) {
		String token = agentID.getToken();
		int index = token.lastIndexOf('/');
		// If the expected separator could not be found, use the whole string.
		if (index < 0) {
			log.severe("Could not find UUID seperator in Agent ID: " + token);
			return token;

		}

		return token.substring(0, index);
	}

	protected void startEnvironment() throws ManagementException {

		IUT2004Server server = utServerConnection.getServerFlag().getFlag();

		if (server != null) {
			Pause resume = new Pause(false, false);
			server.getAct().act(resume);

			log.info("The environment has been started.");
		} else {
			log.warning("We are not connected to ut server and could not start the environment.");
		}
	}

	protected void pauseEvironment() {

		IUT2004Server server = utServerConnection.getServerFlag().getFlag();

		if (server != null) {
			Pause pause = new Pause(true, false);
			server.getAct().act(pause);
			log.info("The environment has been paused.");
		} else {
			log.warning("We are not connected to ut server and could not pause the environment.");
		}
	}

	protected synchronized void killEnvironment() {
		// 1. Unpause the environment.
		IUT2004Server server = utServerConnection.getServerFlag().getFlag();
		if (server != null) {
			Pause resume = new Pause(false, false);
			server.getAct().act(resume);

			log.info("The environment has been unpaused.");
		} else {
			log.warning("We are not connected to ut server and could not unpause the environment.");
		}

		// 2. Shut down all bots.
		for (String id : getEntities()) {
			@SuppressWarnings("unchecked")
			UT2004BotController<UT2004Bot> controller = ((UT2004BotController<UT2004Bot>) getEntity(id));
			UT2004Bot bot = controller.getBot();

			try {
				agentDownListeners.get(id).removeListener();
				bot.stop();
				log.info(bot.getName() + " has been stopped");
			} catch (AgentException e) {
				// When a bot can not be stopped it will be killed.
				log.info(bot.getName() + " has been killed", e);
			}
		}

		// 3. Close the connection to the utServer.
		utServerConnection.stopServer();

		// 4. Clear up bots and server
		botParameters = null;
		environmentParameters = null;

		// 5. Stop pogamut platform. Prevents memory leak see #1727
		Pogamut.getPlatform().close();
	}

	/**
	 * Synchronized version of {@link AbstractEnvironment.deleteEntity}. Used by
	 * {@link AgentDownListener} to removed agents that have shut down.
	 * 
	 * @param entity
	 *            to remove.
	 */
	protected synchronized void synchronizedDeleteEntity(String name) {
		try {
			deleteEntity(name);
		} catch (RelationException e) {
			// TODO: This relationship exception is no longer thrown.
			log.severe("Could not delete entity " + name);
		} catch (EntityException e) {
			// TODO: This should be replaced by an assertion in the default
			// implementation of EIS.
			log.severe("Could not delete entity " + name + ", it was already deleted.");
		}
	}

	/**
	 * Monitors the the agent state, if the agent goes down it is removed from
	 * the environment.
	 * 
	 * @author M.P. Korstanje
	 * 
	 */
	private class AgentDownListener implements FlagListener<IAgentState> {

		private final String key;
		private final UT2004Bot agent;

		public AgentDownListener(String key, UT2004Bot agent) {
			this.key = key;
			this.agent = agent;
			this.agent.getState().addStrongListener(this);
		}

		@Override
		public void flagChanged(IAgentState state) {
			if (state instanceof IAgentStateDown) {
				removeListener();
				synchronizedDeleteEntity(key);
			}
		}

		public void removeListener() {
			agent.getState().removeListener(this);
			agentDownListeners.remove(key);
		}
	}

	@Override
	protected boolean isSupportedByEnvironment(Action arg0) {
		return true;
	}

	@Override
	protected boolean isSupportedByType(Action arg0, String arg1) {
		return true;
	}

}
