/**
 * Emohawk Bot, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and Emohawk. 
 * 
 * Copyright (C) 2012 Emohawk Bot authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */

package nl.tudelft.goal.emohawk.agent;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import nl.tudelft.goal.EIS2Java.annotation.AsAction;
import nl.tudelft.goal.EIS2Java.annotation.AsPercept;
import nl.tudelft.goal.EIS2Java.handlers.AsynchronousEntity;
import nl.tudelft.goal.EIS2Java.translation.Filter.Type;
import nl.tudelft.goal.emohawk.translators.UnrealIdOrLocation;
import nl.tudelft.goal.emohawk.translators.UnrealIdOrLocationTranslator;
import nl.tudelft.goal.unreal.messages.BotParameters;
import nl.tudelft.goal.unreal.messages.Parameters;
import SteeringProperties.ObstacleAvoidanceProperties;
import SteeringProperties.WalkAlongProperties;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.emohawk.agent.module.sensomotoric.EmoticonType;
import cz.cuni.amis.pogamut.emohawk.agent.module.sensomotoric.Place;
import cz.cuni.amis.pogamut.emohawk.bot.impl.EmohawkBotController;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.params.UT2004AgentParameters;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.utils.exception.PogamutException;
import eis.exceptions.ActException;

public class EmohawkBotBehavior extends EmohawkBotController<UT2004Bot> implements AsynchronousEntity {

	private Semaphore logic = new Semaphore(0, true);

	private WalkAlongProperties walkAlongProperties;

	private BotParameters parameters;

	@SuppressWarnings("rawtypes")
	public void initializeController(UT2004Bot bot) {
		super.initializeController(bot);

		// Setup parameters
		IAgentLogger logger = bot.getLogger();
		UT2004AgentParameters parameters = bot.getParams();
		if ((parameters instanceof BotParameters)) {
			this.parameters = (BotParameters) parameters;
		} else {
			log.warning("Provided parameters were not a subclass of UnrealGoalParameters, using defaults.");
			this.parameters = new BotParameters(logger);
		}
		Parameters defaults = BotParameters.getDefaults(logger);
		this.parameters.assignDefaults(defaults);
	}

	protected void initializeModules(UT2004Bot bot) {
		super.initializeModules(bot);

		steering.addObstacleAvoidanceSteering(new ObstacleAvoidanceProperties());
		walkAlongProperties = new WalkAlongProperties();
		walkAlongProperties.setDistanceFromThePartner(200);
		walkAlongProperties.setGiveWayToPartner(false);
		steering.addWalkAlongSteering(new WalkAlongProperties());
	}

	/**
	 * Prepares the initialization message for Gamebots using the
	 * {@link BotParameters} provided to the {@link UT2004BotRunner}.
	 * 
	 */
	@Override
	public Initialize getInitializeCommand() {
		assert parameters != null;

		// Prepare init command
		Initialize init = super.getInitializeCommand();
		init.setDesiredSkill(parameters.getSkill());
		init.setSkin(parameters.getSkin().getUnrealName());
		init.setTeam(parameters.getTeam());
		init.setShouldLeadTarget(parameters.shouldLeadTarget());
		init.setLocation(parameters.getStartLocation());
		init.setRotation(parameters.getStartRotation());
		// Set log level.
		bot.getLogger().setLevel(this.parameters.getLogLevel());

		return init;
	}

	@Override
	public void logic() throws PogamutException {
		super.logic();
		
		// Mark that another logic iteration has began
		log.fine("--- Logic iteration ---");

		// 1. We release a permit for the actions to be executed.
		logic.release();

		// 2. So actions are executed in this window.

		// 3. After which we retract the permit and end the logic.
		try {
			logic.acquire();
		} catch (InterruptedException e) {
			throw new PogamutException("Logic was interupted while aquiring a permit.", e);
		}

	}

	@Override
	public void botShutdown() {
		super.botShutdown();

		/*
		 * The bot has stopped so we can release a permit.
		 * 
		 * TODO: This should not be required as it implies that actions and
		 * percepts are requested after the environment has been killed. However
		 * EIS currently allows different threads to manipulate the environment
		 * at the same time. Thus it may be possible for one thread to be
		 * waiting to acquire this agent while another shuts down the
		 * environment.
		 */
		logic.release();
	}

	@Override
	public void acquire() throws InterruptedException {
		/*
		 * As an extra permit is released in botShutdown the attempt to acquire
		 * should never timeout.
		 */
		if (!logic.tryAcquire(10, TimeUnit.SECONDS)) {
			throw new InterruptedException("Could not aquire a permit within the timeout.");
		}
	}

	@Override
	public void release() {
		logic.release();
	}

	//
	// ============
	// EIS Percepts
	// ============
	//

	@AsPercept(name = "navigation")
	public String navigation() {

		if (steering.isNavigating()) {
			return "following";
			// FIXME: Would like to uss navigator.isExecuting here but it does
			// not consider all it's components. traveling.
		} else if (pathExecutor.isExecuting() || getBackToNavGraph.isExecuting() || runStraight.isExecuting()) {
			return "traveling";
		} else if (pathExecutor.isStuck()) {
			return "stuck";
		} else if (pathExecutor.isPathUnavailable()) {
			return "path_unavailable";
		} else if (pathExecutor.isTargetReached()) {
			return "destination_reached";

		} else {
			return "waiting";
		}
	}

	@AsPercept(name = "navPoint", multiplePercepts = true, filter = Type.ONCE)
	public Collection<NavPoint> perceptNavPoints() {
		return getWorld().getAll(NavPoint.class).values();
	}

	// TODO: Because EIS2Java assumes objects returned are unchanging, which is
	// not the case with pogamut we can't use any other filters but ALWAYS
	// (default) and ONCE.
	@AsPercept(name = "person", multiplePercepts = true)
	public Collection<Player> person() {
		return getPlayers().getVisiblePlayers().values();
	}

	@AsPercept(name = "self")
	public AgentInfo self() {
		return getInfo();
	}

	@AsPercept(name = "emoticon", multiplePercepts = true, filter = Type.ONCE)
	public Collection<EmoticonType> emoticon() {
		return Arrays.asList(EmoticonType.values());
	}

	// @AsPercept(name = "animation", multiplePercepts = true, filter =
	// Type.ONCE)
	// public Set<AnimType> animation() {
	// return animations.getAvailableAnimations();
	// }

	@AsPercept(name = "place", multiplePercepts = true, filter = Type.ONCE)
	public Collection<Place> place() {
		return places.getPlaces();
	}

	//
	// =============
	// EIS ACTIONS
	// =============
	//
	@AsAction(name = "stop")
	public void stop() {
		steering.stopNavigation();
		navigation.stopNavigation();
	}
	
	@AsAction(name = "runTo")
	public void runTo(UnrealIdOrLocation destination) {
		ILocated location = getLocation(destination);

		steering.stopNavigation();
		if (!move.isRunning())
			move.setRun();
		navigation.navigate(location);
	}


	@AsAction(name = "walkTo")
	public void walkTo(UnrealIdOrLocation destination) {
		ILocated location = getLocation(destination);

		steering.stopNavigation();
		if (move.isRunning())
			move.setWalk();
		navigation.navigate(location);
	}
	

	private ILocated getLocation(UnrealIdOrLocation destination) {
		ILocated location;
		if (destination.isLocation()) {
			location = destination.getLocation();
		} else {
			// Assuming ID's were passed properly this must be a player or
			// navpoint.
			location = (ILocated) world.get(destination.getId());
		}
		return location;
	}


	@AsAction(name = "walkAlong")
	public void walkAlong(Player partner) throws ActException {
		navigation.stopNavigation();
		if (!move.isRunning())
			move.setRun();
		walkAlongProperties.setPartnerName(partner.getName());
		steering.setWalkAlongSteering(walkAlongProperties);
		steering.startNavigation();
	}

	@AsAction(name = "emote")
	public void emote(EmoticonType left, EmoticonType center, EmoticonType right) throws ActException {
		emoticons.clearEmoticons();

		if (left == EmoticonType.NONE && center == EmoticonType.NONE && right == EmoticonType.NONE) {
			return;
		}

		if (left == EmoticonType.NONE && center != EmoticonType.NONE && right == EmoticonType.NONE) {
			emoticons.setCenterEmoticonType(center);
			return;
		}

		if (left != EmoticonType.NONE && center == EmoticonType.NONE && right != EmoticonType.NONE) {
			emoticons.setDoubleEmoticon(left, right);
			return;
		}

		emoticons.setTripleEmoticon(left, center, right);

	}

	// @AsAction(name = "animate")
	// public void animate(AnimType animation, boolean loop) throws ActException
	// {
	// animations.stopAnimation();
	// animations.playAnimation(animation, loop);
	// }

	@AsAction(name = "turn")
	public void turn(int amount) throws ActException {
		move.turnHorizontal(amount);
	}

	@AsAction(name = "turnTo")
	public void turnTo(ILocated location) throws ActException {
		move.turnTo(location);
	}

	@AsAction(name = "jump")
	public void jump() throws ActException {
		move.jump();
	}

	@AsAction(name = "skip")
	public void skip() throws ActException {
		// Does nothing.
	}
}
