package cz.cuni.amis.pogamut.sposh.elements;

import java.util.List;
import java.util.NoSuchElementException;

/**
 * This is class used as intermediate class between PoshElement and DataNodes of
 * POSH elements. Some common functions are implemented here.
 *
 * @author HonzaH
 */
public abstract class PoshDummyElement extends PoshElement {

    /**
     * Regular pattern, only text that maches this pattern can be a name of the
     * POSH element. This pattern must be same as token
     * <code>NAME</code> in the PoshParser.jj
     * <p/>
     * Because of dash (-), not all strings that match this pattern are FQN.
     */
    public static final String IDENT_PATTERN = "([a-zA-Z][_\\-a-zA-Z0-9]*\\.)*[a-zA-Z][_\\-a-zA-Z0-9]*";

    /**
     * Function for moving an child element in the list of children. When node
     * wants to change position of a child, it must first change its position in
     * the data structures of the node and then it(the parent node of a child)
     * must notify its listeners that a child has moved. The child is not
     * notified (unless it has a listener on its parent).
     * <p/>
     * This method computes new index of a child in the collection and moves it
     * there. The child will be at the new position and the rest of elements
     * will move in such way that they will stay in the same order as before
     * (imagine it as a single line of balls, you take out specified ball and
     * put it at new position and other balls will move accordingly). Once all
     * elements are in their proper places, the method emits a notification
     * about movement of a child.
     *
     * @param collection list of elements, child must be one of them
     * @param child child element that will be moved
     * @param relativePosition how much will child move, how much will index of
     * a child change (new_idx = org_inx + relativePosition).
     * @return true if child has been moved, false if new position has been
     * illegal
     * @throws NoSuchElementException if child is not in the collection.
     */
    protected final boolean moveNodeInList(List collection, PoshElement child, int relativePosition) {
        // find index of child
        int childIndex = collection.indexOf(child);
        if (childIndex == -1) {
            throw new NoSuchElementException("Child " + child + "is not in collection.");
        }
        int bustPos;
        if (relativePosition > 0) {
            bustPos = childIndex + relativePosition + 1;
        } else {
            bustPos = childIndex + relativePosition;
        }
        // check if move is even possible
        if (bustPos < 0 || bustPos > collection.size()) {
            return false;
        }

        collection.add(bustPos, child);

        // remove former child
        if (relativePosition > 0) {
            collection.remove(childIndex);
        } else {
            collection.remove(childIndex + 1);
        }

        emitChildMove(child, relativePosition);
        return true;
    }

    /**
     * Is passed name used by one of elements?
     *
     * @param name name to be tested
     * @param elements elements againt which the test is done
     * @return true if name is used, false otherwise
     */
    protected static boolean isUsedName(String name, List<? extends NamedLapElement> elements) {
        for (NamedLapElement element : elements) {
            String elementName = element.getName();
            if (elementName.equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get valid name for new element, but name can't be same as any name used
     * by passed elements.
     *
     * @param template string used as prefix of created name
     * @param elements elements that must have different name that created name
     * @return valid name that can be immediately used as OK name of new drive.
     */
    protected static String getUnusedName(String template, List<? extends NamedLapElement> elements) {
        int i = 1;
        while (isUsedName(template + i, elements)) {
            i++;
        }
        return template + i;
    }
}
