package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import java.awt.datatransfer.DataFlavor;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Ancestor of representative of one element in Posh plan.
 *
 * Listens for various modifications in POSH tree structure. Listener is
 * connected on one particular PoshElement, so many events are neighbours and so
 * on.
 *
 * Various uses, manily during construction of new tree from file.
 *
 * BASIC LEMMA IS: Reality DataNode, Representation PoshWidget (Model, view,
 * controller) <p> Add new node: <ul> <li> Context menu/other stuff, like during
 * load of file</li> <li> get PoshElement(N) that should add the new node</li>
 * <li> add new child PoshElement(C) to the N</li> <li> N calls all connected
 * PoshWidgets and tells them that it has new child(newChild(N, C) listener or
 * something).</li> <li> Every PoshWidget PW will:</li> <ul> <li> Create new
 * PoshWidget CW for C</li> <li> connect CW to C</li> <li> make a line from PW
 * to CW</li> <li> position is not important, will be done later</li> </ul> <li>
 * After everything is added and DATAs and WIDGETs are in plase, reconsolidate
 * positions of everything.</li> </ul>
 *
 * <p> Delete node: Result: PoshElement DN will be deleted along with all
 * associated PoshWidgets <ul> <li>context menu(something else) will notify that
 * it should be destroyed</li> <li>DN will call all children to delete
 * themself(recursive process)</li> <li>call all PW that are connected to the DN
 * to delete themself</li> <UL> <li>PW will remove all children(do not delete
 * underlying data node)</li> <li>PW will call parent to delete themself</li>
 * </ul> </ul>
 *
 * @author Honza
 */
public abstract class PoshElement {

    /**
     * PoshElement, that is parent of this DN. Root doesn't have one (is
     * <code>null</code>), rest does have.
     */
    private PoshElement _parent;
    /**
     * Listeners on events of this DN.
     */
    private final Set<PoshElementListener> elementListeners = new HashSet<PoshElementListener>();
    private final Set<PoshElementListener> elementListenersUm = Collections.unmodifiableSet(elementListeners);

    /**
     * Get list of listeners that listen for changes of this node (new child,
     * node deletion, childMoved and change of properties)
     *
     * @return unmodifiable list of listeners, never null,
     */
    public Set<PoshElementListener> getElementListeners() {
        return elementListenersUm;
    }

    /**
     * Add new listener for events of this node.
     *
     * @param listener
     */
    public synchronized void addElementListener(PoshElementListener listener) {
        elementListeners.add(listener);
    }

    /**
     * Remove
     * <code>listener</code> from list of listeners of this node.
     *
     * @param listener
     */
    public synchronized void removeElementListener(PoshElementListener listener) {
        elementListeners.remove(listener);
    }

    /**
     * Well, this thing was originally supposed to use the
     *
     * @param name name of property
     * @param o old value, can be null, but reciever has to be able to handle it
     * @param n new value, can be null, but reciever has to be able to handle it
     */
    protected synchronized void firePropertyChange(String name, Object o, Object n) {
        PoshElementListener[] listeners = elementListenersUm.toArray(new PoshElementListener[0]);
        for (PoshElementListener listener : listeners) {
            listener.propertyChange(new PropertyChangeEvent(this, name, o, n));
        }
    }

    /**
     * Get number of children of this node that have same class as parameter.
     *
     * @param searchedClass what kind of class should the child be if it is
     * counted
     * @return Number of children of this node with same class as parameter
     */
    public int getNumberOfChildInstances(Class searchedClass) {
        int numEl = 0;
        for (PoshElement node : this.getChildDataNodes()) {
            if (node.getClass().equals(searchedClass)) {
                numEl++;
            }
        }
        return numEl;
    }

    /**
     * Get data flavour of posh plan element,used during DnD from palette to
     * PoshScene.
     *
     * @return dataFlavour of posh plan element, never null.
     */
    public abstract DataFlavor getDataFlavor();

    /**
     * Get list of children of this node.
     *
     * @return List of children
     */
    public abstract List<? extends PoshElement> getChildDataNodes();

    /**
     * Notify all listeners (mostly associated Widgets) that this dataNode has a
     * new child. After that emit recursivly children of added element in.
     *
     * @param childNode
     */
    protected final synchronized void emitChildNode(PoshElement emitedChild) {
        // emit child
        PoshElementListener[] listenersArray = elementListenersUm.toArray(new PoshElementListener[]{});
        for (PoshElementListener listener : listenersArray) {
            listener.nodeChanged(PoshTreeEvent.NEW_CHILD_NODE, emitedChild);
        }

        // emit children of emitted child
        List<? extends PoshElement> children = emitedChild.getChildDataNodes();
        for (PoshElement child : children) {
            emitedChild.emitChildNode(child);
        }
    }

    /**
     * Notify all listeners (associated Widgets) that one child of this node has
     * changed order(position).
     *
     * @param childNode
     */
    protected final synchronized void emitChildMove(PoshElement childNode, int relativeMove) {
        PoshElementListener[] listenersArray = elementListenersUm.toArray(new PoshElementListener[]{});

        for (PoshElementListener listener : listenersArray) {
            listener.nodeChanged(PoshTreeEvent.CHILD_NODE_MOVED, childNode);
        }
    }

    /**
     * Tell all listeners that this node will be deleted. Dont remove listeners,
     * it is job of listener to remove itself.
     */
    protected final synchronized void emitNodeDeleted() {
        PoshElementListener[] listenersArray = elementListenersUm.toArray(new PoshElementListener[]{});

        for (PoshElementListener listener : listenersArray) {
            listener.nodeChanged(PoshTreeEvent.NODE_DELETED, this);
        }
    }

    /**
     * Set parent
     * <code>PoshElement</code> of the node.
     *
     * @param parent
     */
    protected void setParent(PoshElement parent) {
        this._parent = parent;
    }

    /**
     * Get parent of the node. Null, if the node is root.
     *
     * @return parent of the node, or null if node is root.
     */
    public PoshElement getParent() {
        return this._parent;
    }

    /**
     * Get root node of POSH plan this node belongs to.
     *
     * @return Root node of POSH plan or null if I am unable to reach it.
     */
    public final PoshPlan getRootNode() {
        PoshElement cur = this;
        while (cur.getParent() != null) {
            cur = cur.getParent();
        }
        if (cur instanceof PoshPlan) {
            return (PoshPlan) cur;
        }
        return null;
    }

    /**
     * This will add new child to the tree. Every subclass that implements this
     * have to decide, if passed
     * <code>newChild</code> can be children (real type or some other things)
     * and add it to its children. If child is not accepted, don't add it
     *
     * REMEBER: that in widget view I should see nodes in groups(not Comp, Freq,
     * Comp), so adjust getChildDataNodes accordingly (Comp, Comp, Freq)
     *
     * @param newChild Child that may be added
     */
    public abstract void addChildDataNode(PoshElement newChild) throws ParseException;

    /**
     * Move child node in list of children up or down. After child was moved (if
     * it was moved), notify listeners.
     *
     * @return if node succesfully moved
     */
    public abstract boolean moveChild(PoshElement child, int relativePosition);

    /**
     * Neutralize childNode of the node.
     *
     * Take subtree consisting from <tt>childNode</tt> and all its children and
     * remove it from the POSH plan tree + add some things to keep the tree
     * consistent (add action if removed action was last one in ActionPattern or
     * add CompetenceElement if removed one was last CompetenceElement in
     * Competence).
     *
     * @param childNode node that will be neutralized
     */
    public abstract void neutralizeChild(PoshElement childNode);

    /**
     * What is relative position of this PoshElement respective to newNode.
     *
     * @param newNode
     * @return How much do you need to change position for newNode to move to
     * position of this node.
     */
    protected int getRelativePosition(PoshElement newNode) {
        int dataNodePos = 0;
        int newNodePos = 0;
        int index = 0;

        for (PoshElement sibling : _parent.getChildDataNodes()) {
            if (sibling == this) {
                dataNodePos = index;
            }
            if (sibling == newNode) {
                newNodePos = index;
            }
            index++;
        }

        return dataNodePos - newNodePos;
    }

    /**
     * Go through subtree by DFS and for every node emit that it was deleted.
     * Clear listeners in whole subtree. Doesn't destroy the structure of
     * childNode
     */
    final public void remove() {
        // neutralize
        for (PoshElement child : this.getChildDataNodes()) {
            child.remove();
        }

        // remove listeners
        emitNodeDeleted();

//XXX: listeners should remove themselves        this.elementListeners.clear();
    }

    /**
     * Neutralize this node (ask parent to neutralize this child).
     * Neutralization is similar to removal, but ensures correctness of posh
     * plan, i.e. if you try to remove something that is necessary, the
     * neutralization will insert stub instead.
     */
    final public void neutralize() {
        getParent().neutralizeChild(this);
    }
}
