package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.engine.timer.SystemClockTimer;
import cz.cuni.amis.pogamut.sposh.engine.timer.ITimer;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.Set;
import java.util.logging.Logger;

/**
 * This class is responsible for executing the valid posh plan.
 * 
 * @author Honza
 */
public class PoshEngine {

    public enum EvaluationResult {
        GOAL_SATISFIED,
        ELEMENT_FIRED,
        NO_ELEMENT_FIRED
    }
    
    public static class EvaluationResultInfo {
    	
    	public EvaluationResult result;
    	public FireResult.Type type;
    	
    	public EvaluationResultInfo(EvaluationResult result, FireResult.Type type) {
    		this.result = result;
    		this.type = type;
    	}
    	
    }

    protected PoshPlan plan;
    protected ITimer timer;
    protected Logger log;
    private DCExecutor dcExecutor;

    public PoshEngine(PoshPlan plan) {
        this(plan, new SystemClockTimer());
    }

    public PoshEngine(PoshPlan plan, ITimer timer) {
        this(plan, timer, null);
    }

    public PoshEngine(PoshPlan plan, ITimer timer, Logger log) {
        this.plan = plan;
        this.timer = timer;
        this.log = log;
        reset();
    }

    /**
     * Reset the posh engine, all stacks and variables will be reseted.
     * Use this to return engine to former state, it had when first initialized.
     */
    public final synchronized void reset() {
        dcExecutor = new DCExecutor(plan, new VariableContext(), timer, log);
    }

    public synchronized EvaluationResultInfo evaluatePlan(IWorkExecutor workExecuter) {
        return dcExecutor.fire(workExecuter);
    }

    ElementStackTrace getStackForDE(String name) {
        return dcExecutor.getStackForDE(name);
    }

    
    ElementStackTrace getStackForDE(int index) {
        return dcExecutor.getStackForDE(index);
    }
    
    public Logger getLog() {
    	return log;
    }

    /**
     * Get set of action names utilized by this engine.
     * @return
     */
    public Set<String> getActions() {
        return plan.getActionsNames();
    }

    /**
     * Get set of sense names utilized by this engine.
     * @return
     */
    public Set<String> getSenses() {
        return plan.getSensesNames();
    }

    /**
     * Get plan of this engine (serialize the parsed plane and return it)
     * @return
     */
    public String getPlan() {
        return plan.toString();
    }

    /**
     * Return serialized state of the engine. 
     * @return
     */
    public String getState() {
        StringBuilder sb = new StringBuilder();
        for (DEExecutor de : dcExecutor.deExecutors) {
            sb.append("/Root:");
            sb.append(plan.getDriveCollection().getName());
            sb.append("/Drive:");
            sb.append(de.getName());
            TriggerResult triggerResult = de.getTriggerResult();
            if (triggerResult != null) {
                for (SenseResult senseResult : triggerResult) {
                    sb.append('[');
                    sb.append(senseResult.name);
                    sb.append(':');
                    sb.append(senseResult.wasSuccessful() ? '1' : '0');
                    sb.append(']');
                }
            }
            sb.append(de.getStackTrace().toString());
            sb.append('\n');
        }
        return sb.toString();
    }
}
