package cz.cuni.amis.pogamut.sposh.elements;

import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;

/**
 * CompetenceElement is basic unit of competence.
 * <p>
 * It has name, list of triggers and defined action that should be done
 * if triggers are fullfilled.
 * <p>
 * Example in POSH:
 * <pre>(move (trigger ((see-player))) move-player)</pre>
 * 
 * @author HonzaH
 */
public class CompetenceElement extends NamedLapElement {
    // XXX: more refined, later
    static CompetenceElement create() {
        return new CompetenceElement("choice", new Sense("succeed"), "doNothing", null);
    }

    /**
     * Create new element with specified name
     * @param name name of new element
     * @return created competence element 
     */
    static CompetenceElement create(String name) {
        return new CompetenceElement(name, new Sense("succeed"), "doNothing", null);
    }

    private String name;
    private ElementList<Sense> triggers;
    private TriggeredAction action;
    private Integer retries;
    private String comment;
    public static final String caName = "caName";
    public static final String caRetries = "caRetries";
    public static final String caComment = "caComment";

    /**
     * Create competence atom from passed parameters and assign
     * parents propertly.
     *
     * @param name Name of competence atom. Mostly for use, not used in execution of POSH plan.
     * @param triggers List of senses. If triggers are fulfilled, competence can execute the action.
     * @param actionName Name of action, can be normal action, ActionPattern name or another competence name. Cycles are not allowed.
     * @param retries can be null.
     */
    public CompetenceElement(String name,
            ElementList<Sense> triggers,
            String actionName,
            Integer retries,
            String comment) {
        this(name, triggers, new Sense.SenseCall(actionName), retries, comment);
    }
    
    public CompetenceElement(String name,
            ElementList<Sense> triggers,
            Sense.SenseCall actionCall,
            Integer retries,
            String comment) {
        if (triggers == null) {
            triggers = new ElementList<Sense>();
        }

        this.name = name;
        this.triggers = triggers;
        this.action = new TriggeredAction(actionCall);
        this.retries = retries;
        this.comment = comment;

        action.setParent(this);
        for (Sense sense : triggers) {
            sense.setParent(this);
        }
    }

    /**
     * Create an CompetenceElement from passed parameters.
     * See {@link CompetenceElement#CompetenceElement(java.lang.String, cz.cuni.pogamut.posh.tree.ElementList, java.lang.String, java.lang.Integer) other constructor}
     * for parameters.
     */
    public CompetenceElement(String name,
            Sense triggers,
            String actionName,
            Integer retries) {
        this(name, new ElementList<Sense>(new Sense[]{triggers}), actionName, retries, null);
    }

    /**
     * Add trigger sense to this competence atom.
     * Emit new child node after addition.
     *
     * @param trigger Sense, that is necessary to trigger this action.
     */
    public void addTriggerAct(Sense trigger) {
        trigger.setParent(this);
        triggers.add(trigger);
        emitChildNode(trigger);
    }

    /**
     * This method differs from {@link CompetenceElement.addTriggerAct} because if
     * the only trigger is <b>fail</b> or <b>succeed</b>, than replace it with
     * passed sense.
     * <p>
     * Otherwise same as {@link CompetenceElement.addTriggerAct}.
     */
    public void addUserTrigger(Sense trigger) {
        if (triggers.size() == 1 && (
                "fail".equals(triggers.get(0).getSenseName()) ||
                "succeed".equals(triggers.get(0).getSenseName()))) {
            triggers.get(0).changeTo(trigger);
        } else {
            addTriggerAct(trigger);
        }
    }

    /**
     * Set action (by removing old action and emiting new one) of this
     * CompetenceElement.
     *
     * @param newAction
     */
    public void setAction(TriggeredAction newAction) {
        // Remove old action
        action.remove();
        action.setParent(null);

        // and set new one
        newAction.setParent(this);
        this.action = newAction;
        emitChildNode(newAction);
    }

    public TriggeredAction getAction() {
        return action;
    }

    @Override
    public String toString() {
        String ret;
        if (triggers.size() == 0) {
            ret = "(" + name + " " + action.getName();
        } else {
            ret = "(" + name + " (trigger " + triggers.toString() + ") " + action.getName();
        }

        if (retries != null) {
            ret += " " + retries.toString();
        }

        if (comment != null) {
            ret += " \"" + comment + "\"";
        }

        return ret + ")";
    }

    @Override
    public List<PoshElement> getChildDataNodes() {
        List<PoshElement> children = new ArrayList<PoshElement>(triggers);
        children.add(action);
        return children;
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        this.comment = comment;
    }

    @Override
    public String getName() {
        return this.name;
    }

    /**
     * Set name of Competence atom, name has to match <tt>IDENT_PATTERN</tt>, trimmed.
     * @param name
     */
    public void setName(String name) {
        name = name.trim();
        if (name.matches(IDENT_PATTERN)) {
            this.name = name;
            firePropertyChange(caName, null, name);
        }
    }

    public Integer getRetries() {
        return this.retries;
    }

    public void setRetries(Integer retries) {
        this.retries = retries;
        firePropertyChange(caRetries, null, retries);
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return moveNodeInList(this.triggers, child, relativePosition);
    }
    public static final DataFlavor dataFlavor = new DataFlavor(CompetenceElement.class, "competence-atom");

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement newChild) {
        if (newChild instanceof Sense) {
            this.addTriggerAct((Sense) newChild);
        } else if (newChild instanceof TriggeredAction) {
            this.setAction((TriggeredAction) newChild);
        }
    }

    @Override
    public void neutralizeChild(PoshElement childNode) {
        if (this.action == childNode) {
            this.setAction(new TriggeredAction("do_nothing"));
        } else if (this.triggers.contains(childNode)) {
            if (this.triggers.size() <= 1) {
                this.addTriggerAct(new Sense("succeed"));
            }
            this.triggers.remove(childNode);
            childNode.remove();
        }
    }

    /**
     * Get unmodifiable list of trigger senses in the CE
     */
    public List<Sense> getTriggerSenses() {
        return Collections.unmodifiableList(triggers);
    }
}
