package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.exceptions.*;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Structure for holding information about competences in POSH file. Competence
 * is basic reactive plans. They allow a BOD agent to complete a task robustly
 * and opportunistically.
 *
 * @author HonzaH
 */
public final class Competence extends NamedLapElement implements Comparable<Competence> {

    /**
     * Name of this competence, can be referenced
     */
    private String name;
    /**
     * Formal parameters of the competence
     */
    private FormalParameters params = new FormalParameters();
    /**
     * List of all possible elements (choices) of this competence
     */
    private final List<CompetenceElement> elements = new LinkedList<CompetenceElement>();
    /**
     * Unmodifiable proxy list of elements
     */
    private final List<CompetenceElement> elementsUm = Collections.unmodifiableList(elements);
    /**
     * Property string of competence name
     */
    public static final String cnName = "cnName";
    /**
     * Data flavor of competence classs, used for drag-and-drop
     */
    public static final DataFlavor dataFlavor = new DataFlavor(Competence.class, "competence-node");

    /**
     * Create a new Competence with passed name and assign passed elements to
     * this competence (set parent).
     *
     * @param name Name of competence node, it can be referenced
     * @param elements List of elements that are not part of any other
     * competence. Shallow copy
     * @throws FubarException if names of elements are not unique
     */
    public Competence(String name, FormalParameters params, List<CompetenceElement> elements) {
        this.name = name;
        this.params = new FormalParameters(params);

        try {
            for (CompetenceElement element : elements) {
                addElement(element);
            }
        } catch (DuplicateNameException ex) {
            Logger.getLogger(Competence.class.getName()).log(Level.SEVERE, ex.getMessage(), ex);
            throw new FubarException(ex);
        }
    }

    /**
     * Add passed element as the last element of this competence and emit.
     *
     * @param element element that will be added into this competence
     */
    public void addElement(CompetenceElement element) throws DuplicateNameException {
        // sanity check that element's parent doesn't list element as a child
        if (element.getParent() != null) {
            assert !element.getParent().getChildDataNodes().contains(element);
        }

        if (isUsedName(element.getName(), elementsUm)) {
            throw new DuplicateNameException("Competence " + name + " already has element with name " + element.getName());
        }

        elements.add(element);
        element.setParent(this);

        emitChildNode(element);
    }

    /**
     * Create text representation of this competence, compatible with parser, so
     * we can directly output it.
     *
     * @return
     */
    @Override
    public String toString() {
        String ret = "\t(C " + name;
        // parameters of the competence are right after declaration
        if (!params.isEmpty()) {
            ret += " vars(" + params.toString() + ")";
        }
        ret += "\n\t\t(elements";
        // In order to utilize compatibility with older plans, use dbl  braces
        for (CompetenceElement element : elements) {
            ret += "\n\t\t\t(" + element.toString() + ")";
        }
        return ret + "\n\t\t)\n\t)";
    }

    @Override
    public List<CompetenceElement> getChildDataNodes() {
        return elementsUm;
    }

    /**
     * Change name of competence node.
     *
     * @param name new name of competence node
     */
    public void setName(String name) throws DuplicateNameException, CycleException, InvalidNameException {
        PoshPlan plan = getRootNode();

        name = name.trim();

        if (!name.matches(IDENT_PATTERN)) {
            throw new InvalidNameException("Name " + name + " is not valid.");
        }

        // Check for duplicity
        if (!this.name.equals(name)) {
            if (plan != null && !plan.isUniqueAPorComp(name)) {
                throw new DuplicateNameException("New name for competence '" + this.name + "'(" + name + ") is not unique for reaction plan.");
            }
        }

        String oldName = this.name;
        this.name = name;

        if (plan != null && plan.isCycled()) {
            this.name = oldName;
            throw new CycleException("New name (" + name + ") for competence '" + this.name + "' is causing cycle.");
        }
        firePropertyChange(cnName, oldName, name);
    }

    /**
     * Get name of the competence
     *
     * @return name of the competence
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return moveNodeInList(elements, child, relativePosition);
    }

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement element) throws DuplicateNameException {
        if (element instanceof CompetenceElement) {
            addElement((CompetenceElement) element);
        } else {
            throw new UnexpectedElementException("Class " + element.getClass().getSimpleName() + " not accepted.");
        }
    }

    @Override
    public void neutralizeChild(PoshElement child) {
        if (elements.contains(child)) {
            if (elements.size() == 1) {
                String unusedName = getUnusedName("choice-", elementsUm);
                try {
                    addElement(CompetenceElement.create(unusedName));
                } catch (DuplicateNameException ex) {
                    String msg = "Unused name " + unusedName + " is not unused.";
                    Logger.getLogger(Competence.class.getName()).log(Level.SEVERE, msg, ex);
                    throw new FubarException(msg, ex);
                }
            }
            elements.remove(child);
            child.remove();
        } else {
            throw new UnexpectedElementException("Not expecting " + child.toString());
        }
    }

    @Override
    public int compareTo(Competence o) {
        return toString().compareTo(o.toString());
    }

    /**
     * Get list of formal parametrs of competence (names and default values).
     *
     * @return
     */
    public FormalParameters getParameters() {
        return params;
    }
}
