package cz.cuni.amis.pogamut.sposh.elements;

import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.HashSet;
import javax.swing.JOptionPane;

/**
 * Triggered action is basically identifier, string that maches some function specified
 * outside the POSH plan. It can also be a reference to Action pattern or competence 
 * (if action and reactive plan has same name). Cycles are not allowed.
 *
 * @author Honza
 */
public class TriggeredAction extends NamedLapElement implements Comparable<TriggeredAction> {
	public static final String taName = "taName";
	private Sense.SenseCall actionCall;

	public TriggeredAction(String actionName) {
		this.actionCall = new Sense.SenseCall(actionName);
	}

	public TriggeredAction(Sense.SenseCall actionCall) {
		this.actionCall = actionCall;
	}

	@Override
	public List<PoshElement> getChildDataNodes() {
		return Collections.<PoshElement>emptyList();
	}

	@Override
	public String toString() {
		return actionCall.toString();
	}
    @Override
	public String getName() {
		return actionCall.getName();
	}

        public Sense.SenseCall getActionCall() {
            return actionCall;
        }

	/**
	 * Does this action has a cycle?
	 * Cycle = competence with name X has action with name X or similar situations.
	 *
	 * @return true if it has, false if it doesn't
	 */
	protected boolean detectCycle() {
		PoshPlan root = this.getRootNode();
		if (root == null) {
			throw new RuntimeException("Missing root node.");
		}

		HashSet<String> connected = new HashSet<String>();

		connected.add(getName());

		// now go through whole set and find AP or Competence
		int addedItems;
		do {
			addedItems = 0;
			String[] connectedArray = connected.toArray(new String[]{});
			
			for (String processedAction : connectedArray) {
				List<String> compositeChildren = root.getAPorCompActions(processedAction);
				// and if it exists, add all
				for (String compositeChild : compositeChildren) {
					if (this.getName().equals(compositeChild)) {
						return true;
					}

					if (!connected.contains(compositeChild)) {
						connected.add(compositeChild);
						addedItems++;
					}
				}
			}
		} while (addedItems > 0);

		return false;
	}

	/**
	 * Set action function. It has to match with IDENT_PATTERN and it can't
	 * cause cycle. If cycle is detected, messagebox appears.
	 * 
	 * @param actionName
	 */
	public void setActionName(String actionName) {
		actionName = actionName.trim();

		if (!actionName.matches(IDENT_PATTERN)) {
			throw new IllegalArgumentException("Invalid name.");
		}

		String oldName = this.getName();
		this.actionCall = new Sense.SenseCall(actionName, actionCall.getParameters());

		if (getRootNode() != null && getRootNode().isCycled()) {
			this.actionCall = new Sense.SenseCall(oldName, actionCall.getParameters());

			JOptionPane.showMessageDialog(
                                null,
                                "This namechange is causing cycle, please use another name.",
                                "Cycle detected", JOptionPane.ERROR_MESSAGE);
		} else {
                        // Don't use old name because it doesn't fire when oldname and
                        // new name are same, but I am relying on firing after added competence or
                        // action pattern.
			firePropertyChange(taName, null, actionName);
		}
	}

	@Override
	public boolean moveChild(PoshElement child, int relativePosition) {
		return false;
	}
	
	public static final DataFlavor dataFlavor = new DataFlavor(TriggeredAction.class, "triggered_action");

	@Override
	public DataFlavor getDataFlavor() {
		return dataFlavor;
	}

	@Override
	public void addChildDataNode(PoshElement newChild) {
		throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
	}

	@Override
	public void neutralizeChild(PoshElement childNode) {
		throw new RuntimeException("TriggeredActions doesn't have children.");
	}

    @Override
    public int compareTo(TriggeredAction o) {
        return this.getName().compareTo(o.getName());
    }

}
