package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.elements.SolTime.TimeUnits;
import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.LinkedList;
import javax.swing.JOptionPane;

/**
 * Structure for holding information about competences in POSH file.
 * Competence is basic reactive plans. They allow a BOD agent to complete
 * a task robustly and opportunistically.
 * 
 * @author Honza
 */
public class Competence extends PoshDummyElement implements Comparable<Competence> {

    private String _name;
    private SolTime _timeout;
    private Goal _goal;
    private List<CompetencePriorityElement> _elements;
    private String _comment;
    public static final String cnName = "cnName";
    public static final String cnTimeoutAmmount = "cnTimeoutAmmount";
    public static final String cnTimeoutUnits = "cnTimeoutUnits";
    public static final String cnComment = "cnComment";

    /**
     * Create a new Competence from passed parameters.
     *
     * @param name Name of competence node, it can be referenced by TriggeredAction.
     * @param solTime
     * @param goal
     * @param elementList
     * @param comment
     */
    public Competence(String name,
            SolTime solTime,
            Goal goal,
            List<CompetencePriorityElement> elementList,
            String comment) {
        this._name = name;
        this._timeout = solTime;
        this._goal = goal;
        this._elements = elementList;
        this._comment = comment;

        if (_goal != null) {
            _goal.setParent(this);
        }
        
        for (CompetencePriorityElement priorityElement : elementList) {
            priorityElement.setParent(this);
        }

    }

    public void addCompetencePriorityElement(CompetencePriorityElement competenceElement) {
        competenceElement.setParent(this);
        this._elements.add(competenceElement);

        emitChildNode(competenceElement);
    }

    /**
     * Remove old goal node if present, clear it and emit new goal node.
     * @param goal
     */
    public void setGoalNode(Goal goal) {
        goal.setParent(this);
        if (_goal != null) {
            _goal.remove();
        }
        _goal = goal;

        emitChildNode(goal);
    }

    /**
     * Return goal of competence.
     * @return goal or null, if no goal was specified.
     */
    public Goal getGoal() {
        return this._goal;
    }

    @Override
    public String toString() {
        String ret = "\t(C " + _name;

        // parameters of the competence are right after declaration
        if (!parameters.isEmpty()) {
            ret += " vars(" + parameters.toString() + ")";
        }

        if (_timeout != null) {
            ret += " " + _timeout.toString();
        }

        if (_goal != null) {
            ret += " " + _goal.toString();
        }
        
        ret += "\n\t\t(elements";
        for (CompetencePriorityElement element : _elements) {
            ret += "\n\t\t\t(" + element.toString() + ")";
        }

        return ret + "\n\t\t)\n\t)";
    }

    @Override
    public List<PoshElement> getChildDataNodes() {
        List<PoshElement> children = new ArrayList<PoshElement>(_elements);
        
        if (_goal != null) {
            children.add(0, _goal);
        }
        return children;
    }

    /**
     * Get unmodifiable list of <tt>CompetencePriorityElement</tt>s of this node.
     * @return List of competence elements in this node.
     */
    public List<CompetencePriorityElement> getPriorityElements() {
        return Collections.unmodifiableList(_elements);
    }

    /**
     * Get list of all {@link CompetenceElement}s of this
     * {@link Competence} sorted according to priority.
     * List is created during during the call.
     * @return list of competence elemnts sorted according to priority.
     */
    public LinkedList<CompetenceElement> getElements() {
        LinkedList<CompetenceElement> elementsList = new LinkedList<CompetenceElement>();

        for (CompetencePriorityElement priorityElement : getPriorityElements()) {
            for (CompetenceElement element : priorityElement.getElements()) {
                elementsList.add(element);
            }
        }
        
        return elementsList;
    }

    /**
     * Set comment for this node.
     * @param comment text without "
     */
    public void setNodeComment(String comment) {
        if (!comment.contains("\"")) {
            this._comment = comment;
            firePropertyChange(cnComment, null, comment);
        }

    }

    public String getNodeComment() {
        return this._comment;
    }

    public Double getNodeTimeoutAmmount() {
        return this._timeout.getCount();
    }

    /**
     * Set maw many timeout units should competence node wait until it gives up.
     * @param a
     */
    public void setNodeTimeoutAmmount(Double a) {
        if (a != null) {
            this._timeout.setCount(a);
            firePropertyChange(cnTimeoutAmmount, null, a);
        }

    }

    public Integer getNodeTimeoutUnits() {
        return this._timeout.getUnits().getId();
    }

    /**
     * Set units for timeout of competence node.
     *
     * @param newUnitsId Interger from enum <tt>TimeUnits</tt>.<tt>getId()</tt>.
     */
    public void setNodeTimeoutUnits(Integer newUnitsId) {
        if (newUnitsId != null) {
            for (TimeUnits unit : TimeUnits.values()) {
                if (unit.getId() == newUnitsId) {
                    _timeout.setUnits(unit);
                    firePropertyChange(cnTimeoutUnits, null, unit);
                }

            }
        }
    }

    /**
     * Set name of competence node. This name can be referenced by TriggeredAction.
     * @param name string matching IDENT_PATTERN
     */
    public void setNodeName(String name) throws ParseException {
        name = name.trim();

        if (name.matches(IDENT_PATTERN)) {

            if (!this.getNodeName().equals(name)) {
                if (getRootNode() != null && !getRootNode().isUniqueAPorComp(name)) {
                    String msg = "New name for competence '" + this._name + "'(" + name + ") is not unique for reaction plan.";
                    JOptionPane.showMessageDialog(null, msg, "Duplicate name", JOptionPane.ERROR_MESSAGE);

                    return;
                }
            }

            String oldName = this._name;
            this._name = name;

            if (getRootNode() != null && getRootNode().isCycled()) {
                this._name = oldName;

                String msg = "New name (" + name + ") for competence '" + this.getNodeName() + "' is causing cycle.";
                JOptionPane.showMessageDialog(null, msg, "Cycle detected", JOptionPane.ERROR_MESSAGE);

                return;

            }



            firePropertyChange(cnName, null, name);
        }

    }

    public String getNodeName() {
        return this._name;
    }
/*
*/
    @Override
    public String getDisplayName() {
        return this.getNodeName();
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return moveNodeInList(this._elements, child, relativePosition);
    }
    public static final DataFlavor dataFlavor = new DataFlavor(Competence.class, "competence-node");

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement newChild) {
        if (newChild instanceof Goal) {
            this.setGoalNode((Goal) newChild);
        } else if (newChild instanceof CompetencePriorityElement) {
            this.addCompetencePriorityElement((CompetencePriorityElement) newChild);
        } else {
            throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
        }
    }

    @Override
    public void neutralizeChild(PoshElement childNode) {
        if (childNode == this._goal) {
            this.setGoalNode(new Goal(new Sense("fail")));
        } else if (this._elements.contains(childNode)) {
            if (this._elements.size() <= 1) {
                this.addCompetencePriorityElement(new CompetencePriorityElement(new CompetenceElement("competence element", new Sense("fail"), "do_nothing", null)));
            }
            this._elements.remove(childNode);
            childNode.remove();
        }
    }

    @Override
    public int compareTo(Competence o) {
        return this.toString().compareTo(o.toString());
    }

    protected FormalParameters parameters = new FormalParameters();

    /**
     * Set formal parameters of this competence.
     * @param parameters
     */
    protected void setParameters(FormalParameters parameters) {
        this.parameters = new FormalParameters(parameters);
    }

    /**
     * Get list of formal parametrs of competence (names and default values).
     * @return
     */
    public FormalParameters getParameters() {
        return parameters;
    }
}
