package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.engine.timer.ITimer;
import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.Freq;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.amis.pogamut.sposh.engine.ElementStackTrace.StackElement;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.logging.Logger;

/**
 * Executor of DE.
 * @author Honza
 */
class DEExecutor extends AbstractExecutor {
    // XXX: This should be removed, create a ElementExecutor
    //      in constructor and use copy constructor instead

    private String name;
    private PoshPlan plan;
    private Sense.SenseCall actionCall;
    private SenseListExecutor trigger;
    private Freq freq;
    private long lastFired = Integer.MAX_VALUE;
    private ElementStackTrace stackTrace = new ElementStackTrace();

    DEExecutor(PoshPlan plan, DriveElement de, VariableContext ctx, Logger log) {
        super(ctx, log);
        trigger = new SenseListExecutor(de.getTriggers(), ctx, log);
        freq = new Freq(de.getFreq());

        this.name = de.getDriveName();
        this.plan = plan;
        this.actionCall = de.getTriggeredAction().getActionCall();

        // Place ElementExecutor of the DE action on the top of stackTrace
//        stackTrace.push(createInitialStackElement(plan, actionCall));
    }

    private StackElement createInitialStackElement(PoshPlan plan, Sense.SenseCall actionCall) {
        String apname = actionCall.getName();
        for (ActionPattern ap : plan.getActionPatterns()) {
            if (apname.equals(ap.getNodeName())) {
                APExecutor newAPExecutor = new APExecutor(
                        plan, ap, FireResult.Type.SURFACE,
                        new VariableContext(ctx, actionCall.getParameters(), ap.getParameters()), log);
                
                return new StackElement(ActionPattern.class, apname, newAPExecutor);
            }
        }
        for (Competence c : plan.getCompetences()) {
            if (apname.equals(c.getNodeName())) {
                return new StackElement(Competence.class, apname, new CExecutor(plan, c, new VariableContext(ctx, actionCall.getParameters(), c.getParameters()), log));
            }
        }
        return new StackElement(TriggeredAction.class, apname, new PrimitiveExecutor(actionCall, new VariableContext(ctx, actionCall.getParameters()), log));
    }

    /**
     * Is time since last fire less than specified frequency and
     * are triggers fulfilled?
     * 
     * @param timestamp current time
     * @return true if conditions for evaluation are fulfilled
     */
    public synchronized boolean isReady(long timestamp, IWorkExecutor workExecuter) {
        long passed = timestamp - lastFired;
        // Has to be at least "freq.tick()" ms, was "passed" ms
        if (Freq.compare(freq.tick(), passed) > 0) {
            info("Max.firing frequency exceeded, has to be at least " + freq.tick() + "ms, but was only " + passed);
            return false;
        }

        return trigger.fire(workExecuter, true);
    }

    /**
     *
     * @param workExecuter
     * @param timer
     * @return true if element was fired, false otherwise
     */
    public synchronized FireResult.Type fire(IWorkExecutor workExecuter, ITimer timer) {
        // first, if stack is empty, add initial element
        if (stackTrace.isEmpty()) {
            stackTrace.add(createInitialStackElement(plan, actionCall));
            return FireResult.Type.CONTINUE;
        }

        // fire the element at the top of the stack
        FireResult result = stackTrace.peek().getExecutor().fire(workExecuter);
        lastFired = timer.getTime();

        info("The fired element returned: " + result.getType());
        switch (result.getType()) {
            case FULFILLED:
                // I wonder is this state even has meaning. If element was fulfilled,
                // shouldn't I just move up one level?
                // For now, handle it same way as original, according
                // to original sposh, handle it same way as failed case,
                // just element to DE action.
                stackTrace.removeAllElements();
                break;
            case FAILED:
                // The element has failed, reset call stack, we will try again with a clean plate
                stackTrace.removeAllElements();
                break;
            case FOLLOW:
                // Now, our executing element has decided, that some other element
                // should have preference. Add new element to the stack so it is
                // executed next time.
                stackTrace.push(result.getNextElement());
                break;
            case CONTINUE:
                // We are supposed to continue to execute the currently executed element
                // In other words, do nothing
                break;
            case SURFACE:
                // Return from this element. It is possible that the surfacing 
                // element was the default one, so if empty stack, add default
                stackTrace.pop();
                break;
            default:
                throw new IllegalStateException("State \"" + result.getType() + "\" not expected. Serious error.");
        }
        return result.getType();
    }

    ElementStackTrace getStackTrace() {
        return stackTrace;
    }

    /**
     * Get name of this DEExecutor (same as the de element)
     * @return
     */
    public String getName() {
        return name;
    }
}
