package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

/**
 * This class is root of representation of POSH source file. Basically it represents
 * the outest brackets.
 *
 * This class holds Competences, APs and DriveElement.
 * 
 * @author Honza
 */
public final class PoshPlan extends PoshDummyElement {

    private DocString _docstring = null;
    private DriveCollection _driveCollection = null;
    private List<ActionPattern> _actionPatterns = new ArrayList<ActionPattern>();
    private List<Competence> _competences = new ArrayList<Competence>();
    private String name = null;
    public static final String prName = "prName";

    public PoshPlan() {
        this.name = "";
    }

    public PoshPlan(String name) {
        this.name = name;
    }

    /**
     * Does this plan contain C with specified name?
     * @param name name of C we are checking
     * @return true if C with name is in this plane, false if not.
     */
    public boolean isC(String name) {
        return getC(name) != null;
    }

    /**
     * Return competence from this plan with specified name.
     * @param name name of searched C
     * @return competence or null if it doesn't exists
     */
    public Competence getC(String name) {
        for (Competence c : getCompetences()) {
            if (c.getNodeName().equals(name)) {
                return c;
            }
        }
        return null;
    }

    /**
     * Does this plan contain AP with specified name?
     * @param name name of AP we are checking
     * @return true if AP with name is in this plane, false if not.
     */
    public boolean isAP(String name) {
        return getAP(name) != null;
    }

    /**
     * Return action pattern from this plan with specified name.
     * @param name name of searched AP
     * @return action pattern or null if it doesn't exists
     */
    public ActionPattern getAP(String name) {
        for (ActionPattern ap : getActionPatterns()) {
            if (ap.getNodeName().equals(name)) {
                return ap;
            }
        }
        return null;
    }

    /**
     * Check if passed string is different than names of all competences and
     * action patterns. If it is, it can be used as name of new competence or
     * action pattern.
     *
     * @param id name of tested string.
     * @return true if it is unique, false otherwise.
     */
    public boolean isUniqueAPorComp(String id) {
        for (ActionPattern node : _actionPatterns) {
            if (node.getNodeName().equals(id)) {
                return false;
            }
        }

        for (Competence node : _competences) {
            if (node.getNodeName().equals(id)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Go throught the tree and for every trigger action refresh the name of
     * the action (change from X to X).
     *
     * Do this when new competence arrives because some action can have
     * same name as the newly made competence.
     *
     * Actions are refreshed from leafs to root order.
     *
     * THIS DOESN'T CHECK IF THERE IS A CYCLE, IT MUST BE DONE IN UPPER FUNCTION.
     */
    private void refreshActions() {
        //refreshActions(this);
        // First gather all actions
        List<TriggeredAction> actions = new LinkedList<TriggeredAction>();

        List<ActionPattern> aps = this.getActionPatterns();
        List<Competence> competences = this.getCompetences();

        for (ActionPattern ap : aps) {
            for (TriggeredAction action : ap.getTriggeredActions()) {
                actions.add(action);
            }
        }
        for (Competence competence : competences) {
            for (CompetencePriorityElement cpe : competence.getPriorityElements()) {
                for (CompetenceElement ce : cpe.getElements()) {
                    actions.add(ce.getAction());
                }
            }
        }

        if (getDriveCollection() != null) {
            for (DrivePriorityElement dpe : getDriveCollection().getPriorityElements()) {
                for (DriveElement de : dpe.getDriveElements()) {
                    actions.add(de.getTriggeredAction());
                }
            }
        }
        // And then refresh them
        for (TriggeredAction action : actions) {
            action.setActionName(action.getActionName());
        }
    }

    /**
     * Yay, this function adds documentation node to the tree and emits the
     * new child.
     * @param documentationNode
     */
    public void setDocString(DocString docstring) {
        docstring.setParent(this);
        if (this._docstring != null) {
            this._docstring.remove();
        }
        this._docstring = docstring;
        emitChildNode(_docstring);
    }

    /**
     * Add competence node to the POSH tree (add, emit)
     * @param competenceNode
     */
    public void addCompetence(Competence competence) throws ParseException {
        if (!this.isUniqueAPorComp(competence.getNodeName())) {
            throw new ParseException("Competence  '" + competence.getNodeName() + "' has duplicate name in POSH plan.");
        }

        competence.setParent(this);
        _competences.add(competence);

        if (this.isCycled()) {
            _competences.remove(competence);

            throw new ParseException("Competence " + competence.getNodeName() + " is causing cycle.");
        }

        emitChildNode(competence);

        refreshActions();
    }

    /**
     * Get unmodifiable array of all competences.
     * @return
     */
    public List<Competence> getCompetences() {
        return Collections.unmodifiableList(this._competences);
    }

    /**
     * Add new action pattern to the POSH tree root (add, set parents, emit)
     * @param actionPatternNode
     */
    public void addActionPattern(ActionPattern actionPattern) throws ParseException {
        if (!this.isUniqueAPorComp(actionPattern.getNodeName())) {
            throw new ParseException("Action pattern '" + actionPattern.getNodeName() + "' has duplicate name in POSH plan.");
        }

        actionPattern.setParent(this);
        _actionPatterns.add(actionPattern);

        // test cycle
        if (this.isCycled()) {
            _actionPatterns.remove(actionPattern);

            throw new ParseException("Action pattern '" + actionPattern.getNodeName() + "' is causing cycle.");
        }

        emitChildNode(actionPattern);

        refreshActions();
    }

    /**
     * Get unmodifiable array of all competences.
     * @return
     */
    public List<ActionPattern> getActionPatterns() {
        return Collections.unmodifiableList(this._actionPatterns);
    }

    /**
     * Set the drive collection. I don't think this should be ever used.
     * @param driveCollection
     */
    protected void setDriveCollection(DriveCollection dc) {
        dc.setParent(this);
        if (this._driveCollection != null) {
            this._driveCollection.remove();
        }

        this._driveCollection = dc;

        emitChildNode(this._driveCollection);
    }

    public DriveCollection getDriveCollection() {
        return this._driveCollection;
    }

    /**
     * Is POSH plan cycled?
     * @return true if plan has a cycle, false otherwise
     */
    public boolean isCycled() {

        for (ActionPattern apNode : this._actionPatterns) {
            if (findCycle(apNode, new HashSet<String>())) {
                return true;
            }
        }
        for (Competence compNode : this._competences) {
            if (findCycle(compNode, new HashSet<String>())) {
                return true;
            }
        }
        return false;
    }

    private boolean findCycle(ActionPattern apNode, Set<String> set) {
        if (set.contains(apNode.getNodeName())) {
            return true;
        }
        set.add(apNode.getNodeName());

        for (TriggeredAction action : apNode._actions) {
            ActionPattern actionAP;
            if ((actionAP = getAP(action.getActionName())) != null) {
                if (findCycle(actionAP, set)) {
                    return true;
                }
            }

            Competence actionComp;
            if ((actionComp = getC(action.getActionName())) != null) {
                if (findCycle(actionComp, set)) {
                    return true;
                }
            }
        }
        set.remove(apNode.getNodeName());
        return false;
    }

    private boolean findCycle(Competence compNode, Set<String> set) {
        if (set.contains(compNode.getNodeName())) {
            return true;
        }

        set.add(compNode.getNodeName());

        if (compNode.getPriorityElements() == null) {
            set.remove(compNode.getNodeName());

            return false;
        }

        for (CompetencePriorityElement cElem : compNode.getPriorityElements()) {
            for (CompetenceElement cAtom : cElem.competenceElements) {
                TriggeredAction action = cAtom.getAction();

                ActionPattern actionAP;
                if ((actionAP = getAP(action.getActionName())) != null) {
                    if (findCycle(actionAP, set)) {
                        return true;
                    }
                }

                Competence actionComp;
                if ((actionComp = getC(action.getActionName())) != null) {
                    if (findCycle(actionComp, set)) {
                        return true;
                    }
                }
            }
        }
        set.remove(compNode.getNodeName());
        return false;
    }

    /**
     * Try to find AP or competence with same name as <tt>ident</tt> and if
     * it exists, return list of all actions the AP or competence has. If
     * no such Ap or Competence exists, return empty list.
     *
     * Used for detection of cycles.
     *
     * @return list of names of actions that can be invoked by executing
     *         AP or competence with name <tt>ident</tt>. If nothing found,
     *         return empty list.
     */
    protected List<String> getAPorCompActions(String ident) {
        ActionPattern ap = this.getAP(ident);
        if (ap != null) {
            LinkedList<String> actionList = new LinkedList<String>();

            for (TriggeredAction a : ap._actions) {
                actionList.add(a.getActionName());
            }
            return actionList;
        }

        Competence cn = this.getC(ident);
        if (cn != null) {
            LinkedList<String> actionList = new LinkedList<String>();
            for (CompetencePriorityElement ce : cn.getPriorityElements()) {
                for (CompetenceElement ca : ce.competenceElements) {
                    if (ca.getAction() != null) {
                        actionList.add(ca.getAction().getActionName());
                    }
                }
            }
            return actionList;
        }
        return new LinkedList<String>();
    }

    /**
     * Should return POSH source of tree
     */
    @Override
    public String toString() {
        String ret = "(";

        if (this._docstring != null) {
            ret += "\n" + this._docstring.toString();
        }

        for (Competence node : _competences) {
            ret += "\n" + node.toString();
        }

        for (ActionPattern node : _actionPatterns) {
            ret += "\n" + node.toString();
        }

        if (this._driveCollection != null) {
            ret += "\n" + this._driveCollection.toString();
        }

        ret += "\n)";
        return ret;
    }

    /**
     * Get all nodes that are connected directly to this one (= documentration,
     * actonPatterns, competences, DriveCollections).
     *
     * The list is generated every time.
     *
     * @return All children of this node.
     */
    @Override
    public List<PoshElement> getChildDataNodes() {
        List<PoshElement> children = new ArrayList<PoshElement>();

        if (this._docstring != null) {
            children.add(this._docstring);
        }

        for (Competence competence : _competences) {
            children.add(competence);
        }

        for (ActionPattern actionPattern : _actionPatterns) {
            children.add(actionPattern);
        }

        if (this._driveCollection != null) {
            children.add(this._driveCollection);
        }

        return children;
    }

    public void setTreeName(String newName) {
        this.name = newName;
        firePropertyChange(prName, null, newName);
    }

    public String getTreeName() {
        return this.name;
    }

    /**
     * Go through all nodes in the tree and remove listener from all elements
     * it is registered to.
     *
     * @param listener listener we want to remove from the tree.
     */
    public synchronized void removeListenersFromTree(PoshElementListener listener) {
        getRootNode().removeListenersFromTree(this, listener);
    }

    /**
     * Remove listener <tt>listener</tt> from the tree with root node.
     */
    protected synchronized void removeListenersFromTree(PoshElement node, PoshElementListener listener) {
        node.removePoshTreeChangeListener(listener);

        for (PoshElement child : node.getChildDataNodes()) {
            removeListenersFromTree(child, listener);
        }
    }
    /*
    @Override
    protected Sheet createSheet() {
    Sheet sheet = Sheet.createDefault();
    Sheet.Set set = Sheet.createPropertiesSet();
    sheet.put(set);

    try {
    Property prop = new PropertySupport.Reflection<String>(
    this, String.class,
    "getTreeName",
    "setTreeName");

    prop.setName(prName);

    prop.setDisplayName("Name of POSH plan");
    set.put(prop);

    } catch (NoSuchMethodException ex) {
    Exceptions.printStackTrace(ex);
    }
    return sheet;
    }
     */

    @Override
    public String getDisplayName() {
        return this.getTreeName();
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        if (this._competences.contains(child)) {
            return moveNodeInList(_competences, child, relativePosition);
        }
        if (this._actionPatterns.contains(child)) {
            return moveNodeInList(_actionPatterns, child, relativePosition);
        }
        return false;
    }
    public static final DataFlavor dataFlavor = new DataFlavor(PoshPlan.class, "posh_tree_root");

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement newChild) throws ParseException {
        if (newChild instanceof DocString) {
            this.setDocString((DocString) newChild);
        } else if (newChild instanceof Competence) {
            this.addCompetence((Competence) newChild);
        } else if (newChild instanceof ActionPattern) {
            this.addActionPattern((ActionPattern) newChild);
        } else if (newChild instanceof DriveCollection) {
            this.setDriveCollection((DriveCollection) newChild);
        } else {
            throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
        }

    }

    @Override
    public void neutralizeChild(PoshElement childNode) {
        if (this._docstring == childNode) {
            this._docstring = null;
            childNode.remove();
        } else if (this._competences.contains(childNode)) {
            this._competences.remove(childNode);
            childNode.remove();
        } else if (this._actionPatterns.contains(childNode)) {
            this._actionPatterns.remove(childNode);
            childNode.remove();
        } else if (this._driveCollection == childNode) {
            this._driveCollection = null;
            childNode.remove();
        }
    }

    /**
     * Emit the tree using BFS.
     * <p>
     * For every element emit it as new one.
     * @param root
     */
    public void emitTree() {
        for (PoshElement pe : this.getChildDataNodes()) {
            this.emitChildNode(pe);
        }

        if (true) {
            return;
        }

        // TODO: Here is a dead code, but I anm keeping it for now, because of possible unforeseen effects of new code above
/*
        List<PoshElement> fringe = new LinkedList<PoshElement>();
        fringe.add(this);

        while (!fringe.isEmpty()) {
        // newFringe are all children of current fringe
        List<PoshElement> newFringe = new LinkedList<PoshElement>();

        for (PoshElement processedNode : fringe) {
        for (PoshElement childNode : processedNode.getChildDataNodes()) {
        processedNode.emitChildNode(childNode);
        newFringe.add(childNode);
        }
        }
        fringe = newFringe;
        }
         */
    }
    private Set<PoshElementListener> globalListeners = new HashSet<PoshElementListener>();

    /**
     * Notify global listeners that there has been a change in the tree.
     * @param event What kind of change
     * @param child Which node was source
     */
    synchronized void treeChanged(PoshTreeEvent event, PoshElement child) {
        for (PoshElementListener globalListener : globalListeners) {
            globalListener.nodeChanged(event, child);
        }
    }

    public synchronized void addGlobalTreeListener(PoshElementListener l) {
        assert l != null;
        globalListeners.add(l);
        this.addPropertyChangeListener(l);
    }

    public synchronized void removeGlobalTreeListener(PoshElementListener l) {
        assert l != null;
        globalListeners.remove(l);
        this.removePropertyChangeListener(l);
    }

    /**
     * Notify global listeners that there has been a property change in some elem,ent in the tree.
     * @param event What kind of property was changed, use getSource() to get source element.
     */
    synchronized void treePropertyChanged(PropertyChangeEvent event) {
        for (PoshElementListener globalListener : globalListeners) {
            globalListener.propertyChange(event);
        }
    }
}
