package cz.cuni.pogamut.shed.presenter;

import cz.cuni.amis.pogamut.sposh.elements.ActionPattern;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.DriveCollection;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.LapChain;
import cz.cuni.amis.pogamut.sposh.elements.LapPath;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.elements.Trigger;
import cz.cuni.amis.pogamut.sposh.elements.TriggeredAction;
import cz.cuni.pogamut.shed.widget.ExpandedActionEnvelope;
import cz.cuni.pogamut.shed.widget.ShedActionsEnvelope;
import cz.cuni.pogamut.shed.widget.ShedChoicesEnvelope;
import cz.cuni.pogamut.shed.widget.ShedSenseWidget;
import cz.cuni.pogamut.shed.widget.ShedTriggerEnvelope;
import cz.cuni.pogamut.shed.widget.ShedVariableWidget;

/**
 * Interface for factory creating the presenters for yaposh scene. Presenters
 * are responsible for updating the scene according to underlying changes of the
 * plan.
 *
 * TODO: Once done paths are checked, remove obsolete chain and elements
 * parameters from methods.
 *
 * Be careful when implementing the presenters, you can't store {@link LapPath}
 * in them, because stuff can move around.
 *
 * @author Honza
 */
public interface IPresenterFactory {

    /**
     * Create presenter for primitive action. The action doesn't have any
     * children.
     *
     * @param actionPath Path to the action in the plan
     * @param action Action the presenter is presenting.
     * @param actionWidget The widget that represents the @action in the scene.
     * @param actionChain Parameter chain to the action.
     * @return Presenter for action
     */
    IPresenter createActionPresenter(LapPath actionPath, TriggeredAction action, ShedVariableWidget actionWidget, LapChain actionChain);

    // TODO: Is this even used? I can't seem to find any reference. Check and possibly remove.
    /**
     * Create presenter for expanded action, i.e. the whole expanded structure
     * that can happen from unknown reference. This presenter must reflect that
     * @action can change the reference.
     *
     * @param action Action that was expnanded
     * @param envelope Envelope that contains expanded action in the scene
     * @param chain Parameter chain up to the branch node right before the
     * @action (@action is not included in the chain.).
     * @return Presenter for expanded action.
     */
    IPresenter createExpandedActionPresenter(TriggeredAction action, ExpandedActionEnvelope envelope, LapChain chain);

    /**
     * Create presenter for sense at @sensePath that will present the sense in
     * the @senseWidget.
     *
     * @param sensePath Path to the sense in the plan
     * @param sense Sense that is being presented
     * @param senseWidget Widget that represents the sense in the scene.
     * @param senseChain Parameter chain up to including sense.
     * @return Presenter of sense
     */
    IPresenter createSensePresenter(LapPath sensePath, Sense sense, ShedSenseWidget senseWidget, LapChain senseChain);

    /**
     * Create presenter for trigger, it is responsible for keepeing track of
     * added/moved/removed senses.
     *
     * @param <TRIGGER_PARENT> Owner of the trigger
     * @param parent The posh element this trigger belongs to
     * @param trigger Presented trigger
     * @param triggerEnvelope Envelope representing the trigger in the scene.
     * @param parentChain Parameter chain of the triggers owner
     * @return
     */
    <TRIGGER_PARENT extends PoshElement> IPresenter createTriggerPresenter(TRIGGER_PARENT parent, Trigger<TRIGGER_PARENT> trigger, ShedTriggerEnvelope triggerEnvelope, LapChain parentChain);

    /**
     * Create presenter for drive collection, responsible for keeping track of
     * added/moved/removed drives.
     *
     * @param driveCollectionPath Path to the DC
     * @param driveCollection Presented drive collection
     * @return
     */
    IPresenter createDriveCollectionPresenter(LapPath driveCollectionPath, DriveCollection driveCollection);

    /**
     * Create presenter for action pattern widget. This doesn't affect anything
     * else(APs actions...), only the actionPatternWidget. The presenter should
     * update widget when its properties, e.g.g parameters are changed.
     *
     * @param actionPatternPath Path to the action pattern, path ends with
     * <tt>../A:?/AP:?</tt>
     * @param referencingAction The action that referenced the action pattern
     * @param actionPattern Presented {@link ActionPattern}.
     * @param actionPatternWidget Widget representing the AP.
     * @param chain Parameter chain to AP (incl).
     * @return
     */
    IPresenter createActionPatternPresenter(LapPath actionPatternPath, TriggeredAction referencingAction, ActionPattern actionPattern, ShedVariableWidget actionPatternWidget, LapChain chain);

    /**
     * Create presenter for competence widget. The presenter is responsible only
     * for the widget of competence, not for its choices ect. It should for
     * example update the widget when arguments or parameters of link are
     * changed.
     *
     * @param competencePath Path to the competence, path ends with
     * <tt>../A:?/C:?</tt>
     * @param referencingAction Action referencing the competence.
     * @param competence Presented competence.
     * @param competenceWidget Widget representing the competence in the scene.
     * @param chain Parameter chain up to incl. competence.
     * @return
     */
    IPresenter createCompetencePresenter(LapPath competencePath, TriggeredAction referencingAction, Competence competence, ShedVariableWidget competenceWidget, LapChain chain);

    /**
     * Create presenter for choice widget, only the widget, not its trigger nor
     * action. Basically it is responsible only for showing changes of choice
     * name in the widget.
     *
     * @param choicePath Path to the choice, it will end with
     * <tt>../A:?/C:?/CE:?</tt>
     * @param choice Presented choice.
     * @param choiceWidget Widget representing the choice in the scene.
     * @return
     */
    IPresenter createChoicePresenter(LapPath choicePath, CompetenceElement choice, ShedVariableWidget choiceWidget);

    /**
     * Create presenter for drive widget. It reflects only changes of drive
     * widget, not its trigger or action. Overall it only reflects chganges of
     * name.
     *
     * @param drivePath Path to drive, <tt>../DE:?</tt>
     * @param drive Presented drive
     * @param widget Widget representing the drive in the scene.
     * @return
     */
    IPresenter createDrivePresenter(LapPath drivePath, DriveElement drive, ShedVariableWidget widget);

    /**
     * Create presenter that will take care about keeping adding/moving/removing
     * expanded actions in the {@link ShedActionsEnvelope} for {@link ActionPattern}.
     *
     * @param actionPatternPath Path to the action pattern, ends with
     * <tt>../AP:?</tt>.
     * @param actionPattern AP that contains the presented actions
     * @param actionsEnvelope Envelope that contains representations of the
     * actions.
     * @param actionPatternChain Chain to the @actionPattern of the actions.
     * @return
     */
    IPresenter createActionsPresenter(LapPath actionPatternPath, ActionPattern actionPattern, ShedActionsEnvelope actionsEnvelope, LapChain actionPatternChainchain);

    /**
     * Create presenter responsible for adding, moving and removing {@link ShedChoiceEnvelope choices widgets}
     * in the {@link ShedChoicesEnvelope}.
     *
     * @param competencePath Path to the competence whose choices are managed.
     * Ends with <tt>../A:?/C:?</tt>.
     * @param competence Competence that contains presented choices.
     * @param choicesEnvelope Envelope containing the representations of presented choices.
     * @param competenceChain Chain to the @competence.
     */
    IPresenter createChoicesPresenter(LapPath competencePath, Competence competence, ShedChoicesEnvelope choicesEnvelope, LapChain competenceChainchain);
}
