package cz.cuni.pogamut.posh.view;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import cz.cuni.amis.pogamut.sposh.elements.ParseException;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElementListener;
import cz.cuni.amis.pogamut.sposh.elements.PoshParser;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.pogamut.posh.PoshEditorSupport;
import cz.cuni.pogamut.posh.palette.APsChildren;
import cz.cuni.pogamut.posh.palette.PaletteRoot;
import cz.cuni.pogamut.posh.palette.PoshPaletteFactory;
import java.beans.PropertyChangeEvent;
import java.io.IOException;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.text.BadLocationException;
import org.netbeans.core.spi.multiview.CloseOperationState;
import org.netbeans.core.spi.multiview.MultiViewElementCallback;
import cz.cuni.pogamut.posh.widget.PoshScene;
import cz.cuni.pogamut.posh.widget.kidview.SimpleDriveCollectionWidget;
import javax.swing.JEditorPane;
import javax.swing.JToolBar;
import org.netbeans.core.spi.multiview.MultiViewElement;
import org.netbeans.spi.palette.PaletteController;
import org.openide.awt.UndoRedo;
import org.openide.text.NbDocument;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;
import org.openide.util.lookup.AbstractLookup;
import org.openide.util.lookup.InstanceContent;
import org.openide.util.lookup.ProxyLookup;

/**
 * Show POSH plan as tree-like structure, that is fully expanded (POSH plan starts
 * at drive collection, actions are connected to competences or APs is necessary)
 *
 * @author Honza Havlicek
 */
public class KidViewElement implements MultiViewElement {

    /**
     * This is a root node of posh tree that is shown in the component
     * If no tree, null
     */
    private PoshPlan plan;
    private PoshEditorSupport support;
    /**
     * When you switch from  text to source and back, while you change something
     * in the source, this will try to restore original collapse state of tree
     * even when the tree itsdelf is not the same.
     */
    private TreeCollapseImprint collapseImprint;
    /**
     * Scene we are placing the widgets into
     */
    private PoshScene scene;
    /**
     * The JComponent that is representing this MVE
     */
    private JScrollPane scrollPane;
    /**
     * Toolbar of this MVE, the one bext to buttons for switch between elements
     */
    private JToolBar toolbar;
    /**
     * Lookup, we only want to show palette for this element, not for source stuff
     */
    private AbstractLookup lookup;
    private InstanceContent ic;

    public KidViewElement(PoshEditorSupport support) {
        this.support = support;

        scene = new PoshScene();
        scrollPane = new JScrollPane();
        scrollPane.setViewportView(scene.createView());

        // Create a palette and add it to lookup
        ic = new InstanceContent();
        lookup = new AbstractLookup(ic);
        ic.add(PoshPaletteFactory.createPalette(getLookup()));
    }

    @Override
    public JComponent getVisualRepresentation() {
        return scrollPane;
    }

    private void refresh(PoshPlan root) throws ParseException {
        plan = root;

        scene.clearPoshWidgets();
        scene.setRootWidget(new SimpleDriveCollectionWidget(scene, root.getDriveCollection(), null));

        plan.emitTree();


        plan.addGlobalTreeListener(new SourceUpdater(plan, support));
    }

    private void refreshPalette() {

        PaletteController pc = getLookup().lookup(PaletteController.class);

        if (pc != null) {
            PaletteRoot palRoot = pc.getRoot().lookup(PaletteRoot.class);

            pc.clearSelection();

            plan.addGlobalTreeListener(palRoot.getAPsChildren());
            plan.addGlobalTreeListener(palRoot.getComptencesChildren());
            plan.addGlobalTreeListener(palRoot.getActionsChildren());
            plan.addGlobalTreeListener(palRoot.getSensesChildren());

            palRoot.getAPsChildren().refresh(plan);
            palRoot.getComptencesChildren().refresh(plan);
            palRoot.getActionsChildren().refresh(plan);
            palRoot.getSensesChildren().refresh(plan);
        }

    }

    @Override
    public JComponent getToolbarRepresentation() {
        if (toolbar == null) {
            toolbar = new JToolBar();
        }
        return toolbar;
    }

    @Override
    public Action[] getActions() {
        return support.getDataObject().getNodeDelegate().getActions(false);
    }

    @Override
    public Lookup getLookup() {
        return new ProxyLookup(support.getDataObject().getNodeDelegate().getLookup(), lookup);
    }

    @Override
    public void componentOpened() {
    }

    @Override
    public void componentClosed() {
    }

    @Override
    public void componentShowing() {
        regenerate();
    }

    @Override
    public void componentHidden() {
        if (scene.getRootWidget() != null) {
            collapseImprint = new TreeCollapseImprint(scene.getRootWidget());
        }
    }

    @Override
    public void componentActivated() {
        refreshPalette();
    }

    @Override
    public void componentDeactivated() {
    }

    @Override
    public UndoRedo getUndoRedo() {
        return null;
    }

    @Override
    public void setMultiViewCallback(MultiViewElementCallback mvec) {
    }

    @Override
    public CloseOperationState canCloseElement() {
        return CloseOperationState.STATE_OK;
    }

    /**
     * Regenrate the graph in the PoshScene according to posh plan in document.
     * If there is an syntax error, show error pane.
     */
    private void regenerate() {
        try {
            PoshParser parser = new PoshParser(support.getInputStream());
            PoshPlan newPlan = parser.parsePlan();
            scrollPane.setViewportView(scene.getView());
            refresh(newPlan);

            if (collapseImprint != null) {
                collapseImprint.restore(scene.getRootWidget());
            }
            collapseImprint = null;
        } catch (ParseException ex) {
            scrollPane.setViewportView(new ParseErrorPane(ex));
        } catch (Exception ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    /**
     * Updater of source. KidViewElement registers global listener to this class
     * and when tree changes, this updates document with posh plan.
     */
    private final static class SourceUpdater implements PoshElementListener, Runnable {

        private PoshEditorSupport support;
        private PoshPlan plan;

        private SourceUpdater(PoshPlan plan, PoshEditorSupport support) {
            this.plan = plan;
            this.support = support;
        }

        @Override
        public void nodeChanged(PoshTreeEvent event, PoshElement child) {
            NbDocument.runAtomic(support.getDocument(), this);
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            NbDocument.runAtomic(support.getDocument(), this);
        }

        @Override
        public void run() {
            try {
                String planString = plan.toString();
                support.getDocument().remove(0, support.getDocument().getLength());
                support.getDocument().insertString(0, planString, null);
                support.setModified();
            } catch (IOException ex) {
                Exceptions.printStackTrace(ex);
            } catch (BadLocationException ex) {
                Exceptions.printStackTrace(ex);
            }
        }
    }

    /**
     * Pane that is shown in case of syntax error during creation of syntax tree.
     */
    private static class ParseErrorPane extends JEditorPane {

        private ParseErrorPane(ParseException exception) {
            super("text/html",
                    "<html><head><style>"
                    + ".errorbox {"
                    + //"   width: 400px;" +
                    "}"
                    + ".boxtitle {"
                    + "	background-color: rgb(157, 173, 198);"
                    + "	text-align: center;"
                    + "}"
                    + ".boxtitle h2 {"
                    + "	margin: 0;"
                    + "	padding: 15px 30px 5px;"
                    + "	color: white;"
                    + "	font-weight: bold;"
                    + "	font-size: 1.5em;"
                    + "}"
                    + ".boxtext {"
                    + "	background-color: rgb(230, 230, 230);"
                    + "	padding: 5px 50px 31px;"
                    + "}"
                    + "</style>"
                    + "</head>"
                    + "<body>"
                    + "<div class=\"errorbox\">"
                    + "  <div class=\"boxtitle\"><h2>Syntax error</h2></div>"
                    + "  <div class=\"boxtext\">"
                    + "    <p>There is a problem in syntax of supplied posh plan. To remedy this situation, switch to source view and correct syntax error.</p>"
                    + "    <p>The syntax error will be marked by red exclamation mark at the line of error. Description of error will be in tooltip of the exclamation mark.</p>"
                    + "    <p><b>Error:</b> " + exception.getMessage().replace("<", "&lt;").replace(">", "&gt;") + "</p>"
                    + "  </div>"
                    + "</div>"
                    + "</body>"
                    + "</html>");
        }
    }
}
