/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.cuni.amis.pogamut.ut2004.ut2004testfw;

import cz.cuni.amis.pogamut.ut2004.ut2004testfw.config.BotTemplate;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.config.MatchConfig;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.measure.IMeasure;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.utils.CsvReader;
import cz.cuni.amis.pogamut.ut2004.ut2004testfw.utils.CsvReader.CsvRow;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.maven.doxia.sink.Sink;
import org.apache.maven.doxia.siterenderer.Renderer;
import org.apache.maven.project.MavenProject;
import org.apache.maven.reporting.AbstractMavenReport;
import org.apache.maven.reporting.MavenReportException;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.data.statistics.HistogramDataset;
import org.jfree.data.statistics.HistogramType;

/**
 *
 * @author tommasino
 * @goal performance-report
 * @phase site
 */
public class PerformanceReport extends AbstractMavenReport {

    /**
     * @parameter
     */
    private ArrayList<BotTemplate> bots;
    /**
     * @parameter
     */
    private ArrayList<MatchConfig> matchSets;
    /**
     * @parameter
     */
    private ArrayList<IMeasure> measures;
    /**
     * @parameter
     */
    private String utPath;
    /**
     * Directory where reports will go.
     *
     * @parameter expression="${project.reporting.outputDirectory}"
     * @required
     * @readonly
     */
    private String outputDirectory;
    /**
     * @parameter default-value="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;
    /**
     * @component
     * @required
     * @readonly
     */
    private Renderer siteRenderer;
    /**
     * A map with the tooltips.
     */
    private static HashMap<String, String> tooltips = new HashMap<String, String>();
    static {
        tooltips.put("match id", "A match id specifies the match set of the result. A match set is several matches that occur on the same map in the same conditions.");
        tooltips.put("test id", "A test runs all the match sets, a test id is an incrementing identification number of the test.");
        tooltips.put("run id", "A match set can contain several matches or runs. The run id distinguishes these matches.");
        tooltips.put("measure result", "A measure result is an aggregated result from one match, from all the observed bots. This could be for example an average frag count, or maximum deaths.");
        tooltips.put("percentile", "How much of the results in this dataset are smaller than this result.");
        tooltips.put("replay", "A replay of the match. You can use rk DemoWatcher to play these.");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Renderer getSiteRenderer() {
        return siteRenderer;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getOutputDirectory() {
        return outputDirectory;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected MavenProject getProject() {
        return project;
    }

    /**
     * Generates a histogram from the given data and saves it to the file specified by the param id.
     * @param id The histogram is saved to {getOuputDirectory()}/matches/graphs/{id}.png
     * @param measure X axis label.
     * @param name Title of the histogram.
     * @param series The data.
     * @return Returns a file with the graph.
     * @throws IOException
     */
    private File generateGraph(String id, String measure, String name, ArrayList<Double> series) throws IOException {
        HistogramDataset dataset = new HistogramDataset();
        dataset.setType(HistogramType.RELATIVE_FREQUENCY);
        double[] serie = new double[series.size()];
        for (int i = 0; i < serie.length; i++) {
            serie[i] = series.get(i);
        }
        dataset.addSeries("test", serie, serie.length > 20 ? 20 : serie.length);

        JFreeChart chart = ChartFactory.createHistogram(name, measure, "occurence",
                dataset, PlotOrientation.VERTICAL, false, true, false);
        File saveDir = new File(getOutputDirectory() + File.separator + "matches" + File.separator + "graphs" + File.separator);
        saveDir.mkdirs();
        File saveFile = new File(saveDir, id + ".png");
        ChartUtilities.saveChartAsPNG(saveFile, chart, 600, 400);
        return saveFile;
    }

    /**
     * Reads the results csv and places the data in a handy hashmap.
     * @return A map with the first key the measureName, and second key the matchId (mapname).
     * @throws FileNotFoundException
     * @throws IOException
     */
    private HashMap<String, HashMap<String, List<CsvRow>>> readResults() throws FileNotFoundException, IOException {
        HashMap<String, HashMap<String, List<CsvRow>>> data = new HashMap<String, HashMap<String, List<CsvRow>>>();
        CsvReader reader = new CsvReader(getOutputDirectory() + File.separator + "matches" + File.separator + "results", ";");
        CsvRow row = null;
        while ((row = reader.readRow()) != null) {
            String measureName = row.getString("measureName");
            String matchId = row.getString("matchId");
            if (!data.containsKey(measureName)) {
                data.put(measureName, new HashMap<String, List<CsvRow>>());
            }
            HashMap<String, List<CsvRow>> matchData = data.get(measureName);
            if (!matchData.containsKey(matchId)) {
                matchData.put(matchId, new ArrayList<CsvRow>());
            }
            matchData.get(matchId).add(row);
        }
        return data;
    }

    /**
     * Generates 2 levels of graphs, one for each pair of measure and matchId and then summaries for each measure.
     * @param data Data in a format from the readResults function.
     * @return A map with the generated graphs
     * @throws IOException
     */
    private HashMap<String, File> generateGraphs(HashMap<String, HashMap<String, List<CsvRow>>> data) throws IOException {
        HashMap<String, File> graphs = new HashMap<String, File>();
        for (String measureName : data.keySet()) {
            HashMap<String, List<CsvRow>> measureResults = data.get(measureName);
            ArrayList<Double> measureData = new ArrayList<Double>();
            for (String matchId : measureResults.keySet()) {
                ArrayList<Double> matchData = new ArrayList<Double>();
                for (CsvRow row : measureResults.get(matchId)) {
                    Double result = row.getDouble("measureResult");
                    matchData.add(result);
                    measureData.add(result);
                }
                String id = getId(measureName, matchId);
                File graph = generateGraph(id, measureName, matchId, matchData);
                if (graphs.containsKey(id)) {
                    Logger.getLogger(PerformanceReport.class.getName()).log(Level.WARNING, "Conflicts in measureName, matchName pairing. {0}.png overwritten.", id);
                }
                graphs.put(id, graph);
            }
            String id = getId(measureName, "all");
            File graph = generateGraph(id, measureName, "All matches", measureData);
            if (graphs.containsKey(id)) {
                Logger.getLogger(PerformanceReport.class.getName()).log(Level.WARNING, "Conflicts in measureName, matchName pairing. {0}.png overwritten.", id);
            }
            graphs.put(id, graph);
        }
        return graphs;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void executeReport(Locale locale) throws MavenReportException {
        ArrayList<String> names = new ArrayList<String>();
        int size = measures.size();
        for(int i = 0; i < size; ){
            IMeasure m = measures.get(i);
            if(m.getName() == null || m.getName().equals("")){
                Logger.getLogger(PerformanceReport.class.getName()).log(Level.WARNING, "Measure {0} doesnt have its name set, ommiting", m.getClass().getName());
                measures.remove(i);
                size--;
            }else if(names.contains(m.getName())){
                Logger.getLogger(PerformanceReport.class.getName()).log(Level.WARNING, "Duplicate measure name {0}", m.getName());
                measures.remove(i);
                size--;
            }else{
                i++;
                names.add(m.getName());
            }
        }
        File outputDir = new File(getOutputDirectory() + File.separator + "matches" + File.separator);
        if (!outputDir.exists()) {
            outputDir.mkdir();
        }
        String run = System.getProperty("uttest.runtests");
        boolean runtests = true;
        if(run!=null && run.equals("false"))
            runtests = false;
        if (runtests) {
            MatchesExecutor executor = new MatchesExecutor(matchSets, bots, measures, utPath, getOutputDirectory() + File.separator + "matches");
            executor.executeMatches();
        }
        try {
            writeReport();
        } catch (FileNotFoundException ex) {
            Logger.getLogger(PerformanceReport.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(PerformanceReport.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * Writes the report.
     * @throws FileNotFoundException
     * @throws IOException
     */
    private void writeReport() throws FileNotFoundException, IOException {
        //check whether the resources are installed
        File res = new File(getOutputDirectory() + File.separator + "matches" + File.separator + "res" + File.separator);
        if (!res.exists()) {
            if (!res.mkdirs()) {
                throw new IOException("The resource folder '" + res.getAbsolutePath() + "' could not be created");
            }
            copyResourceTo("/results.css", res);
            copyResourceTo("/display.js", res);
        }

        //generate the images
        HashMap<String, HashMap<String, List<CsvRow>>> data = readResults();
        HashMap<String, File> graphs = generateGraphs(data);

        //generate html
        Sink sink = getSink();
        sink.head();
        sink.title();
        sink.text("Bot Performance");
        sink.title_();
        sink.head_();
        sink.body();
        sink.rawText("<link rel=\"stylesheet\" href=\"./matches/res/results.css\" type=\"text/css\" />");
        sink.rawText("<script type=\"text/javascript\" src=\"./matches/res/display.js\"></script>");
        sink.section1();

        sink.sectionTitle1();
        sink.text("Bot Performance");
        sink.sectionTitle1_();

        boolean first = true;
        sink.rawText("<div id=\"menu\">");
        for (IMeasure measure : measures) {
            sink.rawText("<span class=\"menuMain\">");

            String id = getId(measure.getName(), "all");
            sink.rawText("<a class=\"picker" + (first ? " selected" : "") + "\" id=\"" + measure.getName() + "Picker\" href=\"javascript:;\" onclick=\"displayGroup('" + id + "')\" onmouseover=\"display('" + measure.getName() + "Pick')\" onmouseout=\"hide('" + measure.getName() + "Pick')\">" + measure.getName() + "</a>");
            sink.rawText("<div class=\"menuPick\" id=\"" + measure.getName() + "Pick\" onmouseover=\"interruptHide()\" onmouseout=\"hide('" + measure.getName() + "Pick')\">");
            boolean odd = false;
            for (MatchConfig config : matchSets) {
                id = getId(measure.getName(), config.getId());
                sink.rawText("<div class=\"menuItem" + (odd ? " odd" : "") + "\"><a href=\"javascript:;\" onclick=\"displayGroup('" + id + "')\">" + config.getId() + "</a></div>");
                odd = !odd;
            }
            sink.rawText("</div>");

            sink.rawText("</span>");

            first = false;
        }
        sink.rawText("</div>");

        sink.rawText("<div id=\"graphBox\">");
        
        generateMain();
        
        for (IMeasure measure : measures) {
            List<CsvRow> all = new ArrayList<CsvRow>();
            for (MatchConfig match : matchSets) {
                String id = getId(measure.getName(), match.getId());
                File g = graphs.get(id);
                HashMap<String, List<CsvRow>> get = data.get(measure.getName());
                List<CsvRow> d;
                if(get != null)
                    d = get.get(match.getId());
                else
                    d = new ArrayList<CsvRow>();
                generateMeasure(measure, match, id, g, false, d);
                all.addAll(d);
            }
            String id = getId(measure.getName(), "all");
            File g = graphs.get(id);
            generateMeasure(measure, null, id, g, false, all);
        }

        sink.rawText("</div>");

        sink.section1_();
        sink.body_();
        sink.flush();
        sink.close();
    }
    
    private void generateMain(){
        Sink sink = getSink();
        sink.rawText("<div class=\"group\" id=\"default\">");
        sink.rawText("<h3>Available reports</h3>");
        sink.list();
        for(IMeasure measure : measures){
            sink.listItem();
            String id = getId(measure.getName(), "all");
            sink.rawText("<a href=\"javascript:;\" onclick=\"displayGroup('" + id + "')\"\">" + measure.getName() + ": </a>");
            sink.text(measure.getDescription());
            sink.listItem_();
        }
        sink.list_();
        sink.rawText("</div>");
    }

    /**
     * Generates a single measure-match report.
     * @param measure The measure of the report.
     * @param match The match of the report, can be null, will then generate a general report for the specified measure.
     * @param id Group id.
     * @param chart File with the histogram.
     * @param visible Specifies if this report is visible by default.
     * @param data The report data.
     */
    private void generateMeasure(IMeasure measure, MatchConfig match, String id, File chart, boolean visible, List<CsvRow> data) {
        Sink sink = getSink();
        sink.rawText("<div class=\"group\" id=\"" + id + "\" style=\"visibility:" + (visible ? "visible" : "hidden;display:none") + ";\">");
        sink.rawText("<h3>" + measure.getName() + (match == null ? "" : "-" + match.getId()) + "</h3>");
        if(chart == null)
            sink.rawText("No chart yet.");
        else
            sink.rawText("<img src=\"" + "matches" + File.separator + "graphs" + File.separator + chart.getName() + "\" />");
        List<CsvRow> results = new ArrayList<CsvRow>();
        List<CsvRow> lastResults = new ArrayList<CsvRow>();
        int max = 0;
        for (CsvRow row : data) {
            int testId = row.getInt("testId");
            if (testId > max) {
                lastResults.clear();
                max = testId;
            }
            if (testId == max) {
                lastResults.add(row);
            }
            results.add(row);
        }

        sink.rawText("<h3>Last test results</h3>");
        sink.table();
        if (match == null) {
            printHeader(sink, "match id", "measure result", "percentile", "replay");
        } else {
            printHeader(sink, "measure result", "percentile", "replay");
        }
        DecimalFormat format = new DecimalFormat("#.#'%'");
        for (CsvRow result : lastResults) {
            double percentile = calculatePercentile(result, results);
            String replay = result.getString("matchId") + "-" + result.getString("runId") + "-replay.demo4";
            //#TODO missing replay
            if (match == null) {
                printRow(sink, result.getString("matchId"), result.getDouble("measureResult"), format.format(percentile), "<a href=\"matches/" + replay + "\">Download replay</a>");
            } else {
                printRow(sink, result.getDouble("measureResult"), format.format(percentile), "<a href=\"matches/" + replay + "\">Download replay</a>");
            }
        }
        sink.table_();


        sink.rawText("<h3>Testing history</h3>");
        sink.table();
        if (match == null) {
            printHeader(sink, "match id", "test id", "run id", "measure result");
        } else {
            printHeader(sink, "test id", "run id", "measure result");
        }

        Collections.sort(results, new Comparator<CsvRow>() {

            @Override
            public int compare(CsvRow o1, CsvRow o2) {
                Integer i1 = o1.getInt("testId");
                Integer r1 = o1.getInt("runId");
                String n1 = o1.getString("matchId");
                Integer i2 = o2.getInt("testId");
                Integer r2 = o2.getInt("runId");
                String n2 = o2.getString("matchId");
                int compare = i1.compareTo(i2);
                if (compare != 0) {
                    return compare;
                }
                compare = n1.compareTo(n2);
                if (compare != 0) {
                    return compare;
                }
                return r1.compareTo(r2);
            }
        });
        for (CsvRow row : results) {
            if (match == null) {
                printRow(sink, row.getString("matchId"), row.getString("testId"), row.getString("runId"), row.getString("measureResult"));
            } else {
                printRow(sink, row.getString("testId"), row.getString("runId"), row.getString("measureResult"));
            }
        }

        sink.table_();
        sink.rawText("</div>");
    }

    /**
     *
     * @param cells
     */
    private static void printRow(Sink sink, Object... cells) {
        sink.tableRow();
        for (Object cell : cells) {
            sink.tableCell();
            sink.rawText(cell.toString());
            sink.tableCell_();
        }
        sink.tableRow_();
    }

    /**
     *
     * @param sink
     * @param cells
     */
    private static void printHeader(Sink sink, Object... cells) {
        sink.tableRow();
        for (Object cell : cells) {
            String tooltip = null;
            if (tooltips.containsKey(cell.toString())) {
                tooltip = tooltips.get(cell.toString());
            }
            if (tooltip != null) {
                sink.rawText("<th onmouseout=\"tooltip.hide()\" onmouseover=\"tooltip.show('" + tooltip.replaceAll("'", "\\'") + "', 300)\">");
            } else {
                sink.tableHeaderCell();
            }
            sink.rawText(cell.toString());
            sink.tableHeaderCell_();
        }
        sink.tableRow_();
    }

    /**
     *
     * @return
     */
    @Override
    public String getOutputName() {
        return "performance-report";
    }

    /**
     *
     * @param locale
     * @return
     */
    @Override
    public String getName(Locale locale) {
        return getBundle(locale).getString("report.performance-report.name");

    }

    /**
     *
     * @param locale
     * @return
     */
    @Override
    public String getDescription(Locale locale) {
        return getBundle(locale).getString("report.performance-report.description");
    }

    /**
     *
     * @param locale
     * @return
     */
    private ResourceBundle getBundle(Locale locale) {
        return ResourceBundle.getBundle("performance-report", locale, this.getClass().getClassLoader());
    }

    /**
     *
     * @param resource
     * @param dir
     * @throws FileNotFoundException
     * @throws IOException
     */
    private void copyResourceTo(String resource, File dir) throws FileNotFoundException, IOException {
        PerformanceReport.class.getResourceAsStream(resource);
        InputStream fis = PerformanceReport.class.getResourceAsStream(resource);
        FileOutputStream fos = new FileOutputStream(new File(dir, resource));
        try {
            byte[] buf = new byte[1024];
            int i = 0;
            while ((i = fis.read(buf)) != -1) {
                fos.write(buf, 0, i);
            }
        } catch (IOException e) {
            throw e;
        } finally {
            if (fis != null) {
                fis.close();
            }
            if (fos != null) {
                fos.close();
            }
        }
    }

    /**
     *
     * @param measureName
     * @param matchId
     * @return
     */
    public static String getId(String measureName, String matchId) {
        return measureName.replaceAll(" ", "") + '-' + matchId.replaceAll(" ", "");
    }

    /**
     *
     * @param row
     * @param data
     * @return
     */
    private double calculatePercentile(CsvRow row, List<CsvRow> data) {
        double count = 0;
        double result = row.getDouble("measureResult");
        for (CsvRow r : data) {
            if (r.getDouble("measureResult") <= result) {
                count++;
            }
        }
        return count / data.size() * 100;
    }
}
