package cz.cuni.amis.pogamut.ut3.communication.messages;

/*
 * Copyright (C) 2013 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.utils.maps.HashMapSet;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * 
 * @author Evers
 */
public class UT3ItemType extends ItemType {

	/**
	 * 
	 */
	private static final long serialVersionUID = 6128208226173615461L;

	/**
	 * Map of all registered ItemType prototypes.
	 */
	protected static Map<String, ItemType> protos = new HashMap<String, ItemType>();

	/**
	 * Contains item types that belongs to their groups.
	 */
	public static final HashMapSet<UT3Group, ItemType> GROUPS = new HashMapSet<UT3Group, ItemType>();

	/**
	 * Category of this item.
	 */
	private Category category;

	/**
	 * Group of this item.
	 */
	private UT3Group group;

	/**
	 * Human readable name of the item.
	 */
	private String name;

	public static UT3ItemType getWeaponForAmmo(ItemType itemType) {
		if (itemType == null)
			return UT3ItemType.NONE;
		Set<ItemType> mySet = GROUPS.get(itemType.getGroup());
		if (mySet != null) {
			Iterator<ItemType> myIterator = mySet.iterator();
			while (myIterator.hasNext()) {
				ItemType type = myIterator.next();
				if (type.getGroup() == itemType.getGroup()
						&& type.getCategory() == Category.WEAPON)
					return (UT3ItemType) type;
			}
		}

		return UT3ItemType.NONE;
	}

	public static UT3ItemType getAmmoForWeapon(ItemType itemType) {
		if (itemType == null)
			return UT3ItemType.NONE;
		Set<ItemType> mySet = GROUPS.get(itemType.getGroup());
		if (mySet != null) {
			Iterator<ItemType> myIterator = mySet.iterator();
			while (myIterator.hasNext()) {
				ItemType type = myIterator.next();
				if (type.getGroup() == itemType.getGroup()
						&& type.getCategory() == Category.AMMO)
					return (UT3ItemType) type;
			}
		}

		return UT3ItemType.NONE;
	}

	@Override
	public String toString() {
		return "ItemType[name = " + name + ", category = " + category
				+ ", group = " + group + "]";
	}

	@Override
	public Category getCategory() {
		return category;
	}

	@Override
	public UT3Group getGroup() {
		return group;
	}

	@Override
	public boolean equals(Object obj) {
		// the same object?
		if (this == obj)
			return true;

		// the same type?
		if (obj instanceof UT3ItemType) {
			// same value
			if ((category == ((UT3ItemType) obj).getCategory())
					&& (group == ((UT3ItemType) obj).getGroup())
					&& (name == ((UT3ItemType) obj).getName()))
				return true;
		}

		return false;
	}

	@Override
	public int hashCode() {
		return getName().hashCode();
	}

	@Override
	public int compareTo(ItemType o) {
		if (o == null)
			return 1;

		if (getName() == null) {
			if (o.getName() == null)
				return 0;
			return 1;
		} else {
			if (o.getName() == null)
				return -1;
			return getName().compareTo(o.getName());
		}
	}

	/**
	 * List of all item groups. Groups fine down the categories into specific
	 * groups, based on what the item belongs to. Also, groups join items from
	 * different categories together, if they belong together (e.g. weapon with
	 * its ammo).
	 */
	public enum UT3Group implements Group {
		/** Translocating weapon and accessory. */
		TRANSLOCATOR("Translocator"),
		/** ShieldGun weapon and accessory. */
		IMPACT_HAMMER("ImpactHammer"),
		/** AssaultRifle weapon and accessory. */
		ENFORCER("Enforcer"),
		/** BioRifle weapon and accessory. */
		BIO_RIFLE("BioRifle"),
		/** ShockRifle weapon and accessory. */
		SHOCK_RIFLE("ShockRifle"),
		/** LinkGun weapon and accessory. */
		LINK_GUN("LinkGun"),
		/** Minigun weapon and accessory. */
		STINGER_MINIGUN("Stinger"),
		/** FlakCannon weapon and accessory. */
		FLAK_CANNON("FlakCannon"),
		/** RocketLauncher weapon and accessory. */
		ROCKET_LAUNCHER("RocketLauncher"),
		/** Avril weapon and accessory. */
		AVRIL("Avril"),
		/** SniperRifle weapon and accessory. */
		SNIPER_RIFLE("SniperRifle"),
		/** Redeemer */
		REDEEMER("Redeemer"),
		/** Weaponlocker */
		WEAPON_LOCKER("WeaponLocker"),
		/** SlowField */
		SLOW_FIELD("SlowField"),

		/** Classic health pack. */
		MEDIUM_HEALTH("MediumHealth"),
		/** Mini health vial. */
		HEALTH_VIAL("HealthVial"),
		/** Big health recharger. */
		SUPER_HEALTH("SuperHealth"),

		/** Armor parts. */
		ARMOR_HELMET("Helmet"), ARMOR_VEST("Vest"), ARMOR_THIGHPADS("Thighpads"), ARMOR_SHIELD_BELT(
				"ShieldBelt"),

		/** POWERUPS **/
		JUMP_BOOTS("JumpBoots"),
		/** UDamage bonus items. */
		UDAMAGE("UDamage"), BERSERK("Berserk"), INVULNERABILITY(
				"Invulnerability"), INVISIBILITY("Invisibility"),

		/** Keys. */
		KEY("Key"),
		/** Other items with user-defined group. */
		OTHER("Unknown"),
		/** DEPLOYABLES **/
		XRAY_VOLUME("XRayVolume"), SPIDERMINE_TRAP("SpiderMineTrap"), ENERGY_SHIELD(
				"EnergyShield"), EMP_MINE("EMPMine"), LINK_GENERATOR(
				"LinkGenerator"), SHAPED_CHARGE("ShapedCharge"), SLOW_VOLUME(
				"SlowVolume"),

		/** No group, used for the prototype None */
		NONE("None");

		/* =================================================================== */

		/** Human-readable name of the group. */
		public String name;

		/* =================================================================== */

		/**
		 * Constructor.
		 * 
		 * @param name
		 *            Human-readable name of the group.
		 */
		UT3Group(String name) {
			this.name = name;
		}

		@Override
		public Set<ItemType> getTypes() {
			return GROUPS.get(this);
		}

		@Override
		public String getName() {
			return this.name;
		}
	}

	/* ======================================================================== */

        // TODO: Only 2 parameters actually needed, the first one is the name (Enforcer, ImpactHammer, etc.)
        // and the second one the pickup-type (ShockRifle.WeaponPickup, BioRifle.WeaponPickup), unfortunately 
        // GBUT3 is not always consistent in sending the right type, so until GBUT3 is completely done, some
        // extra UT3 names are needed
        
        // First parameter added to support spawning items from this type in the control connection
        
	/** Translocator. */
	public static final UT3ItemType TRANSLOCATOR = MakePrototype(
			Category.WEAPON, UT3Group.TRANSLOCATOR, new String[] {"UTGame.UTWeap_Translocator_Content",
					"Translocator", "Translocator.WeaponPickup" });

	/** Translocator Beacon. */
	public static final UT3ItemType TRANSLOCATOR_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.TRANSLOCATOR, new String[] {
					"UTProj_TransDisc", "UT_Proj_TransDisc_ContentBlue",
					"UT_Proj_TransDisc_ContentRed" });

	/** ImpactHammer weapon. */
	public static final UT3ItemType IMPACT_HAMMER = MakePrototype(
			Category.WEAPON, UT3Group.IMPACT_HAMMER, new String[] { "UTGame.UTWeap_ImpactHammer",
					"ImpactHammer", "ImpactHammer.WeaponPickup" });

	/** ImpactHammer ammo - sent when the bot is spawned. */
	public static final UT3ItemType IMPACT_HAMMER_AMMO = MakePrototype(
			Category.AMMO, UT3Group.IMPACT_HAMMER, new String[] {
					"ImpactHammerAmmo", "ImpactHammerAmmo.AmmoPickup" });

	/** Enforcer weapon. */
	public static final UT3ItemType ENFORCER = MakePrototype(Category.WEAPON,
			UT3Group.ENFORCER, new String[] { "UTGame.UTWeap_Enforcer", "Enforcer",
					"Enforcer.WeaponPickup" });
	/** Enforcer ammo. */
	public static final UT3ItemType ENFORCER_AMMO = MakePrototype(
			Category.AMMO, UT3Group.ENFORCER, new String[] { "GameBotsUT3.GBAmmo_Enforcer", "EnforcerAmmo",
					"EnforcerAmmo.AmmoPickup" });

	/** BioRifle weapon. */
	public static final UT3ItemType BIO_RIFLE = MakePrototype(Category.WEAPON,
			UT3Group.BIO_RIFLE, new String[] { "UTGameContent.UTWeap_BioRifle_Content", "BioRifle",
					"BioRifle.WeaponPickup" });
	/** BioRifle ammo. */
	public static final UT3ItemType BIO_RIFLE_AMMO = MakePrototype(
			Category.AMMO, UT3Group.BIO_RIFLE, new String[] { "GameBotsUT3.GBAmmo_BioRifle_Content", "BioRifleAmmo",
					"BioRifleAmmo.AmmoPickup" });

	/** BioRifle projectile. */
	public static final UT3ItemType BIO_RIFLE_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.BIO_RIFLE, new String[] {
					"UTProj_BioShot", "UTProj_BioGlob", "UTProj_BioGlobling" });

	/** ShockRifle weapon. */
	public static final UT3ItemType SHOCK_RIFLE = MakePrototype(
			Category.WEAPON, UT3Group.SHOCK_RIFLE, new String[] { "UTGame.UTWeap_ShockRifle", "ShockRifle",
					"ShockRifle.WeaponPickup" });
	/** ShockRifle ammo. */
	public static final UT3ItemType SHOCK_RIFLE_AMMO = MakePrototype(
			Category.AMMO, UT3Group.SHOCK_RIFLE, new String[] { "GameBotsUT3.GBAmmo_ShockRifle",
					"ShockRifleAmmo", "ShockRifleAmmo.AmmoPickup" });

	/** ShockRifle projectile. */
	public static final UT3ItemType SHOCK_RIFLE_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.SHOCK_RIFLE, new String[] {
					"UTProj_ShockBall", "UTProj_HeroShockBall" });

	/** LinkGun weapon. */
	public static final UT3ItemType LINK_GUN = MakePrototype(Category.WEAPON,
			UT3Group.LINK_GUN, new String[] { "UTGame.UTWeap_LinkGun", "LinkGun",
					"LinkGun.WeaponPickup" });
	/** LinkGun ammo. */
	public static final UT3ItemType LINK_GUN_AMMO = MakePrototype(
			Category.AMMO, UT3Group.LINK_GUN, new String[] { "GameBotsUT3.GBAmmo_LinkGun", "LinkGunAmmo",
					"LinkGunAmmo.AmmoPickup" });

	/** LinkGun projectile. */
	public static final UT3ItemType LINK_GUN_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.LINK_GUN,
			new String[] { "XWeapons.LinkProjectile" });

	/** Minigun weapon. */
	public static final UT3ItemType STINGER_MINIGUN = MakePrototype(
			Category.WEAPON, UT3Group.STINGER_MINIGUN, new String[] { "UTGame.UTWeap_Stinger",
					"Stinger", "Stinger.WeaponPickup" });
	/** Minigun ammo. */
	public static final UT3ItemType STINGER_MINIGUN_AMMO = MakePrototype(
			Category.AMMO, UT3Group.STINGER_MINIGUN, new String[] { "GameBotsUT3.GBAmmo_Stinger",
					"StingerAmmo", "StingerAmmo.AmmoPickup" });

	public static final UT3ItemType STINGER_MINIGUN_SHARD = MakePrototype(
			Category.PROJECTILE, UT3Group.STINGER_MINIGUN,
			new String[] { "UTProj_StingerShard" });

	/** FlakCannon weapon. */
	public static final UT3ItemType FLAK_CANNON = MakePrototype(
			Category.WEAPON, UT3Group.FLAK_CANNON, new String[] { "UTGame.UTWeap_FlakCannon", "FlakCannon",
					"FlakCannon.WeaponPickup" });
	/** FlakCannon ammo. */
	public static final UT3ItemType FLAK_CANNON_AMMO = MakePrototype(
			Category.AMMO, UT3Group.FLAK_CANNON, new String[] { "GameBotsUT3.GBAmmo_FlakCannon",
					"FlakCannonAmmo", "FlakCannonAmmo.AmmoPickup" });

	/** FlakCannon chunk projectile. */
	public static final UT3ItemType FLAK_CANNON_CHUNK = MakePrototype(
			Category.PROJECTILE, UT3Group.FLAK_CANNON, new String[] {
					"UTProj_FlakShard", "UTProj_FlakShardMain" });

	/** FlakCannon shell projectile. */
	public static final UT3ItemType FLAK_CANNON_SHELL = MakePrototype(
			Category.PROJECTILE, UT3Group.FLAK_CANNON,
			new String[] { "UTProj_FlakShell" });
        
        /** AVRIL weapon */
	public static final UT3ItemType AVRIL = MakePrototype(Category.WEAPON,
			UT3Group.AVRIL, new String[] { "UTGameContent.UTWeap_Avril_Content", "Avril",
					"Avril.WeaponPickup" });
	/** AVRIL ammo */
	public static final UT3ItemType AVRIL_AMMO = MakePrototype(
			Category.AMMO,
			UT3Group.AVRIL,
			new String[] { "GameBotsUT3.GBAmmo_AVRiL", "AVRiLAmmo",
					"AVRiLAmmo.AmmoPickup" });

	/** RocketLauncher weapon. */
	public static final UT3ItemType ROCKET_LAUNCHER = MakePrototype(
			Category.WEAPON, UT3Group.ROCKET_LAUNCHER, new String[] { "UTGame.UTWeap_RocketLauncher",
					"RocketLauncher", "RocketLauncher.WeaponPickup" });
	/** RocketLauncher ammo. */
	public static final UT3ItemType ROCKET_LAUNCHER_AMMO = MakePrototype(
			Category.AMMO, UT3Group.ROCKET_LAUNCHER, new String[] { "GameBotsUT3.RocketLauncher",
					"RocketLauncherAmmo", "RocketLauncherAmmo.AmmoPickup" });

	/** RocketLauncher projectile. */
	public static final UT3ItemType ROCKET_LAUNCHER_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.ROCKET_LAUNCHER, new String[] {
					"UTProj_Rocket", "UTProj_HeroRocket",
					"UTProj_LoadedRocket", "UTProj_SeekingRocket" });

	/** SniperRifle weapon (classic sniper weapon). */
	public static final UT3ItemType SNIPER_RIFLE = MakePrototype(
			Category.WEAPON, UT3Group.SNIPER_RIFLE, new String[] { "UTGame.UTWeap_SniperRifle",
					"SniperRifle", "SniperRifle.WeaponPickup" });
	/** SniperRifle ammo. */
	public static final UT3ItemType SNIPER_RIFLE_AMMO = MakePrototype(
			Category.AMMO, UT3Group.SNIPER_RIFLE, new String[] { "GameBotsUT3.GBAmmo_SniperRifle",
					"SniperRifleAmmo", "SniperRifleAmmo.AmmoPickup" });

	/** SniperRifle weapon (classic sniper weapon). */
	public static final UT3ItemType REDEEMER = MakePrototype(Category.WEAPON,
			UT3Group.REDEEMER, new String[] { "UTGameContent.UTWeap_Redeemer_Content", "Redeemer",
					"Redeemer.WeaponPickup" });

	public static final UT3ItemType REDEEMER_AMMO = MakePrototype(
			Category.AMMO, UT3Group.REDEEMER, new String[] { "UTGameContent.UTAmmo_Redeemer_Content", "RedeemerAmmo",
					"RedeemerAmmo.AmmoPickup" });

	public static final UT3ItemType REDEEMER_PROJECTILE = MakePrototype(
			Category.PROJECTILE, UT3Group.REDEEMER, new String[] { 
					"UTProj_Redeemer", "UTProj_RedeemerRed" });

	/** Health kit. */
	public static final UT3ItemType MEDIUM_HEALTH = MakePrototype(
			Category.HEALTH, UT3Group.MEDIUM_HEALTH, new String[] { "UTGame.UTPickup_MediumHealth",
					"MediumHealth", "MediumHealth.HealthPickup" });
	/** Health vial. */
	public static final UT3ItemType HEALTH_VIAL = MakePrototype(
			Category.HEALTH, UT3Group.HEALTH_VIAL, new String[] { "UTGame.UTPickup_HealthVial", "HealthVial",
					"HealthVial.HealthPickup" });
	/** SuperHealth charger. */
	public static final UT3ItemType SUPER_HEALTH_PACK = MakePrototype(
			Category.HEALTH, UT3Group.SUPER_HEALTH, new String[] { "UTGame.UTPickup_SuperHealth",
					"SuperHealth", "SuperHealth.HealthPickup" });

	/** Helmet. */
	public static final UT3ItemType ARMOR_HELMET = MakePrototype(
			Category.ARMOR, UT3Group.ARMOR_HELMET, new String[] { "UTGame.UTArmorHelmet", "Helmet",
					"Helmet.ArmorPickup" });
	/** Shieldbelt. */
	public static final UT3ItemType ARMOR_SHIELD_BELT = MakePrototype(
			Category.ARMOR, UT3Group.ARMOR_SHIELD_BELT, new String[] { "UTGame.UTArmor_ShieldBelt",
					"ShieldBelt", "ShieldBelt.ArmorPickup" });
	/** Thighpads. */
	public static final UT3ItemType ARMOR_THIGHPADS = MakePrototype(
			Category.ARMOR, UT3Group.ARMOR_THIGHPADS, new String[] { "UTGame.UTArmor_Thighpads",
					"Thighpads", "Thighpads.ArmorPickup" });
	/** Vest. */
	public static final UT3ItemType ARMOR_VEST = MakePrototype(Category.ARMOR,
			UT3Group.ARMOR_VEST, new String[] { "UTGame.UTArmor_Vest", "Vest", "Vest.ArmorPickup" });

	/** UDamage bonus (damage multiplier). */
	public static final UT3ItemType U_DAMAGE_PACK = MakePrototype(
			Category.OTHER, UT3Group.UDAMAGE, new String[] { "UTGameContent.UTUDamage", "UDamage",
					"UDamage.Pickup" });

	public static final UT3ItemType BERSERK_PACK = MakePrototype(
			Category.OTHER, UT3Group.BERSERK, new String[] { "UTGameContent.UTBerserk", "Berserk",
					"Berserk.Pickup" });

	public static final UT3ItemType JUMP_BOOTS = MakePrototype(Category.OTHER,
			UT3Group.JUMP_BOOTS, new String[] { "UTGameContent.UTJumpBoots", "JumpBoots",
					"JumpBoots.Pickup" });
        
        // @NOTE: The volume_ammo's are needed for correct functioning of the weaponry

	/** XRay Volume **/
	public static final UT3ItemType XRAY_VOLUME = MakePrototype(
			Category.WEAPON, UT3Group.XRAY_VOLUME, new String[] { "UT3Gold.UTDeployableXRayVolume", "UTDeployableXRayVolume",
					"UTDeployableXRayVolume.WeaponPickup" });

	/** XRay Ammo **/
	public static final UT3ItemType XRAY_VOLUME_AMMO = MakePrototype(
			Category.AMMO, UT3Group.XRAY_VOLUME, new String[] {
					"UTDeployableXRayVolumeAmmo", "UTDeployableXRayVolume.AmmoPickup" });

	/** SpiderMine Trap **/
	public static final UT3ItemType SPIDERMINE_TRAP = MakePrototype(
			Category.WEAPON, UT3Group.SPIDERMINE_TRAP, new String[] { "UTGameContent.UTDeployableSpiderMineTrap",
					"UTDeployableSpiderMineTrap", "UTDeployableSpiderMineTrap.WeaponPickup" });

	/** SpiderMine Trap Ammo **/
	public static final UT3ItemType SPIDERMINE_TRAP_AMMO = MakePrototype(
			Category.AMMO, UT3Group.SPIDERMINE_TRAP, new String[] {
					"UTDeployableSpiderMineTrapAmmo", "UTDeployableSpiderMineTrap.AmmoPickup" });

	/** EnergyShield **/
	public static final UT3ItemType ENERGY_SHIELD = MakePrototype(
			Category.WEAPON, UT3Group.ENERGY_SHIELD, new String[] { "UTGameContent.UTDeployableEnergyShield",
					"UTDeployableEnergyShield", "UTDeployableEnergyShield.WeaponPickup" });

	/** EnergyShield Ammo **/
	public static final UT3ItemType ENERGY_SHIELD_AMMO = MakePrototype(
			Category.AMMO, UT3Group.ENERGY_SHIELD, new String[] {
					"UTDeployableEnergyShieldAmmo", "UTDeployableEnergyShield.AmmoPickup" });

	/** EMP Mine **/
	public static final UT3ItemType EMP_MINE = MakePrototype(Category.WEAPON,
			UT3Group.EMP_MINE, new String[] { "EMPMine",
					"EMPMine.WeaponPickup", "UTDeployableEMPMine" });

	/** EMP Mine Ammo **/
	public static final UT3ItemType EMP_MINE_AMMO = MakePrototype(
			Category.AMMO, UT3Group.EMP_MINE, new String[] { "UTGameContent.UTDeployableEMPMine", 
                                "UTDeployableEMPMineAmmo", "UTDeployableEMPMine.AmmoPickup" });

	/** Link generator **/
	public static final UT3ItemType LINK_GENERATOR = MakePrototype(
			Category.WEAPON, UT3Group.LINK_GENERATOR, new String[] { "UTGameContent.UTDeployableLinkGenerator",
					"UTDeployableLinkGenerator", "UTDeployableLinkGenerator.WeaponPickup" });

	/** Link generator Ammo **/
	public static final UT3ItemType LINK_GENERATOR_AMMO = MakePrototype(
			Category.AMMO, UT3Group.LINK_GENERATOR, new String[] {
					"UTDeployableLinkGeneratorAmmo", "UTDeployableLinkGenerator.AmmoPickup" });

	/** ShapedCharge **/
	public static final UT3ItemType SHAPED_CHARGE = MakePrototype(
			Category.WEAPON, UT3Group.SHAPED_CHARGE, new String[] { "UTGameContent.UTDeployableShapedCharge",
					"UTDeployableShapedCharge", "UTDeployableShapedCharge.WeaponPickup" });

	/** ShapedCharge Ammo **/
	public static final UT3ItemType SHAPED_CHARGE_AMMO = MakePrototype(
			Category.AMMO, UT3Group.SHAPED_CHARGE, new String[] {
					"UTDeployableShapedChargeAmmo", "ShapedCharge.AmmoPickup" });

	/** Slow Volume **/
	public static final UT3ItemType SLOW_VOLUME = MakePrototype(
			Category.WEAPON, UT3Group.SLOW_VOLUME, new String[] { "UTGameContent.UTDeployableSlowVolume",
                                "UTDeployableSlowVolume", "UTDeployableSlowVolume.WeaponPickup" });
        
        /** Slow Volume content, this is the field itself **/
        public static final UT3ItemType SLOW_VOLUME_CONTENT = MakePrototype (
                        Category.OTHER, UT3Group.SLOW_VOLUME, new String[] { "UTSlowVolume_Content" });

	/** Slow Volume Ammo **/
	public static final UT3ItemType SLOW_VOLUME_AMMO = MakePrototype(
			Category.AMMO, UT3Group.SLOW_VOLUME, new String[] {
					"UTDeployableSlowVolumeAmmo", "UTDeployableSlowVolume.AmmoPickup" });

	/** Slow Field **/
	public static final UT3ItemType SLOW_FIELD = MakePrototype(Category.OTHER,
			UT3Group.SLOW_FIELD, new String[] { "SlowField",
					"SlowField.Pickup" });
        
        /** Invisibility **/
	public static final UT3ItemType INVISIBILITY = MakePrototype(
			Category.OTHER, UT3Group.INVISIBILITY, new String[] {
					"Invisibility", "Invisibility.Pickup" });

	/** Invulnerability **/
	public static final UT3ItemType INVULNERABILITY = MakePrototype(
			Category.OTHER,
			UT3Group.INVULNERABILITY,
			new String[] { "Invulnerability",
					"Invulnerability.Pickup", });

        /** WeaponLocker. */
        public static final UT3ItemType WEAPON_LOCKER = MakePrototype(Category.OTHER,
			UT3Group.WEAPON_LOCKER, new String[] { "WeaponLocker", "WeaponLocker.Pickup" });
        
	/** Key. */
	public static final UT3ItemType KEY = MakePrototype(Category.OTHER,
			UT3Group.KEY, new String[] { "UnrealGame.KeyPickup" });

	/** No UT3ItemType */
	public static final UT3ItemType NONE = MakePrototype(Category.OTHER,
			UT3Group.NONE, new String[] { "None", "NONE", "none" });

	/**
	 * Public constructor - creates ItemType of the EXTRA category and Group
	 * OTHER.
	 * 
	 * @param name
	 *            Type name from GB engine.
	 */
	public UT3ItemType(String name) {
		this.name = name;
		this.category = Category.OTHER;
		this.group = UT3Group.OTHER;
	}

	/**
	 * Prototypes constructor.
	 */
	private UT3ItemType(String name, Category category, UT3Group group) {
		this.name = name;
		this.category = category;
		this.group = group;
	}

	/**
	 * Proto-constructor.
	 * 
	 * @param category
	 *            Category of the item.
	 * @param group
	 *            Group of the item.
	 * @param utNames
	 *            Names of the item in UT engine.
	 * @return Prototype of known ItemType.
	 */
	public static UT3ItemType MakePrototype(Category category, UT3Group group,
			String[] utNames) {
		UT3ItemType type;
		synchronized (protos) {
			// create new itemtype prototype
			type = new UT3ItemType(utNames[0], category, group);
			// register the itemtype prototype
			for (String utName : utNames)
				protos.put(utName, type);
			// C'est la vie..
			if (category != null) {
				CATEGORIES.get(category).add(type);
			}
			if (group != null) {
				GROUPS.get(group).add(type);
			}
		}
		return type;
	}

	public static UT3ItemType getWeaponItemType(UnrealId id) {
		if (id == null)
			return null;

		return getWeaponItemType(id.getStringId());
	}

	/**
	 * Attempts to recognize the weapon you are currently holding...
	 * <p>
	 * <p>
	 * See {@link Self#getWeapon()}.
	 * <p>
	 * <p>
	 * May return null == weapon was not recognized. ALWAYS CHECK!
	 * 
	 * @return
	 */
	public static UT3ItemType getWeaponItemType(String str) {
		str = str.toLowerCase();
		if (str.contains("enforcer"))
			return UT3ItemType.ENFORCER;
		if (str.contains("impacthammer"))
			return UT3ItemType.IMPACT_HAMMER;
		if (str.contains("flakcannon"))
			return UT3ItemType.FLAK_CANNON;
		if (str.contains("biorifle"))
			return UT3ItemType.BIO_RIFLE;
		if (str.contains("shockrifle"))
			return UT3ItemType.SHOCK_RIFLE;
		if (str.contains("linkgun"))
			return UT3ItemType.LINK_GUN;
		if (str.contains("sniperrifle"))
			return UT3ItemType.SNIPER_RIFLE;
		if (str.contains("rocketlauncher"))
			return UT3ItemType.ROCKET_LAUNCHER;
		if (str.contains("minigun"))
			return UT3ItemType.STINGER_MINIGUN;
		if (str.contains("translocator"))
			return UT3ItemType.TRANSLOCATOR;
		if (str.contains("translauncher"))
			return UT3ItemType.TRANSLOCATOR;
		if (str.contains("xrayvolume"))
			return UT3ItemType.XRAY_VOLUME;
		if (str.contains("spiderminetrap"))
			return UT3ItemType.SPIDERMINE_TRAP;
		if (str.contains("energyshield"))
			return UT3ItemType.ENERGY_SHIELD;
		if (str.contains("empmine"))
			return UT3ItemType.EMP_MINE;
		if (str.contains("linkgenerator"))
			return UT3ItemType.LINK_GENERATOR;
		if (str.contains("shapedcharge"))
			return UT3ItemType.SHAPED_CHARGE;
		if (str.contains("slowvolume"))
			return UT3ItemType.SLOW_VOLUME;
		return null;
	}

	/**
	 * Check if this weapon is a deployable.
	 * 
	 * @param item
	 *            item to be checked.
	 * @note can be removed if we refactor everything in the hunter-bot to
	 *       support deployables.
	 */
	public static boolean isDeployable(ItemType item) {
		return (item == UT3ItemType.EMP_MINE
				|| item == UT3ItemType.ENERGY_SHIELD
				|| item == UT3ItemType.LINK_GENERATOR
				|| item == UT3ItemType.SHAPED_CHARGE
				|| item == UT3ItemType.SLOW_VOLUME
				|| item == UT3ItemType.SPIDERMINE_TRAP || item == UT3ItemType.XRAY_VOLUME);
	}

	@Override
	public String getName() {
		return name;
	}

	/**
	 * Retrieves an ItemType for the specified item type name.
	 * 
	 * @param utName
	 *            e.g. Item.getType()
	 * @return
	 */
	public static ItemType getItemType(String utName) {
		ItemType type;

		synchronized (protos) {
			type = protos.get(utName);
			if (type != null)
				return type;

			type = new UT3ItemType(utName);
			protos.put(utName, type);
		}
		return type;
	}

	/* ======================================================================== */

}
