package cz.cuni.amis.pogamut.ut2004.communication.worldview;

import java.io.File;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.ILockableWorldView;
import cz.cuni.amis.pogamut.base.component.bus.exception.ComponentNotRunningException;
import cz.cuni.amis.utils.exception.PogamutInterruptedException;
import cz.cuni.amis.utils.flag.Flag;
import cz.cuni.amis.utils.flag.FlagListener;
import java.util.logging.Level;

/**
 * Abstract parent class for all sync worldview test cases.
 * For the order of calling certain methods, look into WorldViewAbstractTestCase doTest().
 * @author Radek 'Black_Hand' Pibil
 *
 */
public abstract class AbstractSyncWorldViewTest extends AbstractBatchAwareWorldViewTest {
	private Thread asyncTestThread = null;
	private IWorldChangeEvent lastEvent = null;
	private CyclicBarrier barrier = new CyclicBarrier(2);
	private Flag<Boolean> consistentCriticalSection = new Flag<Boolean>(false);
	
	private FlagListener<Boolean> runningListener = new FlagListener<Boolean>() {
		@Override
		public void flagChanged(Boolean changedValue) {
			if (false) {
				consistentCriticalSection.setFlag(false);
			}
		}
	};
	
	protected AbstractSyncWorldViewTest(File planFile) {
		super(planFile);
	}
	
	@Override
	protected void preNotifications() {
		asyncTestThread = new Thread(new Runnable(){
			@Override
			public void run() {
				AbstractSyncWorldViewTest.this.asyncDoTest();
			}				
		});
		asyncTestThread.start();		
	}	
	
	/**
	 * Notification thread waits on this to meet the worldview locking test thread.
	 */
	@Override
	protected void postTest() {
		try {
			//barrier.await(10000, TimeUnit.MILLISECONDS);
			barrier.await(999999999, TimeUnit.MILLISECONDS);
		} catch (InterruptedException e) {
			e.printStackTrace();
			fail("Test thread interrupted at post-test barrier: " + e.toString());
		} catch (BrokenBarrierException e) {
			e.printStackTrace();
			fail("Barrier broken in test thread at post-test barrier: " + e.toString());
			
		} catch (TimeoutException e) {
			e.printStackTrace();
			fail("Failed to meetup with asyncDoTest thread at the end of test run.");
		}
		if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("leaving postTest");
	}	
	
	/**
	 * Worldview locking test thread execution method.
	 * <p><p>
	 * Run inside a thread, started from {@link AbstractSyncWorldViewTest#preNotifications()}.
	 */
	protected void asyncDoTest() {
		while (getContext().getRunningFlag().getFlag()) {			
			try {
				if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("locking worldview");
				((ILockableWorldView)getContext().getWorldView()).lock();
				if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("locked");
				getInnerRunning().setFlag(true);
				
				try {
					long currTime = System.currentTimeMillis();
					int sleepTime = 100;
					
					if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("sleeping for " + sleepTime + " ms");
				
					getInnerRunning().waitFor(sleepTime, false);
										
					getInnerRunning().setFlag(false);
					// getLogger().info("inConsistentCriticalSection = false");
				} catch (PogamutInterruptedException e) {
					e.printStackTrace();
					if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("Interrupted in asyncDoTest.");
					fail("Interrupted in asyncDoTest while blocking notifications in worldview.");
					break;
				} finally {				
					if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("unlocking worldview");
					((ILockableWorldView)getContext().getWorldView()).unlock();
					if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("unlocked");
				}
				
			} catch (ComponentNotRunningException e) {
				if (getContext().getLog().isLoggable(Level.INFO)) getContext().getLog().info("WorldView has been stopped.");
				break;
			}
		}
		
		try {
			barrier.await(10000, TimeUnit.MILLISECONDS);
		} catch (InterruptedException e) {
			e.printStackTrace();
			fail("Method asyncDoTest interrupted at post-test barrier: " + e.getMessage());
		} catch (BrokenBarrierException e) {
			e.printStackTrace();
			fail("Barrier broken in method asyncDoTest at post-test barrier: " + e.getMessage());
		} catch (TimeoutException e) {
			e.printStackTrace();
			fail("Failed to meetup with main thread at the end of test run.");
		}
	}

	private Flag<Boolean> getInnerRunning() {
		return consistentCriticalSection;
	}	
}
