package cz.cuni.amis.pogamut.usar2004.agent.module.datatypes;

import cz.cuni.amis.pogamut.usar2004.agent.module.sensor.SuperSensor;
import java.util.*;

/**
 * Container of sensor message representatives. Note that this is equvalent with
 * SensorsContainer. The difference is that the Queued version does not throw
 * out records until it is requiered to do so. It lines them up to lists. This
 * is needed when we want the best quality readings and regular ticks are not
 * enough to keep up with the speed of sensor messages being recieved.
 *
 * @author vejmanm
 */
public class SensorsContainerQueued extends HashMap<String, Map<String, Queue<SuperSensor>>>
{
    /**
     * Gets sensor message representatives from local hashmap. Returns null if
     * none matches or this hash map is empty.
     *
     * @param type String representing the type of sensor to return.
     * @return Returns List of specified type of Sensor module.
     */
    public List<SuperSensor> getSensorsByType(String type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        if(this.containsKey(type))
        {
            List<SuperSensor> allAvailible = new ArrayList<SuperSensor>();
            for(String string : this.get(type).keySet())
            {
                allAvailible.addAll(this.get(type).get(string));
                this.get(type).get(string).clear();
            }
            return allAvailible;
        }
        return null;
    }

    /**
     * Iterates through local hashmap values and seeks match. Returns null if
     * this hash map is empty. Note, that if <B>type</B> = UNKNOWN_SENSOR it
     * returns all unknown sensors.
     *
     * @param type SensorType representing the type of sensor to return.
     * @return Returns List of all sensors that suit input SensorType.
     */
    public List<SuperSensor> getSensorsBySensorType(SensorType type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        List<SuperSensor> allAvailible = new ArrayList<SuperSensor>();
        for(Map<String, Queue<SuperSensor>> map : this.values())
        {
            if(map == null || map.isEmpty())
            {
                continue;
            }
            for(String string : map.keySet())
            {
                if(map.get(string).isEmpty())
                {
                    continue;
                }
                if(map.get(string).peek().getSensorType() == type)
                {
                    allAvailible.addAll(map.get(string));
                    map.get(string).clear();
                }
            }
        }
        return allAvailible;
    }

    /**
     * Adds every object that can be casted to initial class to the output list.
     * Note that if feeded with SuperSensor class it will return all available
     * submodules.
     *
     * @param c Class representing the type of which the return list should be.
     * @return Returns a list of eligible objects, that can be casted to Class
     * c.
     */
    public List<SuperSensor> getSensorsByClass(Class c)
    {
        if(c == null)
        {
            return null;
        }
        List<SuperSensor> allAvailible = new ArrayList<SuperSensor>();
        for(Map<String, Queue<SuperSensor>> map : this.values())
        {
            for(Queue<SuperSensor> queue : map.values())
            {
                if(queue.isEmpty())
                {
                    continue;
                }
                if(c.isInstance(queue.peek()))
                {
                    allAvailible.addAll(queue);
                    queue.clear();
                }
            }
        }
        return allAvailible;
    }

    /**
     * Gets sensor message representatives from local hashmap specified by type
     * and by name. Returns null if none matches or this hash map is empty.
     *
     * @param type String representing the type of sensor to return.
     * @param name String representing the name of sensor to return.
     * @return Returns List of specified type of Sensor module.
     */
    public List<SuperSensor> getSensorByTypeName(String type, String name)
    {
        if(this.isEmpty())
        {
            return null;
        }
        if(this.containsKey(type) && !this.get(type).isEmpty())
        {
            List<SuperSensor> allAvailible = new ArrayList<SuperSensor>();
            allAvailible.addAll(this.get(type).get(name));
            this.get(type).get(name).clear();
        }
        return null;
    }

    /**
     * For each type of sensor it adds all individuals to the returnee List as a
     * couple (Type, Name)
     *
     * @return returns Map of couples (Type/Name) of non empty sensor
     * representatives.
     */
    public List<MessageDescriptor> getNonEmptyDescription()
    {
        if(this.isEmpty())
        {
            return null;
        }
        List<MessageDescriptor> list = new ArrayList<MessageDescriptor>();
        for(String type : this.keySet())
        {
            for(String name : this.get(type).keySet())
            {
                list.add(new MessageDescriptor(type, name));
            }
        }
        return list;
    }

    /**
     * Iterates through local hashmap and seeks for the specified sensor type.
     * If found and is not empty returns true. False otherwise.
     *
     * @param type SensorType is the type of sensor we want to be able to work
     * with.
     * @return Returns true if map of specified sensor type is not null nor
     * empty. False otherwise.
     */
    public boolean isReady(SensorType type)
    {
        if(this.isEmpty())
        {
            return false;
        }
        for(Map<String, Queue<SuperSensor>> map : this.values())
        {
            if(map == null || map.isEmpty())
            {
                continue;
            }
            for(String string : map.keySet())
            {
                if(map.get(string).isEmpty())
                {
                    continue;
                }
                if(map.get(string).peek().getSensorType() == type)
                {
                    return true;
                }
            }
        }
        return false;
    }
}
