package cz.cuni.amis.pogamut.usar2004.agent.module.datatypes;

import cz.cuni.amis.pogamut.usar2004.agent.module.geometry.SuperGeometry;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Container of geometry message representatives. It is basicly a hashmap of
 * maps. There is geometry messages logged by type and than by name.
 *
 * @author vejmanm
 */
public class GeometryContainer extends HashMap<String, Map<String, SuperGeometry>>
{
    /**
     * Gets geometry message representatives from local hashmap. Returns null if
     * none matches or this hash map is empty.
     *
     * @param type String representing the type of Geometry to return
     * @return Returns List of specified type of Geometry representatives.
     */
    public List<SuperGeometry> getGeometriesByType(String type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        if(this.containsKey(type))
        {
            return new ArrayList<SuperGeometry>(this.get(type).values());
        }
        return null;
    }

    /**
     * Iterates through local hashmap values and seeks match. Returns null if
     * this hash map is empty. Note, that if <B>type</B> = UNKNOWN_Geometry it
     * returns all unknown Geometries.
     *
     * @param type GeometryType representing the type of Geometry to return
     * @return Returns List of all Geometries that suit input GeometryType.
     */
    public List<SuperGeometry> getGeometriesByGeometryType(GeometryType type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        List<SuperGeometry> list = new ArrayList<SuperGeometry>();
        for(Map<String, SuperGeometry> map : this.values())
        {
            if(map == null || map.isEmpty())
            {
                continue;
            }
            if(map.values().iterator().next().getGeometryType() == type)
            {
                list.addAll(map.values());
            }
        }
        return list;
    }

    /**
     * Adds every object that can be casted to initial class to the output list.
     * Note that if You feed this method with SuperClass it will return all
     * available submodules.
     *
     * @param c Class representing the type of which the return list should be
     * @return Returns a list of eligible objects, that can be casted to Class c
     */
    public List<SuperGeometry> getGeometriesByClass(Class c)
    {
        if(c == null || this.isEmpty())
        {
            return null;
        }
        List<SuperGeometry> list = new ArrayList<SuperGeometry>();
        for(Map<String, SuperGeometry> map : this.values())
        {
            for(SuperGeometry geo : map.values())
            {
                if(c.isInstance(geo))
                {
                    list.add(geo);
                }
            }
        }
        return list;
    }

    /**
     * Gets geometry message representatives from local hashmap specified by
     * type and by name. Returns null if none matches or this hash map is empty.
     *
     * @param type String representing the type of geometry to return.
     * @param name String representing the name of geometry to return.
     * @return Returns List of specified type of Geometry representative.
     */
    public SuperGeometry getGeometryByTypeName(String type, String name)
    {
        if(this.isEmpty())
        {
            return null;
        }
        if(this.containsKey(type) && !this.get(type).isEmpty())
        {
            return this.get(type).get(name);
        }
        return null;
    }

    /**
     * For each type of Geometry it adds all individuals to the returnee List as
     * a couple (Type, Name)
     *
     * @return returns Map of couples (Type/Name) of non empty Geometries
     */
    public List<MessageDescriptor> getNonEmptyDescription()
    {
        if(this.isEmpty())
        {
            return null;
        }
        List<MessageDescriptor> list = new ArrayList<MessageDescriptor>();
        for(String type : this.keySet())
        {
            for(String name : this.get(type).keySet())
            {
                list.add(new MessageDescriptor(type, name));
            }
        }
        return list;
    }
}
