package cz.cuni.amis.pogamut.usar2004.agent.module.datatypes;

import cz.cuni.amis.pogamut.usar2004.agent.module.configuration.SuperConfiguration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Container of configuration message representatives. It is basicly a hashmap
 * of maps. There is geometry messages logged by type and than by name.
 *
 * @author vejmanm
 */
public class ConfigContainer extends HashMap<String, Map<String, SuperConfiguration>>
{
    /**
     * Gets configuration message representatives from local hashmap. Returns
     * null if none matches or this hash map is empty.
     *
     * @param type String representing the type of Configuration to return
     * @return Returns List of specified type of Configuration representatives.
     */
    public List<SuperConfiguration> getConfigurationsByType(String type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        if(this.containsKey(type))
        {
            return new ArrayList<SuperConfiguration>(this.get(type).values());
        }
        return null;
    }

    /**
     * Iterates through local hashmap values and seeks match. Returns null if
     * this hash map is empty. Note, that if <B>type</B> = UNKNOWN_Configuration
     * it returns all unknown Configurations.
     *
     * @param type ConfigurationType representing the type of Configuration to
     * return
     * @return Returns List of all Configurations that suit input
     * ConfigurationType.
     */
    public List<SuperConfiguration> getConfigurationsByConfigType(ConfigType type)
    {
        if(this.isEmpty() || type == null)
        {
            return null;
        }
        List<SuperConfiguration> list = new ArrayList<SuperConfiguration>();
        for(Map<String, SuperConfiguration> map : this.values())
        {
            if(map == null || map.isEmpty())
            {
                continue;
            }
            if(map.values().iterator().next().getConfigType() == type)
            {
                list.addAll(map.values());
            }
        }
        return list;
    }

    /**
     * Adds every object that can be casted to initial class to the output list.
     * Note that if You feed this method with SuperClass it will return all
     * available submodules.
     *
     * @param c Class representing the type of which the return list should be
     * @return Returns a list of eligible objects, that can be casted to Class c
     */
    public List<SuperConfiguration> getConfigurationsByClass(Class c)
    {
        if(c == null || this.isEmpty())
        {
            return null;
        }
        List<SuperConfiguration> list = new ArrayList<SuperConfiguration>();
        for(Map<String, SuperConfiguration> map : this.values())
        {
            for(SuperConfiguration cnf : map.values())
            {
                if(c.isInstance(cnf))
                {
                    list.add(cnf);
                }
            }
        }
        return list;
    }

    /**
     * Gets configuration message representatives from local hashmap specified
     * by type and by name. Returns null if none matches or this hash map is
     * empty.
     *
     * @param type String representing the type of configuration to return.
     * @param name String representing the name of configuration to return.
     * @return Returns List of specified type of Configuration representative.
     */
    public SuperConfiguration getConfigurationByTypeName(String type, String name)
    {
        if(this.isEmpty())
        {
            return null;
        }
        if(this.containsKey(type) && !this.get(type).isEmpty())
        {
            return this.get(type).get(name);
        }
        return null;
    }

    /**
     * For each type of Configuration it adds all individuals to the returnee
     * List as a couple (Type, Name)
     *
     * @return returns Map of couples (Type/Name) of non empty Configurations
     */
    public List<MessageDescriptor> getNonEmptyDescription()
    {
        List<MessageDescriptor> list = new ArrayList<MessageDescriptor>();
        for(String type : this.keySet())
        {
            for(String name : this.get(type).keySet())
            {
                list.add(new MessageDescriptor(type, name));
            }
        }
        return list;
    }
}
