package cz.cuni.amis.pogamut.usar2004.agent.module.configuration;

import cz.cuni.amis.pogamut.usar2004.agent.module.datatypes.ConfigType;
import cz.cuni.amis.pogamut.usar2004.communication.messages.usarinfomessages.ConfigurationMessage;
import java.util.Map;
import java.util.Set;

/**
 * Parent class for all possible configuration subjects. It covers the basic and
 * common properties for all the configuration subjects.
 *
 * @author vejmanm
 */
public abstract class SuperConfiguration
{
    protected ConfigType confType;
    protected ConfigurationMessage lastMessage;

    /**
     * Ctor.
     *
     * @param type Configuration type describes particular subject about which
     * we want to know about. It is used to distinguish incoming message from
     * the server.
     */
    public SuperConfiguration(ConfigType type)
    {
        this.confType = type;
    }

    /**
     * Used to make sure the object is filled.
     *
     * @return Returns true if the object is filled with Conf. message.
     */
    public Boolean isReady()
    {
        return (lastMessage != null);
    }

    /**
     * Type describes the vehicle type. It will be one of the following values:
     * "GroundVehicle”, “LeggedRobot”, “NauticVehicle”, or “AerialVehicle” of
     * "Camera" or some kind of sensor or effecter or "MisPkg".
     *
     * @return Returns type of the geometry message.
     */
    public String getType()
    {
        return lastMessage.getType();
    }

    /**
     * Name of the subject that this geometry message concerns.
     *
     * @return Returns the name of the Geometry item.
     */
    public String getName()
    {
        return lastMessage.getName();
    }

    /**
     * ConfigType is a descriptor used for creating particular instance and for
     * getting type information about particular configuration subject.
     *
     * @return Returns ConfigType.
     */
    public ConfigType getConfigType()
    {
        return this.confType;
    }

    /**
     * For a camera, sensor or effecter, a configuration message conatins a pair
     * ‘{Name Value}’ that describes the feature of this sensor type. Different
     * sensor types have different name value pairs. For detailed information,
     * please refer to section 10 of USARSim manual v3.1.3 about how to
     * configure the sensor.
     *
     * Example: CONF {Type Camera} {CameraDefFov 0.8727} {CameraMinFov 0.3491}
     * {CameraMaxFov 2.0943} {CameraFov 0.8726}
     *
     * @return Returns keySet of features.
     */
    public Set<String> getFeatureNames()
    {
        return lastMessage.getFeatures().keySet();
    }

    /**
     * For a camera, sensor or effecter, a configuration message conatins a pair
     * ‘{Name Value}’ that describes the feature of this sensor type. Different
     * sensor types have different name value pairs. For detailed information,
     * please refer to section 10 of USARSim manual v3.1.3 about how to
     * configure the sensor.
     *
     * Example: CONF {Type Camera} {CameraDefFov 0.8727} {CameraMinFov 0.3491}
     * {CameraMaxFov 2.0943} {CameraFov 0.8726}
     *
     * @return Returns feature value by feature name from the feature map.
     */
    public String getFeatureValueBy(String Name)
    {
        return lastMessage.getFeatures().get(Name);
    }

    /**
     * For a camera, sensor or effecter, a configuration message conatins a pair
     * ‘{Name Value}’ that describes the feature of this sensor type. Different
     * sensor types have different name value pairs. For detailed information,
     * please refer to section 10 of USARSim manual v3.1.3 about how to
     * configure the sensor.
     *
     * Example: CONF {Type Camera} {CameraDefFov 0.8727} {CameraMinFov 0.3491}
     * {CameraMaxFov 2.0943} {CameraFov 0.8726}
     *
     * @return Returns the size of the feature map.
     */
    public int getFeatureSize()
    {
        return lastMessage.getFeatures().size();
    }

    /**
     * For a camera, sensor or effecter, a configuration message conatins a pair
     * ‘{Name Value}’ that describes the feature of this sensor type. Different
     * sensor types have different name value pairs. For detailed information,
     * please refer to section 10 of USARSim manual v3.1.3 about how to
     * configure the sensor.
     *
     * Example: CONF {Type Camera} {CameraDefFov 0.8727} {CameraMinFov 0.3491}
     * {CameraMaxFov 2.0943} {CameraFov 0.8726}
     *
     * @return Returns Map of features.
     */
    public Map<String, String> getFeatures()
    {
        return lastMessage.getFeatures();
    }

    /**
     * Method used for updating the message object that provides particular
     * properties for given type. Note that this object is created by yylex
     * parser and contains properties for all configuration subject types. But
     * only relevat to individual Config Subject are filled.
     *
     * @param message Configuration message from server.
     */
    public void updateMessage(ConfigurationMessage message)
    {
        lastMessage = message;
    }
}
