/*
 * UCCWrapper.java
 *
 */
package cz.cuni.amis.pogamut.udk.utils;

import com.jezhumble.javasysmon.OsProcess;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.state.WaitForAgentStateChange;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateDown;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateUp;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base.utils.logging.LogPublisher;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.Console;
import cz.cuni.amis.pogamut.udk.factory.direct.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.server.exception.UCCStartException;
import cz.cuni.amis.utils.exception.PogamutException;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import com.jezhumble.javasysmon.ProcessVisitor;
import com.jezhumble.javasysmon.JavaSysMon;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.component.exception.ComponentCantStopException;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.ChangeMap;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.MapChange;
import cz.cuni.amis.utils.ExceptionToString;
import cz.cuni.amis.utils.exception.PogamutInterruptedException;
import java.util.ArrayList;
import java.util.List;

/**
 * Wrapper of running instance of UDK server. Implements pooling of instances.
 * The location of UDK executabe will be determined by an environment variable
 * pogamut.udk.home (e.g. c:\Games\UDK). The property cam be set via <i>java ...
 * -Dpogamut.udk.home=c:\Games\UDK</i>. Another posibility is to set it
 * by code <code>System.setProperty("pogamut.udk.home", "c:\\udks\\UDK-2011-05");</code>.
 *
 * <p>
 * As of now, UCCWrapper uses console command exit to kill the server when calling {@link #stop() } or
 * upon JVM shutdown. However, this fails if there are non-Pogamut players connected (e.g. when spectating) and the server stays alive.
 * If the exit command is not effective, UCCWrapper tries to kill the server in a platform dependent way using UDK process PID. However, some
 * tricks are needed to obtain the pid. The biggest problem is that UDK.com which needs to be run in order to get console output from the server
 * spawns another process called UDK.exe and killing the original process does not kill UDK.exe. Moreover,
 * if used on UNIX (see below) the UDK.exe is not a child process of UDK.com for some reason... So there are some not exactly neat tricks. 
 * However, on Windows, the UCCWrapper might be considered reliable (even thread-safe) providing no other code in current JVM spawns 
 * (directly or undirectly)
 * processes called UDK.exe. On UNIX, there are some additional issues (see below).
 * </p>
 * 
 * <h3>UNIX usage</h3>
 * <p>
 *  It is possible to use UCCWrapper under UNIX systems. First you need to setup UDK
 * as described at our WIKI: <a href="http://pogamut.cuni.cz/pogamut-devel/doku.php?id=guidelines:udk_on_linux">
 * pogamut.cuni.cz/pogamut-devel/doku.php?id=guidelines:udk_on_linux</a>. Then UCCWrapper must have access to Wine 
 * in order to this, UCCWrapper uses property called WINE (might be an environment variable or it might be a Java property - see above).
 * Property value defaults to "wine" which should be just fine if you have wine regularly installed. Make sure that WINEPREFIX environment
 * variable is set in the shell that runs your program, otherwise Wine will not have access to correct wineprefix.
 * </p>
 * <p>
 * As noted above, there are some not very neat tricks in getting the process PID of UDK. On UNIX, UCCWrapper relies on the fact
 * that no other program than the current JVM spawns new processes called UDK.exe. If this is not true, UCCWrapper will usually
 * detect this situation and not read PID at all. But on very rare occasions (another process starts UDK.exe at the same time and starting of UDK.exe
 * in this JVM fails or is delayed) UCCWrapper may take pid of a different UDK.exe process. 
 * Since pid is only used to kill the server if the EXIT command fails, it is even more unlikely that this problem will actually
 * exhibit in killing wrong process. 
 * </p>
 * 
 * @author Ik
 */
public class UCCWrapper {

    public static final long DEFAULT_START_TIMEOUT = 2 * 60 * 1000;

    /**
     * Mutex that forces only one child UCC to be spawned at a time. This allows
     * 
     */
    private static final Object spawnProcessMutex = new Object();
    
    /**
     * How many times will the wrapper try to send ChangeMap command to server,
     * until it gives up waiting for notification.
     */
    public static final int CHANGE_MAP_COMMAND_MAX_REPEATS = 5;
    
    public static final long DEFAULT_UDK_EXE_SPAWN_TIMEOUT_WINDOWS = 10000;
    
    /**
     * The ammount of time taken to to start UDK.exe on UNIX is quite large, since wine needs to boot in that time
     */
    public static final long DEFAULT_UDK_EXE_SPAWN_TIMEOUT_UNIX = 30000;
        
    /**
     * How long (msec) will wrapper wait for confirmation of ChangeMap message
     */
    public static final int CHANGE_MAP_CONFIRMATION_TIMEOUT = 500;

    /**
     * Start the server, if not already started. This method blocks, until the server
     * is ready to accept bot connections
     * @throws UCCStartException 
     */
    public synchronized void start() throws UCCStartException {
        if(uccProcess != null){
            throw new IllegalStateException("Wrapper already started");
        }
        initUCCWrapper();
        uccLog.info("Adding shutdown hook");
        Runtime.getRuntime().addShutdownHook(shutDownHook);
    }

    protected String getMapNameWithOptions() {
        // start new ucc instance
        String id = System.currentTimeMillis() + "a" + fileCounter++;
        String fileWithPorts = "GBports" + id;
        String portsSetting;
        if(configuration.botServerPort != null || configuration.controlServerPort != null){
            portsSetting = "";
            if(configuration.controlServerPort != null){
                portsSetting += "?ControlServerPort=" + configuration.controlServerPort;
            }                
            if(configuration.botServerPort != null){
                portsSetting += "?BotServerPort=" + configuration.botServerPort;
            }                
        }
        else if(configuration.startOnUnusedPort){
            portsSetting = "?PortsLog=" + fileWithPorts + "?bRandomPorts=true";
        } else {
            portsSetting = "";
        }
        String gameOptions = configuration.mapName
                             + "?game=" + configuration.gameBotsPack + "." + configuration.gameType
                             + portsSetting + configuration.options;
        return gameOptions;
    }

    private Set<Integer> getNewUDKProcesses(JavaSysMon monitor, Set<Integer> oldUDKProcesses){
            UDKChildProcessVisitor visitor = new UDKChildProcessVisitor();
            if(isWindows()){
                //On Windows, the UDK.exe process is child of current process
                monitor.visitProcessTree(monitor.currentPid(), visitor);
            } else {
                //On UNIX, the UDK.exe is not a child of current process - we need
                //to search the whole process tree, which reduces reliability as noted
                //class javadoc
                for(OsProcess child :(List<OsProcess>)monitor.processTree().children()){
                    monitor.visitProcessTree(child.processInfo().getPid(), visitor);
                }                
            }
            Set<Integer> newProcesses = visitor.getUdkPids();

            newProcesses.removeAll(oldUDKProcesses);
            return newProcesses;
        
    }
    
    /**
     * Spawns a new UDK process and sets {@link #uccPid} to it's PID. To work correctly,
     * the code relies on the fact that no other method in this JVM runs UDK processes and
     * that no method kills a process unless it acquires lock on spawnProcessMutex.
     * @param procBuilder
     * @return 
     */
    private Process spawnUDK(ProcessBuilder procBuilder) throws IOException {
        synchronized (spawnProcessMutex) {
            JavaSysMon monitor = new JavaSysMon();
            UDKChildProcessVisitor beforeVisitor = new UDKChildProcessVisitor();
            
            long udkExeSpawnTimeout;
            if(isWindows()){
                //On Windows, the UDK.exe process is child of current process
                monitor.visitProcessTree(monitor.currentPid(), beforeVisitor);
                udkExeSpawnTimeout = udkExeSpawnTimeoutWindows;
            } else {
                //On UNIX, the UDK.exe is not a child of current process - we need
                //to search the whole process tree, which reduces reliability as noted
                //class javadoc
                for(OsProcess child :(List<OsProcess>)monitor.processTree().children()){
                    monitor.visitProcessTree(child.processInfo().getPid(), beforeVisitor);
                }
                udkExeSpawnTimeout = udkExeSpawnTimeoutUnix;
            }
            Set<Integer> alreadySpawnedProcesses = beforeVisitor.getUdkPids();

            Process proc = procBuilder.start();

            long startTime = System.currentTimeMillis();
            
            Set<Integer> newProcesses;
            
            //actively wait before the UDK.exe process is spawned
            do {
                newProcesses = getNewUDKProcesses(monitor, alreadySpawnedProcesses);
            } while (newProcesses.isEmpty() && System.currentTimeMillis() - startTime < udkExeSpawnTimeout); 
            


            if (newProcesses.isEmpty()) {
                uccLog.severe("There is no new UDK PID.");
            } else if (newProcesses.size() > 1) {
                uccLog.severe("Multiple new candidate UDK PIDs");
            } else {
                uccPid = newProcesses.iterator().next();
            }
            return proc;
        }
    }

    private void killUDKByPID() {
        if (uccPid < 0) {
            uccLog.severe("Cannot kill UCC by PID. PID not set.");
            return;
        }
        synchronized (spawnProcessMutex) {
            JavaSysMon monitor = new JavaSysMon();
            monitor.killProcessTree(uccPid, false);
            uccPid = -1;
        }
    }

    private static class UDKChildProcessVisitor implements ProcessVisitor {

        Set<Integer> udkPids = new HashSet<Integer>();

        @Override
        public boolean visit(OsProcess op, int i) {
            if (op.processInfo().getCommand().contains("UDK.exe") || (op.processInfo().getName().equals("UDK.exe"))) {
                udkPids.add(op.processInfo().getPid());
            }
            return false;
        }

        public Set<Integer> getUdkPids() {
            return udkPids;
        }
    }

    /**
     * Kills all UDK instances with platform specific-system call.
     * Useful to enforce a full cleanup.
     */
    public static void killAllUCCs(LogCategory uccLog) {
        if (!isWindows()) {
            //kill process in linux with comand 'killall -9 <process_name>'

            //System.out.println(plannerRunFile.getName());
            String[] command1 = new String[2];
            command1[0] = "killall";
            command1[1] = "UDK.com";

            String[] command2 = new String[2];
            command2[0] = "killall";
            command2[1] = "UDK.exe";

            try {
                Runtime.getRuntime().exec(command1);
                Runtime.getRuntime().exec(command2);
            } catch (IOException ex) {
                uccLog.log(Level.SEVERE, "Could not kill the UDK process: " + ex, ex);
            }
        } else {
            //kill process with wnodes task kill
            String[] command = new String[4];
            command[0] = "taskkill";
            command[1] = "/F";
            command[2] = "/IM";
            command[3] = "UDK.*";


            try {
                Runtime.getRuntime().exec(command);
            } catch (IOException ex) {
                uccLog.log(Level.SEVERE, "Could not kill the UDK with taskkill: " + ex, ex);
            }

        }

    }
    /** Loger containing all output from running instance of UCC. */
    protected LogCategory uccLog;

    protected static int fileCounter = 0;

    Process uccProcess = null;

    /** Port for bots. */
    protected int gbPort = -1;

    /** Port for server connection. */
    protected int controlPort = -1;

    /** Port for observer connection. */
    protected int observerPort = -1;

    protected IUDKServer utServer = null;

    /** First port assigned to a ucc instance. */
    protected static final int basePort = 39782;

    protected static Integer nextUccWrapperUID = 0;

    /** ID of the wrapper object. Useful for debuging. */
    protected int uccWrapperUID = 0;

    protected String unrealHome = null;

    protected UCCWrapperConf configuration = null;

    protected long startingTimeout = DEFAULT_START_TIMEOUT;
    //protected String mapToLoad

    protected int uccPid = -1;

    protected UDKServerFactory serverFactory;

    /**
     * Agressive killing means that the server is always killed by PID, if PID is available 
     * Otherwise it is killed by EXIT command first and if the connection dies, it is considered killed. In some
     * cases however, the connection dies after exit command, but the server is not down.
     * Setting this to false is conservative as it lessens the chance of killing a wrong process, setting
     * this to true on the other hand raise chances that the server is actually stopped.
     * Set to false by default, since the reading of PID depends on some non-obvious requirements
     * the programmer should check (see class docs)
     */
    protected boolean aggressiveKilling = false;
    
    /**
     * How long to wait for UDK.exe process to be spawned after the UDK.com process
     * has been started. On Windows. 
     */
    private long udkExeSpawnTimeoutWindows = DEFAULT_UDK_EXE_SPAWN_TIMEOUT_WINDOWS;

    /**
     * How long to wait for UDK.exe process to be spawned after the UDK.com process
     * has been started for UNIX systems. The bigger the value, the bigger chance of trouble on UNIX systems (see class comments), however
     * low values might mean that we will not get UDK process pid.
     */
    private long udkExeSpawnTimeoutUnix = DEFAULT_UDK_EXE_SPAWN_TIMEOUT_UNIX;
    
    
    ScannerSink scanner;
    /**
     * @return Log with output of UCC. If you want to listen also for messages 
     * from the startup sequence then use UCCWrapper.create(Logger parent). Set
     * Parent logger of this log and register listeners before creating this
     * instance of UCCWrapper.  
     */
    public Logger getLogger() {
        return uccLog;
    }

    /**
     * @return Server connected to this UCC instance.
     */
    public synchronized IUDKServer getUTServer() {
        stopCheck();
        if (utServer == null) {
            UDKServerRunner serverRunner = new UDKServerRunner(serverFactory, "NBUTServer", getHost(), controlPort);
            utServer = serverRunner.startAgent();
        }
        return utServer;
    }
    
    public synchronized void restartServer(){
        stopCheck();
        utServer = null;
    }

    protected String getUnrealHome() {
        if (unrealHome == null) {
            return Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UNREAL_HOME.getKey());
        } else {
            return unrealHome;
        }
    }

    public UCCWrapper(UCCWrapperConf configuration) {
        this(configuration, new UDKServerFactory());
    }

    public UCCWrapper(UCCWrapperConf configuration, UDKServerFactory factory) {
        this(configuration, factory, true);
    }

    public UCCWrapper(UCCWrapperConf configuration, boolean startImmediately) {
        this(configuration, new UDKServerFactory(), startImmediately);
    }

    /**
     * 
     * @param configuration
     * @param startingTimeout
     * @param serverFactory allows to specify external server factory. As of 3.3.0, this is needed to prevent memory leaks on subsequent creation of multiple UCCWrappers
     * @throws UCCStartException 
     */
    public UCCWrapper(UCCWrapperConf configuration, UDKServerFactory serverFactory, boolean startImmediately) throws UCCStartException {
        this.serverFactory = serverFactory;
        uccLog = new LogCategory("Wrapper");
        uccLog.addHandler(new LogPublisher.ConsolePublisher(new AgentId("UCC")));
        
        configuration.applyDefaults(UCCWrapperConf.DEFAULTS);        
        
        if (configuration.log != null) {
            uccLog.setParent(configuration.log);
        }
        this.configuration = configuration;
        uccWrapperUID = nextUccWrapperUID++;
        if(startImmediately){
            start();
        }
    }

    public UCCWrapperConf getConfiguration() {
        return configuration;
    }

    /**
     * Only makes sense if created with startImmediately = false
     * @param configuration 
     */
    public void setConfiguration(UCCWrapperConf configuration) {
        this.configuration = configuration;
    }

    public UDKServerFactory getServerFactory() {
        return serverFactory;
    }

    /**
     *  Only makes sense if created with startImmediately = false
     * @param serverFactory 
     */
    public void setServerFactory(UDKServerFactory serverFactory) {
        this.serverFactory = serverFactory;
    }

    public long getStartingTimeout() {
        return startingTimeout;
    }

    /**
     * Only makes sense if created with startImmediately = false and the serve was not started yet
     * @param udkExeSpawnTimeout 
     */
    public void setStartingTimeout(long startingTimeout) {
        this.startingTimeout = startingTimeout;
    }

    public long getUdkExeSpawnTimeoutUnix() {
        return udkExeSpawnTimeoutUnix;
    }

    /**
     * Only makes sense if created with startImmediately = false and the serve was not started yet
     * @param udkExeSpawnTimeout 
     */
    public void setUdkExeSpawnTimeoutUnix(long udkExeSpawnTimeoutUnix) {
        this.udkExeSpawnTimeoutUnix = udkExeSpawnTimeoutUnix;
    }

    public long getUdkExeSpawnTimeoutWindows() {
        return udkExeSpawnTimeoutWindows;
    }

    /**
     * Only makes sense if created with startImmediately = false and the serve was not started yet
     * @param udkExeSpawnTimeout 
     */
    public void setUdkExeSpawnTimeoutWindows(long udkExeSpawnTimeoutWindows) {
        this.udkExeSpawnTimeoutWindows = udkExeSpawnTimeoutWindows;
    }

    public boolean isAggressiveKilling() {
        return aggressiveKilling;
    }

    public void setAggressiveKilling(boolean aggressiveKilling) {
        this.aggressiveKilling = aggressiveKilling;
    }
    
    

    
    
    /**
     * Task that will kill the UCC process when user forgets to do so.
     */
    Thread shutDownHook = new Thread("UCC wrapper finalizer") {

        @Override
        public void run() {
            UCCWrapper.this.stopNoWaiting();
        }
    };

    private static boolean isWindows() {
        return System.getProperty("os.name").contains("Windows");
    }
    public static long stamp = System.currentTimeMillis();

    private void cleanupAfterException() {
        if (uccProcess != null) {
            uccProcess.destroy();
        }
        if (uccPid >= 0) {
            killUDKByPID();
        }
    }

    protected void initUCCWrapper() throws UCCStartException {
        try {
            String udkHomePath = getUnrealHome();
            String binariesPath = udkHomePath + File.separator + "Binaries";

            // default ucc executable for Windows
            //udk.exe creates a separate console for output and the output is thus unavaliable to this process, udk.com does not
            String uccFile = "Win32" + File.separator + "UDK.com";
            String uccFileAbsolute = binariesPath + File.separator + uccFile;

            
            // determine OS type, if it isn't win suppose it is Linux and try to run UDK through Wine
            List<String> command = new ArrayList<String>();
            
            if(isWindows()){
                command.add(uccFileAbsolute);
            }
            else {
                if(configuration.wineprefix != null && !configuration.wineprefix.isEmpty()){
                    command.add("WINEPREFIX=" + configuration.wineprefix);
                }
                command.add(Pogamut.getPlatform().getProperty("WINE", "wine"));
                command.add(uccFileAbsolute);
            }
            
            command.add("server");
            command.add(getMapNameWithOptions());
            if(configuration.otherCommandLineParameters != null){
                command.add(configuration.otherCommandLineParameters);
            }
            
            if(!isWindows()){
                command.add("-nohomedir");
            }
            

            ProcessBuilder procBuilder = new ProcessBuilder(command);
            procBuilder.directory(new File(binariesPath));

            uccProcess = spawnUDK(procBuilder);

            uccLog.info("Spawned UDK process. Pid: " + uccPid);

            scanner = new ScannerSink(uccProcess.getInputStream());
            scanner.start();
            new StreamSink(uccProcess.getErrorStream()).start();

            scanner.serverStartedLatch.await();

            if (scanner.exception != null) {
                // ucc failed to start 
                cleanupAfterException();
                throw new UCCStartException("Scanner exception", scanner.exception);
            }

            controlPort = scanner.controlPort;
            gbPort = scanner.botsPort;
            Thread.sleep(5000);
        } catch (InterruptedException ex) {
            cleanupAfterException();
            throw new UCCStartException("Interrupted.", ex);
        } catch (IOException ex) {
            cleanupAfterException();
            throw new UCCStartException("IO Exception.", ex);
        }
    }
    
    /**
     * Tries to change map on the server and blocks until either the map
     * has changed or timeout happens. 
     * @throws PogamutException if the map change was not succesful
     */
    public void changeMap(String mapName, boolean notifyGame, long timeout){
        uccLog.info("Changing map to: " + mapName);
        configuration.setMapName(mapName);
        if(getUTServer().getState().getFlag().isNotState(IAgentStateUp.class)){
            restartServer();
            new WaitForAgentStateChange(getUTServer().getState(), IAgentStateUp.class).await(2, TimeUnit.SECONDS);
        }
        
        //It is important to start to await new game start BEFORE the actual command is sent.
        //Otherwise I might miss the response, if it is fast enough, especially,
        //if I spend some time waiting for map change confirmation
        CountDownLatch gameStartLatch = awaitGameStart();
        
        
        /**
         * Sending the ChangeMap command.
         * In some cases, the ChangeMap command gets lost along the way. So it is
         * a good idea to wait for confirmation and resend it if neccessary
         */
        final CountDownLatch notifiedMapChangeLatch = new CountDownLatch(1);
        final String mapNameWithOptions = getMapNameWithOptions();
        IWorldEventListener<MapChange> mapChangeListener = new IWorldEventListener<MapChange>() {

            @Override
            public void notify(MapChange event) {
                if(event.getMapName().equals(mapNameWithOptions)){
                    notifiedMapChangeLatch.countDown();
                } else {
                    uccLog.severe("Notified of unexpected map change. Contiuning waiting. Notified for map:" + event.getMapName());
                }
            }
        };
        
        
        getUTServer().getWorldView().addEventListener(MapChange.class, mapChangeListener);
        
        boolean mapChangeConfirmed = false;
        
        for(int i = 0; i < CHANGE_MAP_COMMAND_MAX_REPEATS; i++){        
            if(gameStartLatch.getCount() <= 0){
                mapChangeConfirmed = true;
                uccLog.info("New match started while waiting for ChangeMap confirmation. Let's trust it is the correct game.");
                break;
            }
            if(getUTServer().getState().getFlag().isNotState(IAgentStateUp.class)){
                break;
            }
            getUTServer().getAct().act(new ChangeMap(mapNameWithOptions, notifyGame));
            try {
                if(notifiedMapChangeLatch.await(CHANGE_MAP_CONFIRMATION_TIMEOUT, TimeUnit.MILLISECONDS)){
                    uccLog.info("ChangeMap confirmed from GB server");
                    try {
                        getUTServer().stop();
                    } catch (ComponentCantStopException ex){
                        uccLog.severe("Server could not stop: " + ex.getMessage());                        
                    }
                    mapChangeConfirmed = true;
                    break;
                }
            } catch (InterruptedException ex) {                
            }
        }
        
        if(!mapChangeConfirmed){
            uccLog.severe("ChangeMap command was not confirmed. Trying to wait for map change anyway.");
        }
        
        
        try {
            try {
                if (gameStartLatch.await(timeout, TimeUnit.MILLISECONDS)) {
                    if (scanner.exception != null) {
                        throw scanner.exception;
                    }
                    uccLog.info("Map changed.");
                    restartServer();
                } else {
                    uccLog.severe("Map change timed out.");
                    throw new PogamutException("Map change timed out.", this);
                }
            } catch (InterruptedException ex) {
                uccLog.severe("Waiting for map change interrupted.");
                throw new PogamutInterruptedException("Waiting for map change interrupted.", ex, this);
            }
        } finally {
            gameStartLatch.countDown();
        }
    }
    
    
    /**
     * Returns a countdown latch, that is raised after map change message was encountered
     * @return 
     */
    public CountDownLatch awaitGameStart(){
        return scanner.awaitGameStart();
    }

    /**
     * Process of the
     * @return
     */
    public Process getProcess() {
        return uccProcess;
    }
    /** Was this instance already released? */
    protected boolean stopped = false;

    private synchronized void stopNoWaiting() {
        if(stopped){
            getLogger().info("UCCWrapper already stopped, ignoring call to stopNoWaiting()");
            return;
        }
        try {
            //Because of some kind of error, it is possible, that the
            //original server was killed. In this case I need to restart the server component - setting it to null, forces restart upon new 
            //getter call
            if (getUTServer().getState().getFlag().isNotState(IAgentStateUp.class)) {
                getLogger().info("Server component down, creating new to send EXIT command");
                utServer = null;
            }

            getLogger().info("Sending EXIT command");
            getUTServer().getAct().act(new Console("EXIT"));

            //Killing the server connection is useless - it might prevent EXIT from being propagated to the ucc
            //while killing the ucc will surely kill server connection

            //Wait for the server connection to die to ensure that ucc was killed
            WaitForAgentStateChange waitForServerDeath = new WaitForAgentStateChange(getUTServer().getState(), IAgentStateDown.class);
            try {
                waitForServerDeath.await(3, TimeUnit.SECONDS);
            } catch (PogamutInterruptedException ex) {
                uccLog.log(Level.SEVERE, "Waiting for server death interrupted.", ex);
            }

            if (!getUTServer().getState().getFlag().isState(IAgentStateDown.class)) {
                getUTServer().kill();
                uccLog.severe("Server did not die in response to EXIT command. Trying to kill by pid.");
                killUDKByPID();
            } else if (isAggressiveKilling()) {
                uccLog.info("Aggressive killing set to true. Trying to kill by pid.");
                killUDKByPID();
            }

            utServer = null;


            if (uccProcess != null) {
                uccProcess.destroy();
            }
        } catch (Exception ex) {
            uccLog.severe("Exception killing UCCWrapper :" + ex.getMessage());
            uccLog.severe(ExceptionToString.process(ex));
            if(isAggressiveKilling()){
                uccLog.info("Aggressive killing set to true. Trying to kill by pid.");
                killUDKByPID();
            }
        } finally {
            uccProcess = null;
            stopped = true;
        }
    }

    /**
     * Stops the UCC server.
     */
    public synchronized void stop() {
        if (uccProcess != null) {
            stopNoWaiting();
            getLogger().info("Removing shutdown hook");
            Runtime.getRuntime().removeShutdownHook(shutDownHook);
            
            try {
                Thread.sleep(1000);
                // give the process some time to terminate
            } catch (InterruptedException e) {
            }
        }
    }

    /**
     * @return Port for GameBots connection.
     */
    public int getBotPort() {
        stopCheck();
        return gbPort;
    }

    /**
     * @return Port of the Observer of GameBots2004.
     * @deprecated observer port is not used in UDK
     */
    @Deprecated
    public int getObserverPort() {
        stopCheck();
        return observerPort;
    }

    /**
     * @return Port for control connection.
     */
    public int getControlPort() {
        stopCheck();
        return controlPort;
    }

    protected void stopCheck() {
        if (stopped) {
            throw new PogamutException("UCC already stopped.", this);
        }
    }

    public String getHost() {
        return "localhost";
    }

    public SocketConnectionAddress getBotAddress() {
        return new SocketConnectionAddress(getHost(), getBotPort());
    }

    public SocketConnectionAddress getServerAddress() {
        return new SocketConnectionAddress(getHost(), getControlPort());
    }

    /**
     * 
     * @return
     * @deprecated there is no observer connection in UDK
     */
    @Deprecated
    public SocketConnectionAddress getObserverAddress() {
        return new SocketConnectionAddress(getHost(), getObserverPort());
    }

    /**
     * Configuration object of the UCC wrapper instance.
     */
    public static class UCCWrapperConf implements Serializable {

        String mapName = null;

        String gameBotsPack = null;

        String gameType = null;

        String options = null;

        Boolean startOnUnusedPort = null;

        String otherCommandLineParameters = null;

        Integer botServerPort = null;

        Integer controlServerPort = null;

        /**
         * UNIX only option to specify wineprefix for wine installation to use
         */
        String wineprefix = null;                
        
        transient Logger log = null;

        public static final UCCWrapperConf DEFAULTS;

        static {
            DEFAULTS = new UCCWrapperConf();
            DEFAULTS.setMapName("DM-Deck").setGameBotsPack("GameBotsUDK").setGameType("BotDeathMatch").setStartOnUnusedPort(true);

        }

        public UCCWrapperConf() {
        }

        public void applyDefaults(UCCWrapperConf defaults) {
            if (mapName == null) {
                setMapName(defaults.mapName);
            }
            if (gameBotsPack == null) {
                setGameBotsPack(defaults.gameBotsPack);
            }
            if (gameType == null) {
                setGameType(defaults.gameType);
            }
            if (options == null) {
                setOptions(defaults.options);
            }
            if (startOnUnusedPort == null) {
                setStartOnUnusedPort(defaults.startOnUnusedPort);
            }
            if (otherCommandLineParameters == null) {
                setOtherCommandLineParameters(defaults.otherCommandLineParameters);
            }
            if (log == null) {
                setLogger(defaults.log);
            }
            if (botServerPort == null) {
                setBotServerPort(defaults.botServerPort);
            }
            if (controlServerPort == null) {
                setControlServerPort(defaults.controlServerPort);
            }
            if(wineprefix == null){
                setWineprefix(wineprefix);
            }
        }

        /**
         * Forces UCC to find free port and start on it, otherwise it will start on ports 3000 + 3001.
         * @param startOnUnusedPort
         */
        public UCCWrapperConf setStartOnUnusedPort(boolean startOnUnusedPort) {
            this.startOnUnusedPort = startOnUnusedPort;
            return this;
        }

        /**
         * Eg. GameBots2004, GBSceanrio etc.
         * @param gameBotsPack
         */
        public UCCWrapperConf setGameBotsPack(String gameBotsPack) {
            this.gameBotsPack = gameBotsPack;
            return this;
        }

        public UCCWrapperConf setMapName(String mapName) {
            this.mapName = mapName;
            return this;
        }

        /**
         * Eg. BotDeathMatch, BotCTFGame etc. Consult GameBots documentation for
         * complete list available game types.
         */
        public UCCWrapperConf setGameType(String gameType) {
            this.gameType = gameType;
            return this;
        }

        /**
         * Can be used for setting mutators etc.
         * @param options
         */
        public UCCWrapperConf setOptions(String options) {
            this.options = options;
            return this;
        }

        /**
         * Logger used by the UCC.
         * @param log
         */
        public UCCWrapperConf setLogger(Logger log) {
            this.log = log;
            return this;
        }

        public UCCWrapperConf setOtherCommandLineParameters(String otherCommandLineParameters) {
            this.otherCommandLineParameters = otherCommandLineParameters;
            return this;
        }

        public UCCWrapperConf setBotServerPort(Integer botServerPort) {
            this.botServerPort = botServerPort;
            return this;
        }

        public UCCWrapperConf setControlServerPort(Integer controlServerPort) {
            this.controlServerPort = controlServerPort;
            return this;
        }
        
        /**
         * UNIX-only option
         * @param wineprefix
         * @return 
         */
        public UCCWrapperConf setWineprefix(String wineprefix){
            this.wineprefix = wineprefix;
            return this;
        }
    }

    /**
     * Reads content of the stream and discards it.
     */
    protected class StreamSink extends Thread {

        protected InputStream os = null;

        public StreamSink(InputStream os) {
            setName("UCC Stream handler");
            this.os = os;
        }

        protected void handleInput(String str) {
            if (uccLog.isLoggable(Level.INFO)) {
                uccLog.info("ID" + uccWrapperUID + " " + str);
            }
        }

        @Override
        public void run() {
            BufferedReader stdInput = new BufferedReader(new InputStreamReader(os));

            String s = null;
            try {
                while ((s = stdInput.readLine()) != null) {
                    handleInput(s);
                }
                os.close();
            } catch (IOException ex) {
                // the process has been closed so reading the line has failed, 
                // don't worry about it
                //ex.printStackTrace();
            }
        }
    }

    /**
     * Scanns the output of UCC for some specific srings (Ports bounded. START MATCH). 
     */
    public class ScannerSink extends StreamSink {

        /** Exception that ended the startig. Should be checked after the latch is raised. */
        public UCCStartException exception = null;

        public ScannerSink(InputStream is) {
            super(is);
            timer.schedule(task = new TimerTask() {

                @Override
                public void run() {
                    exception = new UCCStartException("Starting timed out. Ports weren't bound in the required time (" + startingTimeout + " ms).", this);
                    timer.cancel();
                    serverStartedLatch.countDown();
                }
            }, startingTimeout);
        }
        public CountDownLatch serverStartedLatch = new CountDownLatch(1);
        public CountDownLatch mapChangedLatch = null;

        public int controlPort = -1;

        public int botsPort = -1;

        /**
         * Thread that kills ucc process after specified time if the ports aren't 
         * read from the console. This prevents freezing the ScannerSink when ucc
         * fails to start.
         */
        Timer timer = new Timer("UDK start timeout");

        TimerTask task = null;
//        private final String defaultPatternStart = "\\[[0-9]*\\.[0-9]*\\] [^:]*: ";

        private final String defaultPatternStart = "";

        private final Pattern portPattern = Pattern.compile(defaultPatternStart + "BotServerPort:(\\d*) ControlServerPort:(\\d*)");

        private final Pattern commandletNotFoundPattern = Pattern.compile(defaultPatternStart + "Commandlet server not found");

        private final Pattern mapNotFoundPattern = Pattern.compile(defaultPatternStart + "No maplist entries found matching the current command line.*");

        private final Pattern matchStartedPattern = Pattern.compile(defaultPatternStart + "START MATCH");

        public CountDownLatch awaitGameStart(){
            if(serverStartedLatch != null && serverStartedLatch.getCount() > 0){
                throw new IllegalStateException("Only one game start await might be in progress");
            }
            serverStartedLatch = new CountDownLatch(1);
            return serverStartedLatch;
        }
        
        
        @Override
        protected void handleInput(String str) {
            super.handleInput(str);
            if (serverStartedLatch.getCount() != 0) {
                // server has not yet started

                Matcher matcher = portPattern.matcher(str);
                if (matcher.find()) {
                    botsPort = Integer.parseInt(matcher.group(1));
                    controlPort = Integer.parseInt(matcher.group(2));
                    //raiseLatch();
                }

                matcher = commandletNotFoundPattern.matcher(str);
                if (matcher.find()) {
                    exception = new UCCStartException("UDK failed to start due to: Commandlet server not found.", this);
                    raiseServerStartedLatch();
                }

                matcher = mapNotFoundPattern.matcher(str);
                if (matcher.find()) {
                    exception = new UCCStartException("UDK failed to start due to: Map not found.", this);
                    raiseServerStartedLatch();
                }

                matcher = matchStartedPattern.matcher(str);
                if (matcher.find()) {
                    // The match has started, raise the latch
                    raiseServerStartedLatch();
                }
            }

        }

        protected void raiseServerStartedLatch() {
            timer.cancel();
            task.cancel();
            serverStartedLatch.countDown();
        }
    }
}
