/*
 * UCCWrapper.java
 *
 */
package cz.cuni.amis.pogamut.udk.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base.utils.logging.LogPublisher;
import cz.cuni.amis.pogamut.udk.factory.direct.remoteagent.UDKServerFactory;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.server.exception.UCCStartException;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Wrapper of running instance of UDK server. Implements pooling of instances.
 * Usage scenario is:
 * <code>
 * UCCWrapper ucc = UCCWrapper.create();
 * ...
 * ucc.release();
 * </code>
 * The location of UDK executabe will be determined by an environment variable
 * pogamut.udk.home (e.g. c:\Games\UDK). The property cam be set via <i>java ...
 * -Dpogamut.udk.home=c:\Games\UDK</i>. Another posibility is to set it
 * by code <code>System.setProperty("pogamut.udk.home", "c:\\udks\\UDK-2011-05");</code>.
 * 
 * @author Ik
 */
public class UCCWrapper {

    /**
     * Configuration object of the UCC wrapper instance.
     */
    public static class UCCWrapperConf implements Serializable {

        String mapName = "DM-Deck";
        String gameBotsPack = "GameBotsUDK";
        String gameType = "BotDeathMatch";
        String mutators = "";
        String options = "";
        boolean startOnUnusedPort = true;
        transient Logger log = null;

        /**
         * Forces UCC to find free port and start on it, otherwise it will start on ports 3000 + 3001.
         * @param startOnUnusedPort
         */
        public UCCWrapperConf setStartOnUnusedPort(boolean startOnUnusedPort) {
            this.startOnUnusedPort = startOnUnusedPort;
            return this;
        }

        /**
         * Eg. GameBots2004, GBSceanrio etc.
         * @param gameBotsPack
         */
        public UCCWrapperConf setGameBotsPack(String gameBotsPack) {
            this.gameBotsPack = gameBotsPack;
            return this;
        }

        public UCCWrapperConf setMapName(String mapName) {
            this.mapName = mapName;
            return this;
        }

        /**
         * Eg. BotDeathMatch, BotCTFGame etc. Consult GameBots documentation for
         * complete list available game types.
         */
        public UCCWrapperConf setGameType(String gameType) {
            this.gameType = gameType;
            return this;
        }

        /**
         * Can be used for setting mutators etc.
         * @param options
         */
        public UCCWrapperConf setOptions(String options) {
            this.options = options;
            return this;
        }

        /**
         * Logger used by the UCC.
         * @param log
         */
        public UCCWrapperConf setLogger(Logger log) {
            this.log = log;
            return this;
        }
    }
    /** Loger containing all output from running instance of UCC. */
    protected LogCategory uccLog;
    protected static int fileCounter = 0;
    Process uccProcess = null;
    /** Port for bots. */
    protected int gbPort = -1;
    /** Port for server connection. */
    protected int controlPort = -1;
    /** Port for observer connection. */
    protected int observerPort = -1;
    protected IUDKServer utServer = null;
    /** First port assigned to a ucc instance. */
    protected static final int basePort = 39782;
    protected static Integer nextUccWrapperUID = 0;
    /** ID of the wrapper object. Useful for debuging. */
    protected int uccWrapperUID = 0;
    protected String unrealHome = null;
    protected UCCWrapperConf configuration = null;
    //protected String mapToLoad

    /**
     * @return Log with output of UCC. If you want to listen also for messages 
     * from the startup sequence then use UCCWrapper.create(Logger parent). Set
     * Parent logger of this log and register listeners before creating this
     * instance of UCCWrapper.  
     */
    public Logger getLogger() {
        return uccLog;
    }

    /**
     * @return Server connected to this UCC instance.
     */
    public IUDKServer getUTServer() {
        stopCheck();
        if (utServer == null) {
            UDKServerFactory factory = new UDKServerFactory();
            UDKServerRunner serverRunner = new UDKServerRunner(factory, "NBUTServer", "localhost", controlPort);
            utServer = serverRunner.startAgent();
        }
        return utServer;
    }

    protected String getUnrealHome() {
        if (unrealHome == null) {
            return Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UNREAL_HOME.getKey());
        } else {
            return unrealHome;
        }
    }

    public UCCWrapper(UCCWrapperConf configuration) throws UCCStartException {
    	uccLog = new LogCategory("Wrapper");
    	uccLog.addHandler(new LogPublisher.ConsolePublisher(new AgentId("UCC")));
    	if (configuration.log != null) {
            uccLog.setParent(configuration.log);
        }
        this.configuration = configuration;
        uccWrapperUID = nextUccWrapperUID++;
        initUCCWrapper();
        Runtime.getRuntime().addShutdownHook(shutDownHook);
    }
    /**
     * Task that will kill the UCC process when user forgets to do so.
     */
    Thread shutDownHook = new Thread("UCC wrapper finalizer") {

        @Override
        public void run() {
            uccProcess.destroy();
        }
    };

    /**
     * Reads content of the stream and discards it.
     */
    protected class StreamSink extends Thread {

        protected InputStream os = null;

        public StreamSink(InputStream os) {
            setName("UCC Stream handler");
            this.os = os;
        }

        protected void handleInput(String str) {
            if (uccLog.isLoggable(Level.INFO)) uccLog.info("ID" + uccWrapperUID + " " + str);
        }

        @Override
        public void run() {
            BufferedReader stdInput = new BufferedReader(new InputStreamReader(os));

            String s = null;
            try {
                while ((s = stdInput.readLine()) != null) {
                    handleInput(s);
                }
                os.close();
            } catch (IOException ex) {
                // the process has been closed so reading the line has failed, 
                // don't worry about it
                //ex.printStackTrace();
            }
        }
    }

    /**
     * Scanns the output of UCC for some specific srings (Ports bounded. START MATCH). 
     */
    public class ScannerSink extends StreamSink {

        public long startingTimeout = 2 * 60 * 1000;
        /** Exception that ended the startig. Should be checked after the latch is raised. */
        public UCCStartException exception = null;

        public ScannerSink(InputStream is) {
            super(is);
            timer.schedule(task = new TimerTask() {

                @Override
                public void run() {
                    exception = new UCCStartException("Starting timed out. Ports weren't bound in the required time (" + startingTimeout + " ms).", this);
                    timer.cancel();
                    portsBindedLatch.countDown();
                }
            }, startingTimeout);
        }
        public CountDownLatch portsBindedLatch = new CountDownLatch(1);
        public int controlPort = -1;
        public int botsPort = -1;
        /**
         * Thread that kills ucc process after specified time if the ports aren't 
         * read from the console. This prevents freezing the ScannerSink when ucc
         * fails to start.
         */
        Timer timer = new Timer("UDK start timeout");
        TimerTask task = null;
//        private final String defaultPatternStart = "\\[[0-9]*\\.[0-9]*\\] [^:]*: ";
        private final String defaultPatternStart = "";
        private final Pattern portPattern = Pattern.compile(defaultPatternStart + "BotServerPort:(\\d*) ControlServerPort:(\\d*)");
        private final Pattern commandletNotFoundPattern = Pattern.compile(defaultPatternStart + "Commandlet server not found");
        private final Pattern mapNotFoundPattern = Pattern.compile(defaultPatternStart + "No maplist entries found matching the current command line.*");
        private final Pattern matchStartedPattern = Pattern.compile(defaultPatternStart + "START MATCH");

        @Override
        protected void handleInput(String str) {
            super.handleInput(str);
            if (portsBindedLatch.getCount() != 0) {
                // ports still havent been found, try to scan the line

                Matcher matcher = portPattern.matcher(str);
                if (matcher.find()) {
                	botsPort = Integer.parseInt(matcher.group(1));
                	controlPort = Integer.parseInt(matcher.group(2));
                    //raiseLatch();
                }

                matcher = commandletNotFoundPattern.matcher(str);
                if (matcher.find()) {
                    exception = new UCCStartException("UDK failed to start due to: Commandlet server not found.", this);
                    raiseLatch();
                }

                matcher = mapNotFoundPattern.matcher(str);
                if (matcher.find()) {
                    exception = new UCCStartException("UDK failed to start due to: Map not found.", this);
                    raiseLatch();
                }

                matcher = matchStartedPattern.matcher(str);
                if (matcher.find()) {
                    // The match has started, raise the latch
                    raiseLatch();
                }
            }

        }

        protected void raiseLatch() {
            timer.cancel();
            task.cancel();
            portsBindedLatch.countDown();
        }
    }
    public static long stamp = System.currentTimeMillis();

    private void cleanupAfterException(){
        if(uccProcess != null){
            uccProcess.destroy();
        }
    }
    
    protected void initUCCWrapper() throws UCCStartException {
        try {
            // start new ucc instance
            String id = System.currentTimeMillis() + "a" + fileCounter++;
            String fileWithPorts = "GBports" + id;
            String udkHomePath = getUnrealHome();
            String binariesPath = udkHomePath + File.separator + "Binaries";

            // default ucc executable for Windows
            //udk.exe creates a separate console for output and the output is thus unavaliable to this process, udk.com does not
            String uccFile = "Win32" + File.separator + "UDK.exe"; 
            String execStr = binariesPath + File.separator + uccFile;

            // determine OS type, if it isn't win suppose it is Linux and try to run UDK through Wine
            String postOptions = "";
            String preOptions = "";
            if (!System.getProperty("os.name").contains("Windows")) {
                postOptions = " -nohomedir";
                preOptions = execStr;
                execStr = Pogamut.getPlatform().getProperty("WINE", "wine");
            }

            String portsSetting = configuration.startOnUnusedPort ? "?PortsLog=" + fileWithPorts + "?bRandomPorts=true" : "";

            ProcessBuilder procBuilder = new ProcessBuilder(execStr, "server", preOptions,
                    configuration.mapName
                    + "?game=" + configuration.gameBotsPack + "." + configuration.gameType
                    + portsSetting + configuration.options + postOptions);
            procBuilder.directory(new File(binariesPath));

            uccProcess = procBuilder.start();
            ScannerSink scanner = new ScannerSink(uccProcess.getInputStream());
            scanner.start();
            new StreamSink(uccProcess.getErrorStream()).start();

//            scanner.portsBindedLatch.await();
//            if (scanner.exception != null) {
//                // ucc failed to start 
//                uccProcess.destroy();
//                throw scanner.exception;
//            }

//            controlPort = scanner.controlPort;
//            gbPort = scanner.botsPort;
            controlPort = 3001;
            gbPort = 3000;
            Thread.sleep(5000);
        } catch (InterruptedException ex) {
            cleanupAfterException();
            throw new UCCStartException("Interrupted.", ex);
        } catch (IOException ex) {
            cleanupAfterException();
            throw new UCCStartException("IO Exception.", ex);
        }
    }

    /**
     * Process of the
     * @return
     */
    public Process getProcess() {
        return uccProcess;
    }
    /** Was this instance already released? */
    protected boolean stopped = false;

    /**
     * Stops the UCC server.
     */
    public synchronized void stop() {
        stopped = true;
        if (uccProcess != null) {
                try {
                    uccProcess.getOutputStream().write((byte)3);
        	try {
				Thread.sleep(1000);
				// give the process some time to terminate
			} catch (InterruptedException e) {

			}
                    uccProcess.getOutputStream().write((byte)3);
                } catch (IOException ex){}
        	uccProcess.destroy();
        	Runtime.getRuntime().removeShutdownHook(shutDownHook);
        	uccProcess = null;
        	try {
				Thread.sleep(1000);
				// give the process some time to terminate
			} catch (InterruptedException e) {

			}
        }
    }

    /**
     * @return Port for GameBots connection.
     */
    public int getBotPort() {
        stopCheck();
        return gbPort;
    }
    
    /**
     * @return Port of the Observer of GameBots2004.
     */
    public int getObserverPort() {
    	stopCheck();
    	return observerPort;
    }

    /**
     * @return Port for control connection.
     */
    public int getControlPort() {
        stopCheck();
        return controlPort;
    }

    protected void stopCheck() {
        if (stopped) {
            throw new PogamutException("UCC already stopped.", this);
        }
    }

	public String getHost() {
		return "localhost";
	}
	
	public SocketConnectionAddress getBotAddress() {
		return new SocketConnectionAddress(getHost(), getBotPort());
	}
	
	public SocketConnectionAddress getServerAddress() {
		return new SocketConnectionAddress(getHost(), getControlPort());
	}
	
	public SocketConnectionAddress getObserverAddress() {
		return new SocketConnectionAddress(getHost(), getObserverPort());
	}
	
}
