package cz.cuni.amis.pogamut.udk.agent.navigation.stuckdetector;

import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.navigation.IStuckDetector;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.SendMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GlobalChat;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.Self;

/**
 * Simple {@link IStuckDetector} that watches whether the bot moves at all.
 * <p><p>
 * You may define a period of time (timeout) in which the bot should move on, i.e.,
 * if the bot won't move a bit in a specified amount of time, it will report a stuck.
 *
 * @author Jimmy, knight
 */
public class UDKTimeStuckDetector implements IStuckDetector {

	/**
	 * Default distance that is regarded as "bot did not move a bit".
	 */
	private static final double NO_MOVEMENT_SIZE = 10;

	/**
	 * Default timeout after which the detector reports stuck if the bot did not move.
         * In miliseconds!
	 */
	private static int DEFAULT_TIMEOUT = 3000;

	/**
	 * Default timeout when the bot is waiting for something...
         * In miliseconds!
	 */
	private static int DEFAULT_WAITING_TIMEOUT = 10000;

	/**
	 * Owner of the detector.
	 */
	private UDKBot bot;

	/**
	 * Timeout used by the detector, when the bot is not waiting, initialized in the constructor.
	 */
	private double timeout;

	/**
	 * Timeout used by the detector when the bot is waiting, initialized in the constructor.
	 */
	private double waitingTimeout;

	private boolean botWaiting = false;
        /** In reset set to false. In isStuck set to true - indicator if this detector was already querried after reset about bot stucking. */
        private boolean bWasIsStuckCalled = false;
        /** Here we store current time of the simulation. */
        private long currentTime;

	/**
	 * Listener watching for the {@link Self} message. Recalls {@link UDKTimeStuckDetector#eventSelf(IWorldObjectEvent)}.
	 *
	 * @author Jimmy
	 */
	private class SelfListener implements IWorldObjectListener<Self> {

		public SelfListener(IWorldView worldView) {
			worldView.addObjectListener(Self.class, this);
		}

		@Override
		public void notify(IWorldObjectEvent<Self> event) {
			eventSelf(event);
		}

	};

	/**
	 * Listener that triggers {@link UDKTimeStuckDetector#eventSelf(IWorldObjectEvent)}.
	 */
	private SelfListener selfListener;

	/**
	 * Last time when the bot has moved (its velocity had been greater than {@link UDKTimeStuckDetector#NO_MOVEMENT_SIZE}.
	 */
	private double lastMovementTime = Double.NEGATIVE_INFINITY;

	/**
	 * Whether we should report that the bot has stuck.
	 */
	private boolean stuck = false;

	private Logger log;

	public UDKTimeStuckDetector(UDKBot bot) {
		this(bot, DEFAULT_TIMEOUT, DEFAULT_WAITING_TIMEOUT);
	}

	public UDKTimeStuckDetector(UDKBot bot, double timeout, double waitingTimeout) {
		if (this.log == null) {
			this.log = bot.getLogger().getCategory(this.getClass().getSimpleName());
		}
		this.bot = bot;
		this.timeout = timeout;
		this.waitingTimeout = waitingTimeout;
		selfListener = new SelfListener(bot.getWorldView());
	}

	public void eventSelf(IWorldObjectEvent<Self> event) {
                //we always update current time
                currentTime = event.getObject().getSimTime();
                //if we were not yet querried about stucking we will simply do nothing!
                if (!bWasIsStuckCalled)
                    return;
		if (event.getObject().getVelocity().size() > NO_MOVEMENT_SIZE || lastMovementTime == Double.NEGATIVE_INFINITY) {
			lastMovementTime = event.getObject().getSimTime();
		}
		if (botWaiting) {
			if (event.getObject().getSimTime() - lastMovementTime >= waitingTimeout) {
				stuck = true;
			}
		} else {
			if (event.getObject().getSimTime() - lastMovementTime >= timeout) {
				stuck = true;
			}
		}
	}

	@Override
	public void setBotWaiting(boolean state) {
		botWaiting = state;
	}

	@Override
	public boolean isStuck() {
                if (!bWasIsStuckCalled) {
                    //isStuck called for the first time after reset - we return false and we reset lastMovementTime
                    lastMovementTime = currentTime;
                    bWasIsStuckCalled = true;
                    return false;
                }
		return stuck;
	}

	@Override
	public void reset() {
		lastMovementTime = Double.NEGATIVE_INFINITY;
                bWasIsStuckCalled = false;
		stuck = false;
	}

}
