package cz.cuni.amis.pogamut.udk.test;

import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.junit.After;
import org.junit.Before;

import cz.cuni.amis.pogamut.base.agent.IAgent;
import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AbstractAgent;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IRemoteAgentParameters;
import cz.cuni.amis.pogamut.base.agent.state.WaitForAgentStateChange;
import cz.cuni.amis.pogamut.base.agent.state.level0.IAgentState;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateDown;
import cz.cuni.amis.pogamut.base.agent.state.level1.IAgentStateUp;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base.utils.logging.LogPublisher;
import cz.cuni.amis.pogamut.udk.bot.IUDKBotController;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotModule;
import cz.cuni.amis.pogamut.udk.observer.IUDKObserver;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.server.exception.UCCStartException;
import cz.cuni.amis.pogamut.udk.utils.PogamutUDKProperty;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UDKBotRunner;
import cz.cuni.amis.pogamut.udk.utils.UDKObserverRunner;
import cz.cuni.amis.pogamut.udk.utils.UDKServerRunner;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Base class for tests that need a running UCC instance. If you inherit from
 * this class then before the first @Test method will be called the UCC server
 * will be executed. The ports where it is listening can be obtained by
 * <code>ucc.getGbPort()</code> and <code>ucc.getControlPort()</code> calls.
 * Don't forget to specify pogamut.unreal.home and pogamut.unreal.serverexec
 * variables.
 * 
 * @author ik
 */
public class BaseUDKTest {

	protected IAgentId testId;
	
	protected LogCategory log;

    protected UCCWrapper ucc = null;
    
    /**
     * If true then use ucc executed through uccwrapper, othervise use externaly
     * executed instance.
     */
    protected boolean useInternalUcc = !Pogamut.getPlatform().getBooleanProperty(PogamutUDKProperty.POGAMUT_UNREAL_TEST_EXT_SERVER.getKey());

    public BaseUDKTest() {
    	this.testId = new AgentId("Test");
    	this.log = new LogCategory("UDKTest");
    	this.log.addHandler(new LogPublisher.ConsolePublisher(testId));
    }
        
    /**
     * Starts UCC server.
     *
     * @throws cz.cuni.amis.pogamut.ut2004.server.exceptions.UCCStartException
     */
    public void startUCC(UCCWrapperConf uccConf) throws UCCStartException {
        if (useInternalUcc) {            
            ucc = new UCCWrapper(uccConf);
        }
    }
    
    public void endUcc() {
    	if (useInternalUcc) {
            ucc.stop();
        }
    }

    /**
     * Initialize UCC server.
     * @throws UCCStartException
     */
    @Before
    public void startTest() throws UCCStartException {
    	startUCC(new UCCWrapperConf());    	
    }

    /**
     * Kills the UCC server and closes PogamutPlatform.
     */
    @After
    public void endTest() {
    	endUcc();
        Pogamut.getPlatform().close();
    }    

    /**
     * Waits till 'agent' changes its state to {@link IAgentStateUp}.
     * <p><p>
     * 60s timeout.
     * 
     * @param agent
     * @return
     */
    protected boolean awaitAgentUp(AbstractAgent agent) {
    	System.out.println("Awaiting server UP(timeout 60s)...");
    	IAgentState state = new WaitForAgentStateChange(agent.getState(), IAgentStateUp.class).await(60000, TimeUnit.MILLISECONDS);
    	return state != null && state instanceof IAgentStateUp;
    }
    
    /**
     * Waits till 'agent' changes its state to {@link IAgentStateDown}.
     * <p><p>
     * 60s timeout.
     * 
     * @param agent
     * @return
     */
    protected boolean awaitAgentDown(AbstractAgent agent) {
    	System.out.println("Awaiting server DOWN (timeout 60s)...");
    	IAgentState state = new WaitForAgentStateChange(agent.getState(), IAgentStateDown.class).await(120000, TimeUnit.MILLISECONDS);
    	return state != null && state instanceof IAgentStateDown;
    }
    
    /**
     * Starts new bot in the environment.
     * @param <T>
     * @param controller controller that will be used for newly created bot
     * @return running bot with the given controller
     */
    protected <T extends IUDKBotController> UDKBot startUTBot(Class<T> controller) {

        UDKBotFactory factory = new UDKBotFactory(new UDKBotModule(
                controller));

        String host = Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_HOST.getKey());
        int port = Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_PORT.getKey());
        if (useInternalUcc) {
            host = ucc.getHost();
            port = ucc.getBotPort();
        }
        UDKBotRunner botRunner = new UDKBotRunner(factory,
                "TEST factory",
                host, port);
        UDKBot bot = (UDKBot) botRunner.startAgent();
        return bot;
    }
    
    /**
     * Starts new UTServer.
     * @param <T>
     * @return running server connected to UCC instance.
     */
    protected IUDKServer startUTServer(IAgentFactory<IUDKServer, IRemoteAgentParameters> factory) {
        String host = Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_HOST.getKey());
        int port = Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_PORT.getKey());
        if (useInternalUcc) {
            host = ucc.getHost();
            port = ucc.getControlPort();
        }

        UDKServerRunner runner = new UDKServerRunner(factory,
                "TEST server",
                host, port) {
            @Override
            protected void preStartHook(IAgent agent) throws PogamutException {
            	super.preStartHook(agent);
            	agent.getLogger().setLevel(Level.ALL);                
            }
        };
        return (IUDKServer) runner.startAgent();
    }
    
    /**
     * Starts new UTServer.
     * @param <T>
     * @return running server connected to UCC instance.
     */
    protected IUDKObserver startUTObserver(IAgentFactory<IUDKObserver, IRemoteAgentParameters> factory) {
        String host = Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_HOST.getKey());
        int port = Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_SERVER_PORT.getKey());
        if (useInternalUcc) {
            host = ucc.getHost();
            port = ucc.getObserverPort();
        }

        UDKObserverRunner runner = new UDKObserverRunner(factory,
                "TEST observer",
                host, port) {
            @Override
            protected void preStartHook(IAgent agent) throws PogamutException {
            	super.preStartHook(agent);
            	agent.getLogger().setLevel(Level.ALL);
            }
        };
        return (IUDKObserver) runner.startAgent();
    }

}
