package cz.cuni.amis.pogamut.udk.agent.module.sensor;

import java.util.List;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.module.SensorModule;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.udk.bot.IUDKBotController;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.BeginMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.FlagInfo;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.Mutator;
import cz.cuni.amis.pogamut.udk.communication.translator.shared.events.MutatorListObtained;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;

/**
 * Memory module specialized on general info about the game.
 * <p><p>
 * It is designed to be initialized inside {@link IUDKBotController#prepareBot(UDKBot)} method call
 * and may be used since {@link IUDKBotController#botInitialized(cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.GameInfo, cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.ConfigChange, cz.cuni.amis.pogamut.udk.communication.messages.gbinfomessages.InitedMessage)}
 * is called.
 *
 * @author Juraj 'Loque' Simlovic
 * @author Jimmy
 */
public class Game extends SensorModule<UDKBot>
{
	/**
	 * Enums for game types that shields you from Unreal's string ids of game types.
	 * @author Jimmy
	 *
	 */
	public enum GameType
	{
		/** Classic death-match: Kill or get killed. You're on you own! */
		BotDeathMatch,
		/** Team death-match: Strategic team killing. Shoot opponents only. */
		BotTeamGame,
		/** Capture the Flag! Raid the enemy base, steal their flag. */
		BotCTFGame,
		/** Capture the Flag! Raid the enemy base, steal their flag, with vehicles */
		BotVehicleCTFGame,
		/** This type of game is not supported. */
		Unknown;

		/**
		 * Tedious work this is.. Let's do it once, shall we?
		 *
		 * @param type Name of the type of the game type.
		 * @return Game type associated with given name.
		 */
		public static GameType getType(String type)
		{
			if (type.equalsIgnoreCase("BotDeathMatch"))       return BotDeathMatch;
			if (type.equalsIgnoreCase("BotTeamGame"))         return BotTeamGame;
			if (type.equalsIgnoreCase("BotCTFGame"))          return BotCTFGame;
			if (type.equalsIgnoreCase("BotVehicleCTFGame"))   return BotVehicleCTFGame;
			return Unknown;
		}
	}

	/**
	 * Retreives the type of the game.
	 *
	 * @return Type of the game.
	 */
	public GameType getGameType()
	{
		// retreive from GameInfo object and translate
                if (lastGameInfo == null) return null;
		return GameType.getType(lastGameInfo.getGametype());
	}

	/*========================================================================*/

	/**
	 * Retreives the name of current map.
	 *
	 * @return Name of the current map.
	 */
	public String getMapName()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.getLevel();
	}

	/*========================================================================*/

	/**
	 * Retreives current game time, since the game started.
	 *
	 * @return Current game timestamp.
	 *
	 * @todo Test, whether it is correct..
	 */
	public double getTime()
	{
		// retreive from last BeginMessage object
                if (lastBeginMessage == null) return 0;
		return lastBeginMessage.getTime();
	}

	/**
	 * Retreives time limit for the game.
	 *
	 * <p>Note: Then the time limit is reached and the game is tie, special
	 * game modes might be turned on, e.g. <i>sudden death overtime</i>.
	 * Depends on the game type and game settings.
	 *
	 * @return Time limit of the game.
	 *
	 * @see getRemainingTime()
	 */
	public Double getTimeLimit()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.getTimeLimit();
	}

	/**
	 * Retreives time remaining for the game.
	 *
	 * <p>Note: Then the time limit is reached and the game is tie, special
	 * game modes might be turned on, e.g. <i>sudden death overtime</i>.
	 * Depends on the game type and game settings.
	 *
	 * @return Time limit of the game.
	 *
	 * @see getTime()
	 * @see getTimeLimit()
	 *
	 * @todo Test, whether it is correct..
	 */
	public Double getRemainingTime()
	{
		// derive from the time limit and current time
                if (getTimeLimit() == null) return null;
		return getTimeLimit() - getTime();
	}

 	/*========================================================================*/

	/**
	 * <i>BotDeathMatch only:</i><p>
	 * Number of points (e.g. kills) needed to win the game.
	 *
	 * @return Frag limit of the game.
	 *
	 * @see getTeamScoreLimit()
	 */
	public Integer getFragLimit()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.getFragLimit();
	}

	/**
	 * <i>BotTeamGame, BotCTFGame, BotBombingRun, BotDoubleDomination only:</i><p>
	 * Number of points a team needs to win the game.
	 *
	 * @return Team score limit of the game.
	 *
	 * @see getFragLimit()
	 */
	public Integer getTeamScoreLimit()
	{
		// retreive from GameInfo object
		// we have to cast double to int because UT2004 exports it as double (duuno why)
                if (lastGameInfo == null) return null;
		return (int)lastGameInfo.getGoalTeamScore();
	}

	/*========================================================================*/

	/**
	 * <i>BotTeamGame, BotCTFGame, BotDoubleDomination only:</i><p>
	 * Retrieves number of teams in the game.
	 *
	 * <p> Team numbers start from 0.  Usually, there are just two teams: 0 and 1.
	 *
	 * @return Number of teams in the game.
	 */
	public Integer getMaxTeams()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.getMaxTeams();
	}

	/**
	 * <i>BotTeamGame, BotCTFGame, BotDoubleDomination only:</i><p>
	 * Retreives maximum number of players per team.
	 *
	 * @return Maximum number of players per team.
	 */
	public Integer getMaxTeamSize()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.getMaxTeamSize();
	}

	/*========================================================================*/

	/**
	 * Retreives starting level of health. This is the level of health the
	 * players spawn with into the game.
	 *
	 * @return Starting level of health.
	 *
	 * @see getMaxHealth()
	 * @see getFullHealth()
	 */
	public Integer getStartHealth()
	{
		// retreive from InitedMessage object
                if (lastInitedMessage == null) return null;
		return lastInitedMessage.getHealthStart();
	}

	/**
	 * Retreives maximum level of <i>non-boosted</i> health. This is the level
	 * achievable by foraging standard health kits.
	 *
	 * @return Maximum level of <i>non-boosted</i> health.
	 *
	 * @see getStartHealth()
	 * @see getMaxHealth()
	 */
	public Integer getFullHealth()
	{
		// retreive from InitedMessage object
                if (lastInitedMessage == null) return null;
		return lastInitedMessage.getHealthFull();
	}

	/**
	 * Retreives maximum level of <i>boosted</i> health. This is the total
	 * maximum health achievable by any means of health kits, super health,
	 * or health vials.
	 *
	 * @return Maximum level of <i>boosted</i> health.
	 *
	 * @see getStartHealth()
	 * @see getFullHealth()
	 */
	public Integer getMaxHealth()
	{
		// retreive from InitedMessage object
                if (lastInitedMessage == null) return null;
		return lastInitedMessage.getHealthMax();
	}

	/*========================================================================*/

	/**
	 * Retreives maximum level of combined armor. The armor consist of two
	 * parts, which are summed together into combined armor value. However,
	 * each part is powered-up by different item (either by <i>small shield</i>
	 * or by <i>super-shield</i>).
	 *
	 * @return Maximum level of combined armor.
	 *
	 * @see getMaxLowArmor()
	 * @see getMaxHighArmor()
	 */
	public Integer getMaxArmor()
	{
		// retreive from InitedMessage object
                if (lastInitedMessage == null) return null;
		return lastInitedMessage.getShieldStrengthMax();
	}

	/**
	 * Retreives maximum level of low armor. The armor consist of two
	 * parts, which are summed together into combined armor value. However,
	 * each part is powered-up by different item (either by <i>small shield</i>
	 * or by <i>super-shield</i>).
	 *
	 * <p>Low armor is powered-up by <i>small shield</i>.
	 *
	 * @return Maximum level of low armor.
	 *
	 * @see getMaxArmor()
	 * @see getMaxHighArmor()
	 */
	public int getMaxLowArmor()
	{
		// FIXME[js]: Where do we retreive the max low-armor info?
		return 50;
	}

	/**
	 * Retreives maximum level of high armor. The armor consist of two
	 * parts, which are summed together into combined armor value. However,
	 * each part is powered-up by different item (either by <i>small shield</i>
	 * or by <i>super-shield</i>).
	 *
	 * <p>High armor is powered-up by <i>super-shield</i>.
	 *
	 * @return Maximum level of high armor.
	 *
	 * @see getMaxArmor()
	 * @see getMaxLowArmor()
	 */
	public int getMaxHighArmor()
	{
		// FIXME[js]: Where do we retreive the max high-armor info?
		return 100;
	}

	/*========================================================================*/

	/**
	 * Retreives starting level of adrenaline. This is the level of adrenaline
	 * the players spawn with into the game.
	 *
	 * @return Starting level of adrenaline.
	 */
	public Integer getStartAdrenaline()
	{
		// retreive from InitedMessage object
		// ut2004 exports it as double, must cast to int, ut's weirdness
                if (lastInitedMessage == null) return null;
		return (int)lastInitedMessage.getAdrenalineStart();
	}

	/**
	 * Retreives target level of adrenaline that need to be gained to start
	 * special bonus actions.
	 *
	 * <p>Once the agent's adrenaline reaches this designated level, it can be
	 * used to start special bonus booster-actions like <i>invisibility</i>,
	 * <i>speed</i>, <i>booster</i>, etc. The adrenaline is then spent on the
	 * invoked action.
	 *
	 * @return Maximum level of adrenaline that can be gained.
	 */
	public Integer getTargetAdrenaline()
	{
		// retreive from InitedMessage object
		// ut2004 exports it as double, must cast to int, ut's weirdness
                if (lastInitedMessage == null) return null;
		return (int)lastInitedMessage.getAdrenalineMax();
	}

	/**
	 * Retreives maximum level of adrenaline that can be gained.
	 *
	 * @return Maximum level of adrenaline that can be gained.
	 */
	public Integer getMaxAdrenaline()
	{
		// retreive from InitedMessage object
		// FIXME[js]: Return type!
                if (lastInitedMessage == null) return null;
		return (int)lastInitedMessage.getAdrenalineMax();
	}

	/*========================================================================*/

	/**
	 * Tells, whether the weapons stay on pick-up points, even when they are
	 * picked-up by players.
	 *
	 * <p>If so, each weapon type can be picked up from pick-up points only
	 * once. If the player already has the weapon the pick-up point offers, he
	 * can not pick it up. Also, each weapon pick-up point always contains its
	 * associated weapon.
	 *
	 * <p>If not, weapons can be picked up from pick-up points repeatedly.
	 * If the player already has the weapon the pick-up point offers, the
	 * pick-up will simply replenish ammo for that weapon. Also, upon each
	 * pick-up by a player, the offered weapon disappears (it is "taken" by
	 * that player). Weapons respawn on empty pick-up points after a while.
	 *
	 * @return True, if weapons stay on pick-up points.
	 */
	public Boolean getWeaponsStay()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.isWeaponStay();
	}

	/*========================================================================*/

	/**
	 * Retreives the maximum number of multi-jumping combos.
	 *
	 * <p>Note: Multi-jump combos are currently limited to double-jumps for
	 * bots.
	 *
	 * @return Maximum number of multi-jumping combos.
	 */
	public Integer getMaxMultiJump()
	{
		// retreive from InitedMessage object
                if (lastInitedMessage == null) return null;
		return lastInitedMessage.getMaxMultiJump();
	}

	/*========================================================================*/

	/**
	 * Returns list of mutators that are active in the current game.
	 * 
	 * @return Current game's mutators
	 */
	public List<Mutator> getMutators()
	{
                if (lastMutatorListObtained == null) return null;
		return lastMutatorListObtained.getMutators();
	}

	/*========================================================================*/

	/**
	 * Tells, whether the game is paused or running. When the game is paused,
	 * nobody can move or do anything (usually except posting text messages).
	 *
	 * @return True, if the game is paused. False otherwise.
	 *
	 * @see areBotsPaused()
	 */
	public Boolean isPaused()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.isGamePaused();
	}

	/**
	 * Tells, whether the bots are paused or running. When the bots are paused,
	 * but the game is not paused as well, human controlled players can move.
	 * The bots are standing still and can do nothing  (usually except posting
	 * text messages).
	 *
	 * @return True, if the bots are paused. False otherwise.
	 *
	 * @see isPaused()
	 */
	public Boolean isBotsPaused()
	{
		// retreive from GameInfo object
                if (lastGameInfo == null) return null;
		return lastGameInfo.isBotsPaused();
	}

        /**
         * Returns a map indexed by team numbers, holding all flags in the game.
         * In non-Capture the Flag (CTF) gametypes the result map will be empty.
         *
         * @return Map containing all the flags in the game indexed by owner team number.
         */
	public Map<Integer, FlagInfo> getAllCTFFlags()
	{
		return allCTFFlags;
	}

        /**
         * Returns a collection of all the flags in the game.
         * In non-Capture the Flag (CTF) gametypes the result collection will be empty.
         *
         * @return Collection containing all the flags in the game.
         */
	public Collection<FlagInfo> getAllCTFFlagsCollection()
	{		
		return allCTFFlags.values();
	}

	/*========================================================================*/

	/** Most rescent message containing info about the game. */
	GameInfo lastGameInfo = null;

	/** Most rescent message containing info about the game frame. */
	InitedMessage lastInitedMessage = null;

	/** Most rescent message containing info about the game frame. */
	BeginMessage lastBeginMessage = null;
	
	/** Most recent info about game's mutators. */
	MutatorListObtained lastMutatorListObtained = null;

        /** All flags in the game - will be filled only in CTF games */
        Map<Integer, FlagInfo> allCTFFlags = new HashMap();

	/*========================================================================*/

	/**
	 * GameInfo listener.
	 */
	private class GameInfoListener implements IWorldObjectEventListener<GameInfo, IWorldObjectEvent<GameInfo>>
	{
		@Override
		public void notify(IWorldObjectEvent<GameInfo> event)
		{
			lastGameInfo = event.getObject();
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * @param worldView WorldView object to listent to.
		 */
		public GameInfoListener(IWorldView worldView)
		{
			worldView.addObjectListener(GameInfo.class, this);
		}
	}

	/** GameInfo listener */
	GameInfoListener gameInfoListener;

	/*========================================================================*/

	/**
	 * InitedMessage listener.
	 */
	private class InitedMessageListener implements IWorldObjectEventListener<InitedMessage, WorldObjectUpdatedEvent<InitedMessage>>
	{
		@Override
		public void notify(WorldObjectUpdatedEvent<InitedMessage> event)
		{
			lastInitedMessage = event.getObject();
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * @param worldView WorldView object to listent to.
		 */
		public InitedMessageListener(IWorldView worldView)
		{
			worldView.addObjectListener(InitedMessage.class, WorldObjectUpdatedEvent.class, this);
		}
	}

	/** InitedMessage listener */
	InitedMessageListener initedMessageListener;

	/*========================================================================*/

	/**
	 * BeginMessage listener.
	 */
	private class BeginMessageListener implements IWorldEventListener<BeginMessage>
	{
		@Override
		public void notify(BeginMessage event)
		{
			lastBeginMessage = event;
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * @param worldView WorldView object to listent to.
		 */
		public BeginMessageListener(IWorldView worldView)
		{
			worldView.addEventListener(BeginMessage.class, this);
		}
	}

	/** BeginMessage listener */
	BeginMessageListener beginMessageListener;

	/*========================================================================*/
	
	/**
	 * MutatorListObtained listener.
	 */
	private class MutatorListObtainedListener implements IWorldEventListener<MutatorListObtained>
	{
		@Override
		public void notify(MutatorListObtained event)
		{
			lastMutatorListObtained = event;
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * @param worldView WorldView object to listent to.
		 */
		public MutatorListObtainedListener(IWorldView worldView)
		{
			worldView.addEventListener(MutatorListObtained.class, this);
		}
	}

	/** MutatorListObtained listener */
	MutatorListObtainedListener mutatorListObtainedListener;

	/*========================================================================*/

	/**
	 * FlagInfo object listener.
	 */
	private class FlagInfoObjectListener implements IWorldObjectEventListener<FlagInfo,WorldObjectFirstEncounteredEvent<FlagInfo>>
	{
		/**
                 * Save flag in our HashMap.
                 * 
                 * @param event
                 */
		public void notify(WorldObjectFirstEncounteredEvent<FlagInfo> event)
		{
                        if (allCTFFlags.containsKey(event.getObject().getTeam()))
                                if (log.isLoggable(Level.WARNING)) log.warning("Saving second Flag for team: " + event.getObject().getTeam() + ". Bug?");

			allCTFFlags.put(event.getObject().getTeam(), event.getObject());
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * @param worldView WorldView object to listent to.
		 */
		public FlagInfoObjectListener(IWorldView worldView)
		{
			worldView.addObjectListener(FlagInfo.class, WorldObjectFirstEncounteredEvent.class, this);
		}        
	}

	/** FlagInfo object listener */
	FlagInfoObjectListener flagInfoObjectListener;

	/*========================================================================*/

	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * @param bot owner of the module that is using it
	 */
	public Game(UDKBot bot) {
		this(bot, null);
	}
	
	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * @param bot owner of the module that is using it
	 * @param log Logger to be used for logging runtime/debug info. If <i>null</i>, the module creates its own logger.
	 */
	public Game(UDKBot bot, Logger log)
	{
		super(bot, log);

		// create listeners
		gameInfoListener = new GameInfoListener(worldView);
		beginMessageListener = new BeginMessageListener(worldView);
		initedMessageListener = new InitedMessageListener(worldView);
		mutatorListObtainedListener = new MutatorListObtainedListener(worldView);
                flagInfoObjectListener = new FlagInfoObjectListener(worldView);
	}
	
	/**
	 * Provides initialization of the module (clearing internal data structures). Called automatically
	 * during the agent starting sequence.
	 */
	@Override
	protected void start(boolean startPaused) {
		super.start(startPaused);
		lastGameInfo = null;
		lastInitedMessage = null;
		lastBeginMessage = null;
		lastMutatorListObtained = null;
	}
}
