/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunResult;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentRunner;
import cz.cuni.amis.pogamut.udk.experiments.IExperimentSuite;
import cz.cuni.amis.pogamut.udk.experiments.IResultWriter;
import java.io.IOException;

/**
 *
 * @author Martin Cerny
 */
public class SimpleExperimentSuiteRunner<RESULT, PARAMETERS> extends AbstractSingleThreadExperimentSuiteRunner<RESULT, PARAMETERS> {

    private IExperimentRunner<RESULT, PARAMETERS> runner;

    private IResultWriter<RESULT, PARAMETERS> resultWriter;

    private LogCategory log;

    public SimpleExperimentSuiteRunner(LogCategory log, long singleExperimentTimeout) {
        this(new ExperimentRunner<RESULT, PARAMETERS>(log, singleExperimentTimeout), log);
    }

    public SimpleExperimentSuiteRunner(IExperimentRunner<RESULT, PARAMETERS> runner, LogCategory log) {
        this(runner, null, log);
    }

    public SimpleExperimentSuiteRunner(IResultWriter<RESULT, PARAMETERS> resultWriter, LogCategory log, long singleExperimentTimeout) {
        this(new ExperimentRunner<RESULT, PARAMETERS>(log, singleExperimentTimeout), resultWriter, log);
    }

    public SimpleExperimentSuiteRunner(IExperimentRunner<RESULT, PARAMETERS> runner, IResultWriter<RESULT, PARAMETERS> resultWriter, LogCategory log) {
        this.runner = runner;
        this.resultWriter = resultWriter;
        this.log = log;
    }

    @Override
    protected void handleExperimentResult(IExperiment<RESULT, PARAMETERS> experiment, IExperimentRunResult<RESULT> result) {
        if (resultWriter == null) {
            return;
        }
        try {
            resultWriter.writeResult(result, experiment.getParameters());
        } catch (IOException ex) {
            log.severe("Could not write experiment result", ex);
        }
    }

    @Override
    protected IExperimentRunResult<RESULT> runSingleExperiment(IExperiment<RESULT, PARAMETERS> experiment) {
        return runner.runExperiment(experiment);
    }

    @Override
    public void runExperimentSuite(IExperimentSuite<RESULT, PARAMETERS> suite, int startExperiment) {
        if (resultWriter != null) {
            try {
                resultWriter.init();
            } catch (IOException ex) {
                log.severe("Error initializing result writer", ex);
                return;
            }
        }

        super.runExperimentSuite(suite, startExperiment);

        if (resultWriter != null) {
            try {
                resultWriter.close();
            } catch (IOException ex) {
                log.severe("Error closing result writer", ex);
                return;
            }
        }
    }
}
