package cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.WeakHashMap;

import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.batchClock.ISimulationClock;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.batchClock.SimulationTime;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry.IObjectListing;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry.IViewableObjectRegistry;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry.IWorldObjectRegistry;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.iface.worldObject.IViewableObjectBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.impl.action.ActionRegistryBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.impl.game.GameBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.impl.pawn.PawnBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.impl.player.PlayerBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.impl.worldObject.AbstractViewableObjectBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.ISnapshotMemorizer;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.iface.object.IObjectMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.iface.worldObject.IViewableObjectMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.impl.action.ActionRegistryMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.impl.game.GameMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.impl.pawn.PawnMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.snapshotMemorizer.memorization.impl.player.PlayerMemorization;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.iface.worldObject.IViewableObjectReplica;
import cz.cuni.amis.utils.ExceptionToString;

/** Emohawk sighting memory implementation.
 *
 * @author Paletz
 */
public class SightingMemory implements ISightingMemory {
	
	protected ISnapshotMemorizer memorizer;
	protected IViewableObjectRegistry inputWorldObjectRegistry;
	protected IWorldObjectRegistry outputWorldObjectRegistry;
	protected ISimulationClock simulationClock;
	
	protected Map<Class<? extends IObjectMemorization>, Class<? extends AbstractViewableObjectBelief>> classMap;
	
	protected double shortTermMemoryDuration;

	protected Map<WorldObjectId,AbstractViewableObjectBelief> sightings;
	// weak map to keep used beliefs fresh
	protected WeakHashMap<AbstractViewableObjectBelief,Boolean> beliefs; // Belief -> Tracked visible
	
	public SightingMemory() {
		super();
		sightings = new HashMap<WorldObjectId,AbstractViewableObjectBelief>();
		beliefs = new WeakHashMap<AbstractViewableObjectBelief,Boolean>();
		classMap = new HashMap<Class<? extends IObjectMemorization>, Class<? extends AbstractViewableObjectBelief>>();
		
		classMap.put( ActionRegistryMemorization.class, ActionRegistryBelief.class );
		
		classMap.put( GameMemorization.class, GameBelief.class );
		
		classMap.put( PawnMemorization.class, PawnBelief.class );
		
		classMap.put( PlayerMemorization.class, PlayerBelief.class );
	}
		
	@Override
	public void refresh() {
		IObjectListing<WorldObjectId, IViewable> visibleObjects = inputWorldObjectRegistry.getVisibleObjectListing();
		
		// register new sightings
		{
			Set<WorldObjectId> newObjects = new HashSet<WorldObjectId>( visibleObjects.getMap().keySet() );
			newObjects.removeAll( sightings.keySet() );
			
			for ( WorldObjectId id : newObjects ) {
				AbstractViewableObjectBelief belief = getBeliefOfVisible( id );
				sightings.put( id, belief ); 
				outputWorldObjectRegistry.registerObject( belief.getImmutableFacade(), simulationClock.getTime() );
			}		
		}
		
		// refresh beliefs		
		for ( Entry<AbstractViewableObjectBelief,Boolean> entry : beliefs.entrySet() ) {
			AbstractViewableObjectBelief belief = entry.getKey();
			Boolean trackedVisible = entry.getValue();
			if ( belief == null ) {
				// them weak sets behave funny :]
				continue;
			}
			
			WorldObjectId id = belief.getImmutableFacade().getId();
			IViewableObjectReplica replica = visibleObjects.getById( IViewableObjectReplica.class, id );
			if ( replica == null ) {
				if ( trackedVisible ) {
					entry.setValue( false );
					outputWorldObjectRegistry.notifyObjectUpdated( belief.getImmutableFacade(), simulationClock.getTime() );
				}
				
				// nothing to update
				continue;
			} 
			
			entry.setValue( true );			
			IObjectMemorization memorization = memorizer.getMemorization( replica ); 
			
			belief.updateMemorization( memorization );
			outputWorldObjectRegistry.notifyObjectUpdated( belief.getImmutableFacade(), simulationClock.getTime() );
		}
		
		// remove irrelevant beliefs
		Iterator<Entry<WorldObjectId,AbstractViewableObjectBelief>> iterator = sightings.entrySet().iterator();
		while ( iterator.hasNext() ) {
			Entry<WorldObjectId,AbstractViewableObjectBelief> entry = iterator.next();
			AbstractViewableObjectBelief belief = entry.getValue();
			if ( belief.getImmutableFacade().isPlausible() ) {
				continue;
			}
			
			long ageMilliseconds = simulationClock.getTime().getMilliSeconds() - belief.getImmutableFacade().getSimTime();
			
			if ( ageMilliseconds >= shortTermMemoryDuration/1000 ) {
				iterator.remove();
				outputWorldObjectRegistry.forgetObject( belief.getImmutableFacade(), simulationClock.getTime() );
			}
		}
	}
	
	@Override
	public double getShortTermMemoryDuration() {
		return shortTermMemoryDuration;
	}
	
	@Override
	public void setShortTermMemoryDuration( double value ) {
		this.shortTermMemoryDuration = value;
	}
		
	@Override
	public void initInputViewableWorldObjectRegistry( IViewableObjectRegistry inputViewableWorldObjectRegistry ) {
		this.inputWorldObjectRegistry = inputViewableWorldObjectRegistry;
	}

	@Override
	public void initOutputWorldObjectRegistry( IWorldObjectRegistry outputWorldObjectRegistry ) {
		this.outputWorldObjectRegistry = outputWorldObjectRegistry;
	}
	
	@Override
	public void initSimulationClock(ISimulationClock simulationClock) {
		this.simulationClock = simulationClock;
	}

	@Override
	public void initMemorizer( ISnapshotMemorizer memorizer ) {
		this.memorizer = memorizer;
	}
	
	@Override
	public void contradict( IViewableObjectBelief belief) {
		if ( belief.isVisible() ) {
			throw new IllegalArgumentException( "Contradicted belief must not be visible." );
		}
		
		if ( !belief.isPlausible() ) {
			return;
		}
		
		sightings.get( belief.getId() ).contradict();
		outputWorldObjectRegistry.notifyObjectUpdated( belief, simulationClock.getTime() );
	}
	
	/** Make unregistered belief
	 * 
	 * @param memorization memorization to base the belief upon
	 * @return unregistered belief - it won't be auto-updated by more recent observations until registered
	 */
	protected AbstractViewableObjectBelief makeUnregisteredBelief( IObjectMemorization memorization ) {
		
		assert( memorization != null );
		
		Class<? extends IObjectMemorization> memorizationClass = memorization.getClass();
		Class<? extends AbstractViewableObjectBelief> beliefClass = classMap.get( memorizationClass );
		
		try {
			return beliefClass.getConstructor( memorizationClass, BeliefSupport.class ).newInstance( memorization, beliefSupport );
		} catch (InstantiationException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		} catch (IllegalAccessException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		} catch (IllegalArgumentException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		} catch (InvocationTargetException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		} catch (NoSuchMethodException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		} catch (SecurityException e) {
			throw new AssertionError( "Failed to create belief." + ExceptionToString.process(e) );
		}
	}

	/** Get freshest available belief
	 * 
	 * @param fallBackMemorization used to base the belief upon if no more recent observation is available
	 * @return freshest available belief (auto-updating)
	 */
	protected AbstractViewableObjectBelief getFreshestBelief( IViewableObjectMemorization fallBackMemorization ) {
		if ( fallBackMemorization == null ) {
			return null;
		}
		
		// check existing beliefs
		for ( AbstractViewableObjectBelief belief : beliefs.keySet() ) {
			if ( belief == null ) { 
				// them weak sets behave funny :]
				continue;
			}
			
			if ( belief.getImmutableFacade().getId() == fallBackMemorization.getId() ) {
				return belief;
			}
		}
		
		// skipping visibility check, if object is visible it surely is in world view and hence would be found within existing beliefs
		
		return (AbstractViewableObjectBelief) makeBelief( fallBackMemorization );
	}
	
	/** Get belief of an visible world object
	 * 
	 * @param id world object id
	 * @return auto-updating belief based upon current state of the world object
	 */
	protected AbstractViewableObjectBelief getBeliefOfVisible( WorldObjectId id ) {
		
		IObjectListing<WorldObjectId, IViewable> visibleObjectListing = inputWorldObjectRegistry.getVisibleObjectListing();
		IViewable replica = visibleObjectListing.getById( id );
		IViewableObjectMemorization memorization = (IViewableObjectMemorization) memorizer.getMemorization( replica );
		return getFreshestBelief( memorization );
	}
	
	/** Make belief based on a memorization
	 * 
	 * @param memorization memorization to base the belief upon
	 * @return auto-updating belief
	 */
	protected AbstractViewableObjectBelief makeBelief( IObjectMemorization memorization ) {
		AbstractViewableObjectBelief retval = makeUnregisteredBelief(memorization);
		beliefs.put( (AbstractViewableObjectBelief) retval, retval.getImmutableFacade().isVisible() );
		return retval;
	}
	

	protected final BeliefSupport beliefSupport = new BeliefSupport();
	
	/** Support for belief implementation
	 * 
	 * @author Paletz
	 */
	public class BeliefSupport {
		
		public IViewableObjectBelief getFreshestBelief( IViewableObjectMemorization fallBackMemorization ) {
			if ( fallBackMemorization == null ) {
				return null;
			}
			return SightingMemory.this.getFreshestBelief( fallBackMemorization ).getImmutableFacade();
		}
		
		public SimulationTime getSnapshotTime() {
			return memorizer.getSnapshotTime();
		}
	}
}