package cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectRegistry;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import cz.cuni.amis.utils.ClassUtils;
import cz.cuni.amis.utils.maps.HashMapMap;

/** Object listing implementation
 * 
 * Maps are synchronized.
 * 
 * @author Paletz
 *
 * @param <TObject>
 * @param <TId>
 */
public class ObjectListing<TId,TObject> implements IObjectListing<TId,TObject> {

	/** Map that holds all the objects.
	 */
	protected Map<TId, TObject> objects;
	
	/** Immutable view of objects
	 */
	protected Map<TId, TObject> immutableObjectView;
	
	/** Map that sorts the objects to groups by their class.
	 * 
	 * Class -> ( ID->object ).
	 */
	protected Map<Class<? extends TObject>, Map<TId, TObject> > classSortedObjects;
	
	/** Immutable view of class-sorted objects
	 */
	protected Map<Class<? extends TObject>, Map<TId, TObject> > immutableClassSortedObjectView;
	
	/** Constructor
	 */
	public ObjectListing() {
		objects = new HashMap<TId, TObject>();
		immutableObjectView = Collections.unmodifiableMap( objects );
		classSortedObjects = new HashMapMap<Class<? extends TObject>, TId, TObject>();
		immutableClassSortedObjectView = Collections.unmodifiableMap( classSortedObjects );
	}
	
	@Override
	public Map<Class<? extends TObject>, Map<TId, TObject>> getClassGroupedMap() {
		return immutableClassSortedObjectView;
	}

	@Override
	public <TFilterClass extends TObject> Map<TId, TFilterClass> filterByClass( Class<TFilterClass> filterClass) {
		@SuppressWarnings("unchecked")
		Map<TId, TFilterClass> map = (Map<TId, TFilterClass>) classSortedObjects.get( filterClass );
		return Collections.unmodifiableMap( map );
	}
	
	@Override
	public Map<TId, TObject> getMap() {
		return immutableObjectView;
	}
	
	@Override
	public TObject getById(TId id) {
		return objects.get( id );
	}
	
	@Override
	public <TTypedObject extends TObject> TTypedObject getById( Class<TTypedObject> type, TId id ) {
		TObject object = getById(id);
		if ( object != null && !type.isInstance( object ) ) {
			throw new ClassCastException( "Found object with ID: "+id+" but it's type ("+object.getClass()+") doesn't match required type ("+type+")." );
		}
		@SuppressWarnings("unchecked")
		TTypedObject retval = (TTypedObject) object;
		return retval;
	}
	
	/** Add object
	 * 
	 * @param id ID
	 * @param object object
	 */
	public void add( TId id, TObject object ) {
		objects.put( id, object );
	
		// getSubclasses() is actually getSupers() :]
		for ( Class<?> superClass : ClassUtils.getSubclasses( object.getClass() )) {
			classSortedObjects.get( superClass ).put( id, object );
		}	
	}
	
	/** Remove object
	 * 
	 * @param id object's ID
	 */
	public void remove( TId id ) {
		assert( objects.containsKey( id ) );
		
		// getSubclasses() is actually getSupers() :]
		for ( Class<?> superClass : ClassUtils.getSubclasses( objects.get( id ).getClass() )) {
			classSortedObjects.get( superClass ).remove( id );
		}			
		
		objects.remove( id );
	}
	
	/** Remove all objects
	 */
	public void clear() {
		objects.clear();
		classSortedObjects.clear();
	}
}
